import { Component, Prop, State, Element, Watch, h } from '@stencil/core';
import { Product } from '../../types/api';

@Component({
  tag: 'bcx-product-slider',
  styleUrl: 'bcx-product-slider.scss',
  shadow: true,
})
export class BcxProductSlider {
  @Element() el: HTMLElement;

  @Prop() products: Product[] = [];
  @Prop() language: 'pl' | 'en' = 'en';
  @Prop() showAfterStreaming: boolean = false;
  @State() currentIndex: number = 0;
  @State() isVisible: boolean = false;

  @Watch('products')
  onProductsChange() {
    // Reset to first slide when products change
    if (this.products.length > 0) {
      this.currentIndex = 0;
      // Use setTimeout to ensure DOM is updated
      setTimeout(() => {
        this.animateToSlide(0);
      }, 0);
    }
  }

  componentWillLoad() {
    if (this.showAfterStreaming && this.products.length > 0) {
      this.isVisible = true;
    }
  }

  @Watch('showAfterStreaming')
  onShowAfterStreamingChange() {
    if (this.showAfterStreaming && this.products.length > 0) {
      this.isVisible = true;
    }
  }

  private sliderRef: HTMLDivElement;
  private isDragging: boolean = false;
  private startX: number = 0;
  private currentX: number = 0;
  private initialTranslate: number = 0;

  componentDidLoad() {
    if (this.products.length > 1) {
      this.setupTouchHandlers();
    }
  }

  componentDidUpdate() {
    // Setup or cleanup touch handlers based on product count
    if (this.products.length > 1) {
      this.setupTouchHandlers();
    } else {
      this.cleanupTouchHandlers();
    }
  }

  disconnectedCallback() {
    this.cleanupTouchHandlers();
  }

  private setupTouchHandlers() {
    if (!this.sliderRef) return;

    this.sliderRef.addEventListener('touchstart', this.handleTouchStart, { passive: false });
    this.sliderRef.addEventListener('touchmove', this.handleTouchMove, { passive: false });
    this.sliderRef.addEventListener('touchend', this.handleTouchEnd, { passive: false });
    this.sliderRef.addEventListener('mousedown', this.handleMouseDown);
    this.sliderRef.addEventListener('mousemove', this.handleMouseMove);
    this.sliderRef.addEventListener('mouseup', this.handleMouseUp);
    this.sliderRef.addEventListener('mouseleave', this.handleMouseUp);
  }

  private cleanupTouchHandlers() {
    if (!this.sliderRef) return;

    this.sliderRef.removeEventListener('touchstart', this.handleTouchStart);
    this.sliderRef.removeEventListener('touchmove', this.handleTouchMove);
    this.sliderRef.removeEventListener('touchend', this.handleTouchEnd);
    this.sliderRef.removeEventListener('mousedown', this.handleMouseDown);
    this.sliderRef.removeEventListener('mousemove', this.handleMouseMove);
    this.sliderRef.removeEventListener('mouseup', this.handleMouseUp);
    this.sliderRef.removeEventListener('mouseleave', this.handleMouseUp);
  }

  private handleTouchStart = (e: TouchEvent) => {
    if (this.products.length <= 1) return;

    // Only start dragging if touch moves (don't interfere with click)
    const touch = e.touches[0];
    this.startX = touch.clientX;
    this.initialTranslate = this.getCurrentTranslate();
    // Don't set isDragging yet - wait for touchmove
  };

  private handleTouchMove = (e: TouchEvent) => {
    if (this.products.length <= 1) return;

    this.currentX = e.touches[0].clientX;
    const diff = Math.abs(this.currentX - this.startX);

    // Only start dragging if touch moves significantly (to distinguish from click)
    if (!this.isDragging && diff > 10) {
      this.isDragging = true;
    }

    if (!this.isDragging) return;

    const containerWidth = this.sliderRef.parentElement?.offsetWidth || 0;
    const diffPercentage = ((this.currentX - this.startX) / containerWidth) * 100;
    const newTranslate = this.initialTranslate + diffPercentage;

    this.updateSliderPosition(newTranslate);
    e.preventDefault();
  };

  private handleTouchEnd = () => {
    if (this.products.length <= 1) return;

    const wasDragging = this.isDragging;
    this.isDragging = false;

    // Only snap to slide if user was actually dragging
    if (wasDragging) {
      this.snapToNearestSlide();
    }
  };

  private handleMouseDown = (e: MouseEvent) => {
    if (this.products.length <= 1) return;

    this.isDragging = true;
    this.startX = e.clientX;
    this.initialTranslate = this.getCurrentTranslate();
    e.preventDefault();
  };

  private handleMouseMove = (e: MouseEvent) => {
    if (!this.isDragging || this.products.length <= 1) return;

    this.currentX = e.clientX;
    const diff = this.currentX - this.startX;
    const containerWidth = this.sliderRef.parentElement?.offsetWidth || 0;
    const diffPercentage = (diff / containerWidth) * 100;
    const newTranslate = this.initialTranslate + diffPercentage;

    this.updateSliderPosition(newTranslate);
    e.preventDefault();
  };

  private handleMouseUp = () => {
    if (!this.isDragging || this.products.length <= 1) return;

    this.isDragging = false;
    this.snapToNearestSlide();
  };

  private getCurrentTranslate(): number {
    if (!this.sliderRef) return 0;
    const transform = window.getComputedStyle(this.sliderRef).transform;
    const matrix = new DOMMatrix(transform);
    const containerWidth = this.sliderRef.parentElement?.offsetWidth || 0;
    // Convert pixel translate to percentage
    return (matrix.m41 / containerWidth) * 100;
  }

  private updateSliderPosition(translate: number) {
    if (!this.sliderRef) return;
    this.sliderRef.style.transform = `translateX(${translate}%)`;
  }

  private snapToNearestSlide() {
    if (!this.sliderRef || this.products.length <= 1) return;

    const currentTranslate = this.getCurrentTranslate();
    const slidePercentage = 100 / this.products.length;
    const targetIndex = Math.round(-currentTranslate / slidePercentage);
    const clampedIndex = Math.max(0, Math.min(targetIndex, this.products.length - 1));

    this.currentIndex = clampedIndex;
    this.animateToSlide(clampedIndex);
  }

  private animateToSlide(index: number) {
    if (!this.sliderRef || this.products.length <= 1) return;

    const translatePercentage = -index * (100 / this.products.length);

    this.sliderRef.style.transition = 'transform 0.3s cubic-bezier(0.16, 1, 0.3, 1)';
    this.sliderRef.style.transform = `translateX(${translatePercentage}%)`;

    setTimeout(() => {
      if (this.sliderRef) {
        this.sliderRef.style.transition = '';
      }
    }, 300);
  }

  private goToSlide(index: number) {
    if (index < 0 || index >= this.products.length || this.products.length <= 1) return;

    this.currentIndex = index;
    this.animateToSlide(index);
  }

  private handleProductClick = (product: Product, e: Event) => {
    // Prevent event from bubbling to prevent conflict with touch handlers
    e.stopPropagation();

    // Only navigate if not dragging (user is actually clicking, not swiping)
    if (!this.isDragging) {
      // Open product URL in new tab
      window.open(product.product_url, '_blank', 'noopener,noreferrer');
    }
  };

  private getTranslation(key: string): string {
    const translations = {
      view_product: {
        en: 'View Product',
        pl: 'Zobacz produkt',
      },
    };

    return translations[key]?.[this.language] || translations[key]?.['en'] || key;
  }

  render() {
    if (!this.products || this.products.length === 0) {
      return null;
    }

    // Don't render at all if we're waiting for streaming to finish
    if (this.showAfterStreaming && !this.isVisible) {
      return null;
    }
    return (
      <div class="bcx-product-slider">
        <div class={`bcx-product-slider__container ${this.products.length > 1 ? 'bcx-product-slider__container--no-radius' : ''}`}>
          <div
            class="bcx-product-slider__track"
            ref={el => (this.sliderRef = el)}
            style={{
              width: `${this.products.length * 100}%`,
              transform: this.products.length > 1 ? `translateX(-${this.currentIndex * (100 / this.products.length)}%)` : 'none',
            }}
          >
            {this.products.map((product, index) => (
              <div key={index} class="bcx-product-slider__card" style={{ width: `${100 / this.products.length}%` }} onClick={e => this.handleProductClick(product, e)}>
                <div class="bcx-product-slider__card-image">
                  <img
                    src={product.image_url}
                    alt={product.product_name}
                    loading="lazy"
                    onError={e => {
                      const target = e.target as HTMLImageElement;
                      target.style.display = 'none';
                      const placeholder = target.parentElement?.querySelector('.bcx-product-slider__card-placeholder');
                      if (placeholder) {
                        (placeholder as HTMLElement).style.display = 'flex';
                      }
                    }}
                  />
                  <div class="bcx-product-slider__card-placeholder">
                    <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                      <rect x="3" y="3" width="18" height="18" rx="2" ry="2" />
                      <circle cx="8.5" cy="8.5" r="1.5" />
                      <polyline points="21,15 16,10 5,21" />
                    </svg>
                  </div>
                </div>
                <div class="bcx-product-slider__card-content">
                  <h3 class="bcx-product-slider__card-title">{product.product_name}</h3>
                  <div class="bcx-product-slider__card-action">
                    <span class="bcx-product-slider__card-link">{this.getTranslation('view_product')}</span>
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                      <path d="M7 17L17 7" />
                      <path d="M7 7h10v10" />
                    </svg>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>

        {/* Navigation dots - only show if more than 1 product */}
        {this.products.length > 1 && (
          <div class="bcx-product-slider__dots">
            {this.products.map((_, index) => (
              <button
                key={index}
                class={`bcx-product-slider__dot ${index === this.currentIndex ? 'bcx-product-slider__dot--active' : ''}`}
                onClick={() => this.goToSlide(index)}
                aria-label={`Go to product ${index + 1}`}
              />
            ))}
          </div>
        )}
      </div>
    );
  }
}
