import { newSpecPage } from '@stencil/core/testing';
import { BcxProductSlider } from '../bcx-product-slider';

describe('bcx-product-slider', () => {
  const mockProducts = [
    { product_name: 'Product 1', image_url: 'http://example.com/img1.jpg', product_url: 'http://example.com/p1' },
    { product_name: 'Product 2', image_url: 'http://example.com/img2.jpg', product_url: 'http://example.com/p2' },
    { product_name: 'Product 3', image_url: 'http://example.com/img3.jpg', product_url: 'http://example.com/p3' },
  ];

  beforeEach(() => {
    // Mock window.open
    Object.defineProperty(window, 'open', {
      writable: true,
      value: jest.fn()
    });

    // Mock DOMMatrix
    (global as any).DOMMatrix = class DOMMatrix {
      m41 = 0;
      constructor(transform?: string) {
        if (transform && transform.includes('translateX')) {
           const match = /translateX\((-?\d+)px\)/.exec(transform);
           if (match) {
             this.m41 = parseInt(match[1], 10);
           }
        }
      }
    };
  });

  it('renders nothing if no products provided', async () => {
    const page = await newSpecPage({
      components: [BcxProductSlider],
      html: `<bcx-product-slider></bcx-product-slider>`,
    });
    expect(page.root).toEqualHtml(`
      <bcx-product-slider>
        <mock:shadow-root></mock:shadow-root>
      </bcx-product-slider>
    `);
  });

  it('renders nothing if showAfterStreaming is true but visibility flag is false', async () => {
    const page = await newSpecPage({
      components: [BcxProductSlider],
      html: `<bcx-product-slider show-after-streaming="true"></bcx-product-slider>`,
    });

    page.rootInstance.products = mockProducts;
    await page.waitForChanges();

    // Initially isVisible is false in component logic if initialized with empty products then updated?
    // Let's check logic: componentWillLoad sets isVisible = true if products > 0
    // If products added later via prop...

    // Actually, if showAfterStreaming is true, it waits for something?
    // Logic:
    // @Watch('showAfterStreaming')
    // onShowAfterStreamingChange() {
    //   if (this.showAfterStreaming && this.products.length > 0) {
    //     this.isVisible = true;
    //   }
    // }

    // So if we start with false, then set to true, it shows.
    // If we start with true and products, it shows.

    // Let's test the "waiting" state logic if applicable, or just simple render
  });

  it('renders product cards correctly', async () => {
    const page = await newSpecPage({
      components: [BcxProductSlider],
      html: `<bcx-product-slider></bcx-product-slider>`,
    });

    page.rootInstance.products = mockProducts;
    await page.waitForChanges();

    const cards = page.root.shadowRoot.querySelectorAll('.bcx-product-slider__card');
    expect(cards).toHaveLength(3);

    const firstCard = cards[0];
    expect(firstCard.querySelector('.bcx-product-slider__card-title').textContent).toEqual('Product 1');
    expect(firstCard.querySelector('img').getAttribute('src')).toEqual('http://example.com/img1.jpg');
  });

  it('renders navigation dots when multiple products exist', async () => {
    const page = await newSpecPage({
      components: [BcxProductSlider],
      html: `<bcx-product-slider></bcx-product-slider>`,
    });

    page.rootInstance.products = mockProducts;
    await page.waitForChanges();

    const dots = page.root.shadowRoot.querySelectorAll('.bcx-product-slider__dot');
    expect(dots).toHaveLength(3);
    expect(dots[0]).toHaveClass('bcx-product-slider__dot--active');
  });

  it('does not render dots for single product', async () => {
    const page = await newSpecPage({
      components: [BcxProductSlider],
      html: `<bcx-product-slider></bcx-product-slider>`,
    });

    page.rootInstance.products = [mockProducts[0]];
    await page.waitForChanges();

    const dots = page.root.shadowRoot.querySelectorAll('.bcx-product-slider__dot');
    expect(dots).toHaveLength(0);
  });

  it('navigates to slide when dot is clicked', async () => {
    const page = await newSpecPage({
      components: [BcxProductSlider],
      html: `<bcx-product-slider></bcx-product-slider>`,
    });

    page.rootInstance.products = mockProducts;
    await page.waitForChanges();

    const dots = page.root.shadowRoot.querySelectorAll('.bcx-product-slider__dot');
    const secondDot = dots[1] as HTMLElement;

    secondDot.click();
    await page.waitForChanges();

    expect(dots[1]).toHaveClass('bcx-product-slider__dot--active');
    expect(dots[0]).not.toHaveClass('bcx-product-slider__dot--active');

    // Check transform logic if possible, or just state
    expect(page.rootInstance.currentIndex).toBe(1);
  });

  it('opens product url on card click', async () => {
    const page = await newSpecPage({
      components: [BcxProductSlider],
      html: `<bcx-product-slider></bcx-product-slider>`,
    });

    page.rootInstance.products = mockProducts;
    await page.waitForChanges();

    const firstCard = page.root.shadowRoot.querySelector('.bcx-product-slider__card') as HTMLElement;
    firstCard.click();

    expect(window.open).toHaveBeenCalledWith('http://example.com/p1', '_blank', expect.stringContaining('noopener'));
  });

  it('translates text based on language prop', async () => {
    const page = await newSpecPage({
      components: [BcxProductSlider],
      html: `<bcx-product-slider language="pl"></bcx-product-slider>`,
    });

    page.rootInstance.products = mockProducts;
    await page.waitForChanges();

    const linkText = page.root.shadowRoot.querySelector('.bcx-product-slider__card-link').textContent;
    expect(linkText).toBe('Zobacz produkt');
  });

  it('handles dragging logic (mouse events)', async () => {
    const page = await newSpecPage({
      components: [BcxProductSlider],
      html: `<bcx-product-slider></bcx-product-slider>`,
    });

    page.rootInstance.products = mockProducts;
    await page.waitForChanges();

    const track = page.root.shadowRoot.querySelector('.bcx-product-slider__track') as HTMLElement;

    // Mock getComputedStyle for track transform check logic inside component
    // Ideally we should test if internal state changes or style updates

    // Simulate Mouse Down
    const mousedown = new MouseEvent('mousedown', { clientX: 100, bubbles: true });
    track.dispatchEvent(mousedown);

    // Simulate Mouse Move (drag left by 50px)
    const mousemove = new MouseEvent('mousemove', { clientX: 50, bubbles: true });
    track.dispatchEvent(mousemove); // or document/window listener if attached there
    // The component attaches listeners to sliderRef which is track? No, listeners in didLoad
    // listeners are attached to sliderRef

    // Simulate Mouse Up
    const mouseup = new MouseEvent('mouseup', { clientX: 50, bubbles: true });
    track.dispatchEvent(mouseup);

    // Since jsdom doesn't do layout, we can't verify exact pixel transforms easily
    // without mocking offsetWidth etc.
    // But we can check if event handlers were called if we spy on them,
    // or check side effects like `isDragging` property if public/accessible.
    // In Stencil spec tests, private props are accessible on rootInstance (casted to any).

    // For now, verifying that basic rendering and interaction handlers don't crash is good baseline.
  });

  it('prevents click when dragging', async () => {
    const page = await newSpecPage({
      components: [BcxProductSlider],
      html: `<bcx-product-slider></bcx-product-slider>`,
    });

    page.rootInstance.products = mockProducts;
    await page.waitForChanges();

    // Set dragging state manually
    (page.rootInstance as any).isDragging = true;

    const firstCard = page.root.shadowRoot.querySelector('.bcx-product-slider__card') as HTMLElement;
    firstCard.click();

    expect(window.open).not.toHaveBeenCalled();
  });
});

