import { newSpecPage } from '@stencil/core/testing';
import { BcxMessageComposer } from '../bcx-message-composer';

describe('bcx-message-composer', () => {
  it('renders with default props', async () => {
    const page = await newSpecPage({
      components: [BcxMessageComposer],
      html: '<bcx-message-composer></bcx-message-composer>',
    });

    expect(page.root).toEqualHtml(`
      <bcx-message-composer>
        <mock:shadow-root>
          <form aria-label="Message composer" class="bcx-composer" data-adblock-bypass="true" role="form">
            <input accept="image/png,image/jpg,image/jpeg,image/gif,image/webp" data-adblock-bypass="true" multiple="" style="display: none;" type="file">
            <div class="bcx-composer__field">
              <textarea aria-label="Type your message" class="bcx-composer__input" data-adblock-bypass="true" maxlength="1000" placeholder="Type your message..." rows="1" value=""></textarea>
              <div class="bcx-composer__actions">
                <button aria-label="Add image" class="bcx-composer__media-btn" data-adblock-bypass="true" type="button">
                  <svg aria-hidden="true" fill="none" height="18" stroke="currentColor" stroke-linecap="round" stroke-linejoin="round" stroke-width="2" viewBox="0 0 24 24" width="18">
                    <rect height="18" rx="2" ry="2" width="18" x="3" y="3"></rect>
                    <circle cx="8.5" cy="8.5" r="1.5"></circle>
                    <polyline points="21,15 16,10 5,21"></polyline>
                  </svg>
                </button>
                <button aria-label="Send message" class="bcx-composer__submit" data-adblock-bypass="true" data-loading="false" disabled="" type="submit">
                  <span aria-hidden="true" class="bcx-composer__submit-icon">
                    <svg fill="none" height="18" stroke="currentColor" stroke-linecap="round" stroke-linejoin="round" stroke-width="2" viewBox="0 0 24 24" width="18">
                      <line x1="22" x2="11" y1="2" y2="13"></line>
                      <polygon points="22,2 15,22 11,13 2,9 22,2"></polygon>
                    </svg>
                  </span>
                </button>
              </div>
              </div>
            </form>
        </mock:shadow-root>
      </bcx-message-composer>
    `);
  });

  it('renders with custom props', async () => {
    const page = await newSpecPage({
      components: [BcxMessageComposer],
      html: '<bcx-message-composer placeholder="Custom placeholder" max-length="500" disabled></bcx-message-composer>',
    });

    const textarea = page.root.shadowRoot.querySelector('textarea');
    const button = page.root.shadowRoot.querySelector('button[type="submit"]');

    expect(textarea.getAttribute('placeholder')).toBe('Custom placeholder');
    expect(textarea.getAttribute('maxlength')).toBe('500');
    expect(button.hasAttribute('disabled')).toBe(true);
  });

  it('enables submit button when message is entered', async () => {
    const page = await newSpecPage({
      components: [BcxMessageComposer],
      html: '<bcx-message-composer></bcx-message-composer>',
    });

    const component = page.rootInstance as BcxMessageComposer;
    const button = page.root.shadowRoot.querySelector('button[type="submit"]');

    // Initially disabled
    expect(button.hasAttribute('disabled')).toBe(true);

    // Enter message
    component.message = 'Hello world';
    await page.waitForChanges();

    expect(button.hasAttribute('disabled')).toBe(false);
  });

  it('emits messageSubmit event on form submit', async () => {
    const page = await newSpecPage({
      components: [BcxMessageComposer],
      html: '<bcx-message-composer></bcx-message-composer>',
    });

    const component = page.rootInstance as BcxMessageComposer;
    const form = page.root.shadowRoot.querySelector('form');

    // Set up event listener
    const messageSubmitSpy = jest.fn();
    page.root.addEventListener('messageSubmit', messageSubmitSpy);

    // Enter message and submit
    component.message = 'Test message';
    await page.waitForChanges();

    form.dispatchEvent(new Event('submit'));
    await page.waitForChanges();

    expect(messageSubmitSpy).toHaveBeenCalledWith(
      expect.objectContaining({
        detail: { content: 'Test message', images: [] },
      }),
    );
  });

  it('handles Enter key submission', async () => {
    const page = await newSpecPage({
      components: [BcxMessageComposer],
      html: '<bcx-message-composer></bcx-message-composer>',
    });

    const component = page.rootInstance as BcxMessageComposer;
    const textarea = page.root.shadowRoot.querySelector('textarea');

    // Set up event listener
    const messageSubmitSpy = jest.fn();
    page.root.addEventListener('messageSubmit', messageSubmitSpy);

    // Enter message
    component.message = 'Test message';
    await page.waitForChanges();

    // Simulate Enter key
    const enterEvent = new KeyboardEvent('keydown', { key: 'Enter' });
    textarea.dispatchEvent(enterEvent);
    await page.waitForChanges();

    expect(messageSubmitSpy).toHaveBeenCalledWith(
      expect.objectContaining({
        detail: { content: 'Test message', images: [] },
      }),
    );
  });

  it('does not submit on Shift+Enter', async () => {
    const page = await newSpecPage({
      components: [BcxMessageComposer],
      html: '<bcx-message-composer></bcx-message-composer>',
    });

    const component = page.rootInstance as BcxMessageComposer;
    const textarea = page.root.shadowRoot.querySelector('textarea');

    // Set up event listener
    const messageSubmitSpy = jest.fn();
    page.root.addEventListener('messageSubmit', messageSubmitSpy);

    // Enter message
    component.message = 'Test message';
    await page.waitForChanges();

    // Simulate Shift+Enter key
    const shiftEnterEvent = new KeyboardEvent('keydown', { key: 'Enter', shiftKey: true });
    textarea.dispatchEvent(shiftEnterEvent);
    await page.waitForChanges();

    expect(messageSubmitSpy).not.toHaveBeenCalled();
  });
});
