<?php
/**
 * Plugin Name: BetterCX Widget
 * Plugin URI: https://wordpress.org/plugins/bettercx-widget/
 * Description: Professional AI-powered chat widget for BetterCX platform. Seamlessly integrate intelligent customer support into any website with full WordPress compatibility. Fully functional out of the box with no trial limitations.
 * Version: 1.0.20
 * Author: BetterCX
 * Author URI: https://bettercx.ai
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: bettercx-widget
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.8
 * Requires PHP: 7.4
 *
 * @package BetterCX_Widget
 * @version 1.0.20
 * @author BetterCX
 * @license GPLv2+
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('BETTERCX_WIDGET_VERSION', '1.0.20');
define('BETTERCX_WIDGET_PLUGIN_FILE', __FILE__);
define('BETTERCX_WIDGET_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('BETTERCX_WIDGET_PLUGIN_URL', plugin_dir_url(__FILE__));
define('BETTERCX_WIDGET_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Main BetterCX Widget Plugin Class
 *
 * This class handles all plugin functionality including:
 * - Widget rendering and shortcode support
 * - Admin settings and configuration
 * - Asset management and enqueueing
 * - Event handling and JavaScript integration
 * - WordPress standards compliance
 *
 * @since 1.0.0
 */
class BetterCX_Widget {

    /**
     * Plugin instance
     *
     * @since 1.0.0
     * @var BetterCX_Widget
     */
    private static $instance = null;

    /**
     * Plugin settings
     *
     * @since 1.0.0
     * @var array
     */
    private $settings = array();

    /**
     * Get plugin instance
     *
     * @since 1.0.0
     * @return BetterCX_Widget
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     *
     * @since 1.0.0
     */
    private function __construct() {
        $this->init_hooks();
        $this->load_settings();
    }

    /**
     * Initialize WordPress hooks
     *
     * @since 1.0.0
     */
    private function init_hooks() {
        // Plugin activation/deactivation
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));

        // WordPress hooks
        add_action('init', array($this, 'init'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('admin_enqueue_scripts', array($this, 'admin_enqueue_scripts'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('wp_footer', array($this, 'render_widget'));

        // Shortcode
        add_shortcode('bettercx_widget', array($this, 'widget_shortcode'));

        // Widget for widget areas
        add_action('widgets_init', array($this, 'register_widget'));

        // AJAX handlers
        add_action('wp_ajax_bettercx_save_settings', array($this, 'ajax_save_settings'));
        add_action('wp_ajax_bettercx_test_connection', array($this, 'ajax_test_connection'));

        // Allow custom elements in content
        add_filter('wp_kses_allowed_html', array($this, 'allow_custom_elements'), 10, 2);

        // Allow script tags for event handling
        add_filter('wp_kses_allowed_html', array($this, 'allow_script_tags'), 10, 2);

        // Privacy policy content
        add_action('admin_init', array($this, 'add_privacy_policy_content'));
    }

    /**
     * Initialize plugin
     *
     * @since 1.0.0
     */
    public function init() {
        // Text domain is automatically loaded by WordPress for plugins hosted on WordPress.org
    }

    /**
     * Load plugin settings
     *
     * @since 1.0.0
     */
    private function load_settings() {
        $this->settings = wp_parse_args(
            get_option('bettercx_widget_settings', array()),
            $this->get_default_settings()
        );
    }

    /**
     * Get default settings
     *
     * @since 1.0.0
     * @return array
     */
    private function get_default_settings() {
        return array(
            'public_key' => '',
            'theme' => 'auto',
            'position' => 'right',
            'language' => 'auto',
            'embedded' => false,
            'embedded_size' => 'full',
            'embedded_placement' => 'center',
            'debug' => false,
            'auto_init' => true,
            'show_credits' => false,
            'enable_analytics' => false,
            'privacy_consent' => false,
            'custom_css' => '',
            'widget_enabled' => true,
            'logged_in_only' => false, // Show widget only for logged-in users
            'base_url' => 'https://api.bettercx.ai',
            'ai_service_url' => 'https://widget.bettercx.ai/prod',
            'is_attachments_disabled' => false,
        );
    }

    /**
     * Plugin activation
     *
     * @since 1.0.0
     */
    public function activate() {
        // Set default options
        add_option('bettercx_widget_settings', $this->get_default_settings());
        add_option('bettercx_widget_version', BETTERCX_WIDGET_VERSION);

        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Plugin deactivation
     *
     * @since 1.0.0
     */
    public function deactivate() {
        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Enqueue frontend scripts and styles
     *
     * @since 1.0.0
     */
    public function enqueue_scripts() {
        // Only load if widget is enabled and we have a public key
        if (!$this->settings['widget_enabled'] || empty($this->settings['public_key'])) {
            return;
        }

        // Check if widget should only be shown to logged-in users
        if (!empty($this->settings['logged_in_only']) && !is_user_logged_in()) {
            return;
        }

        // Enqueue Stencil component as ES module
        wp_enqueue_script(
            'bettercx-widget',
            BETTERCX_WIDGET_PLUGIN_URL . 'assets/bettercx-widget.esm.js',
            array(),
            BETTERCX_WIDGET_VERSION,
            true
        );

        // Add type="module" attribute
        add_filter('script_loader_tag', array($this, 'add_module_type_attribute'), 10, 3);

        wp_enqueue_style(
            'bettercx-widget',
            BETTERCX_WIDGET_PLUGIN_URL . 'assets/bettercx-widget.css',
            array(),
            BETTERCX_WIDGET_VERSION
        );

        // Add inline configuration with proper sanitization
        $config = array(
            'publicKey' => sanitize_text_field($this->settings['public_key']),
            'theme' => sanitize_text_field($this->settings['theme']),
            'position' => sanitize_text_field($this->settings['position']),
            'language' => sanitize_text_field($this->settings['language']),
            'embedded' => (bool) $this->settings['embedded'],
            'embeddedSize' => sanitize_text_field($this->settings['embedded_size']),
            'embeddedPlacement' => sanitize_text_field($this->settings['embedded_placement']),
            'debug' => (bool) $this->settings['debug'],
            'autoInit' => (bool) $this->settings['auto_init'],
            'showCredits' => (bool) $this->settings['show_credits'],
            'enableAnalytics' => (bool) $this->settings['enable_analytics'],
            'privacyConsent' => (bool) $this->settings['privacy_consent'],
            'baseUrl' => esc_url_raw($this->settings['base_url']),
            'aiServiceUrl' => esc_url_raw($this->settings['ai_service_url']),
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('bettercx_widget_nonce'),
            'isAttachmentsDisabled' => (bool) $this->settings['is_attachments_disabled'],
        );

        wp_localize_script('bettercx-widget', 'bettercxWidgetConfig', $config);

        // Add custom CSS if provided (with proper sanitization)
        if (!empty($this->settings['custom_css'])) {
            $custom_css = wp_strip_all_tags($this->settings['custom_css']);
            $custom_css = wp_kses($custom_css, array());
            wp_add_inline_style('bettercx-widget', $custom_css);
        }

        // Add event handling script
        $this->enqueue_event_handling_script();
    }

    /**
     * Add type="module" attribute to script tags
     *
     * @since 1.0.0
     * @param string $tag The script tag HTML
     * @param string $handle The script handle
     * @param string $src The script source URL
     * @return string Modified script tag
     */
    public function add_module_type_attribute($tag, $handle, $src) {
        if ('bettercx-widget' === $handle) {
            return str_replace('<script ', '<script type="module" ', $tag);
        }
        return $tag;
    }

    /**
     * Enqueue admin scripts and styles
     *
     * @since 1.0.0
     */
    public function admin_enqueue_scripts($hook) {
        if ('settings_page_bettercx-widget' !== $hook) {
            return;
        }

        wp_enqueue_script(
            'bettercx-widget-admin',
            BETTERCX_WIDGET_PLUGIN_URL . 'assets/admin.js',
            array('jquery'),
            BETTERCX_WIDGET_VERSION,
            true
        );

        wp_enqueue_style(
            'bettercx-widget-admin',
            BETTERCX_WIDGET_PLUGIN_URL . 'assets/admin.css',
            array(),
            BETTERCX_WIDGET_VERSION
        );

        wp_localize_script('bettercx-widget-admin', 'bettercxWidgetAdmin', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('bettercx_widget_admin_nonce'),
            'strings' => array(
                'testConnectionSuccess' => __('Connection test successful!', 'bettercx-widget'),
                'testConnectionError' => __('Connection test failed. Please check your settings.', 'bettercx-widget'),
                'settingsSaved' => __('Settings saved successfully!', 'bettercx-widget'),
                'settingsError' => __('Error saving settings. Please try again.', 'bettercx-widget'),
            ),
        ));
    }

    /**
     * Add admin menu
     *
     * @since 1.0.0
     */
    public function add_admin_menu() {
        add_options_page(
            __('BetterCX Widget Settings', 'bettercx-widget'),
            __('BetterCX Widget', 'bettercx-widget'),
            'manage_options',
            'bettercx-widget',
            array($this, 'admin_page')
        );
    }

    /**
     * Register settings
     *
     * @since 1.0.0
     */
    public function register_settings() {
        register_setting(
            'bettercx_widget_settings',
            'bettercx_widget_settings',
            array($this, 'sanitize_settings')
        );

        // Settings sections
        add_settings_section(
            'bettercx_widget_general',
            __('General Settings', 'bettercx-widget'),
            array($this, 'general_section_callback'),
            'bettercx-widget'
        );

        add_settings_section(
            'bettercx_widget_appearance',
            __('Appearance Settings', 'bettercx-widget'),
            array($this, 'appearance_section_callback'),
            'bettercx-widget'
        );

        add_settings_section(
            'bettercx_widget_advanced',
            __('Advanced Settings', 'bettercx-widget'),
            array($this, 'advanced_section_callback'),
            'bettercx-widget'
        );

        // General settings fields
        add_settings_field(
            'public_key',
            __('Public Key', 'bettercx-widget'),
            array($this, 'public_key_field_callback'),
            'bettercx-widget',
            'bettercx_widget_general'
        );

        add_settings_field(
            'widget_enabled',
            __('Enable Widget', 'bettercx-widget'),
            array($this, 'widget_enabled_field_callback'),
            'bettercx-widget',
            'bettercx_widget_general'
        );

        // Appearance settings fields
        add_settings_field(
            'theme',
            __('Theme', 'bettercx-widget'),
            array($this, 'theme_field_callback'),
            'bettercx-widget',
            'bettercx_widget_appearance'
        );

        add_settings_field(
            'position',
            __('Position', 'bettercx-widget'),
            array($this, 'position_field_callback'),
            'bettercx-widget',
            'bettercx_widget_appearance'
        );

        add_settings_field(
            'language',
            __('Language', 'bettercx-widget'),
            array($this, 'language_field_callback'),
            'bettercx-widget',
            'bettercx_widget_appearance'
        );

        add_settings_field(
            'show_credits',
            __('Show Credits', 'bettercx-widget'),
            array($this, 'show_credits_field_callback'),
            'bettercx-widget',
            'bettercx_widget_appearance'
        );

        // Advanced settings fields
        add_settings_field(
            'debug',
            __('Debug Mode', 'bettercx-widget'),
            array($this, 'debug_field_callback'),
            'bettercx-widget',
            'bettercx_widget_advanced'
        );

        add_settings_field(
            'auto_init',
            __('Auto Initialize', 'bettercx-widget'),
            array($this, 'auto_init_field_callback'),
            'bettercx-widget',
            'bettercx_widget_advanced'
        );

        add_settings_field(
            'embedded',
            __('Embedded Mode', 'bettercx-widget'),
            array($this, 'embedded_field_callback'),
            'bettercx-widget',
            'bettercx_widget_advanced'
        );

        add_settings_field(
            'embedded_size',
            __('Embedded Size', 'bettercx-widget'),
            array($this, 'embedded_size_field_callback'),
            'bettercx-widget',
            'bettercx_widget_advanced'
        );

        add_settings_field(
            'embedded_placement',
            __('Embedded Placement', 'bettercx-widget'),
            array($this, 'embedded_placement_field_callback'),
            'bettercx-widget',
            'bettercx_widget_advanced'
        );

        add_settings_field(
            'enable_analytics',
            __('Enable Analytics', 'bettercx-widget'),
            array($this, 'enable_analytics_field_callback'),
            'bettercx-widget',
            'bettercx_widget_advanced'
        );

        add_settings_field(
            'privacy_consent',
            __('Privacy Consent Required', 'bettercx-widget'),
            array($this, 'privacy_consent_field_callback'),
            'bettercx-widget',
            'bettercx_widget_advanced'
        );

        add_settings_field(
            'logged_in_only',
            __('Logged-in Users Only', 'bettercx-widget'),
            array($this, 'logged_in_only_field_callback'),
            'bettercx-widget',
            'bettercx_widget_advanced'
        );

        add_settings_field(
            'is_attachments_disabled',
            __('Disable Attachments', 'bettercx-widget'),
            array($this, 'is_attachments_disabled_field_callback'),
            'bettercx-widget',
            'bettercx_widget_advanced'
        );

        add_settings_field(
            'custom_css',
            __('Custom CSS', 'bettercx-widget'),
            array($this, 'custom_css_field_callback'),
            'bettercx-widget',
            'bettercx_widget_advanced'
        );

        add_settings_field(
            'base_url',
            __('Base URL', 'bettercx-widget'),
            array($this, 'base_url_field_callback'),
            'bettercx-widget',
            'bettercx_widget_advanced'
        );

        add_settings_field(
            'ai_service_url',
            __('AI Service URL', 'bettercx-widget'),
            array($this, 'ai_service_url_field_callback'),
            'bettercx-widget',
            'bettercx_widget_advanced'
        );
    }

    /**
     * Deep sanitize text array recursively
     *
     * @since 1.0.0
     * @param array $array
     * @return array
     */
    private function deep_sanitize_text_array($array) {
        foreach ($array as $key => $value) {
            $array[$key] = is_array($value)
                ? $this->deep_sanitize_text_array($value)
                : sanitize_text_field($value);
        }
        return $array;
    }

    /**
     * Sanitize settings
     *
     * @since 1.0.0
     * @param array $input
     * @return array
     */
    public function sanitize_settings($input) {
        $sanitized = array();

        if (isset($input['public_key'])) {
            $sanitized['public_key'] = sanitize_text_field($input['public_key']);
        }

        if (isset($input['theme'])) {
            $allowed_themes = array('auto', 'light', 'dark');
            $sanitized['theme'] = in_array($input['theme'], $allowed_themes) ? $input['theme'] : 'auto';
        }

        if (isset($input['position'])) {
            $allowed_positions = array('left', 'right');
            $sanitized['position'] = in_array($input['position'], $allowed_positions) ? $input['position'] : 'right';
        }

        if (isset($input['language'])) {
            $allowed_languages = array('auto', 'pl', 'en');
            $sanitized['language'] = in_array($input['language'], $allowed_languages) ? $input['language'] : 'auto';
        }

        $sanitized['embedded'] = isset($input['embedded']) ? (bool) $input['embedded'] : false;

        if (isset($input['embedded_size'])) {
            $allowed_sizes = array('full', 'medium', 'small');
            $sanitized['embedded_size'] = in_array($input['embedded_size'], $allowed_sizes) ? $input['embedded_size'] : 'full';
        }

        if (isset($input['embedded_placement'])) {
            $allowed_placements = array('top', 'center', 'bottom');
            $sanitized['embedded_placement'] = in_array($input['embedded_placement'], $allowed_placements) ? $input['embedded_placement'] : 'center';
        }

        $sanitized['debug'] = isset($input['debug']) ? (bool) $input['debug'] : false;
        $sanitized['auto_init'] = isset($input['auto_init']) ? (bool) $input['auto_init'] : true;
        $sanitized['show_credits'] = isset($input['show_credits']) ? (bool) $input['show_credits'] : false;
        $sanitized['enable_analytics'] = isset($input['enable_analytics']) ? (bool) $input['enable_analytics'] : false;
        $sanitized['privacy_consent'] = isset($input['privacy_consent']) ? (bool) $input['privacy_consent'] : false;
        $sanitized['widget_enabled'] = isset($input['widget_enabled']) ? (bool) $input['widget_enabled'] : true;
        $sanitized['logged_in_only'] = isset($input['logged_in_only']) ? (bool) $input['logged_in_only'] : false;
        $sanitized['is_attachments_disabled'] = isset($input['is_attachments_disabled']) ? (bool) $input['is_attachments_disabled'] : false;

        if (isset($input['custom_css'])) {
            $sanitized['custom_css'] = wp_strip_all_tags($input['custom_css']);
        }

        if (isset($input['base_url'])) {
            $sanitized['base_url'] = esc_url_raw($input['base_url']);
        }

        if (isset($input['ai_service_url'])) {
            $sanitized['ai_service_url'] = esc_url_raw($input['ai_service_url']);
        }

        return $sanitized;
    }

    /**
     * Widget shortcode handler
     *
     * @since 1.0.0
     * @param array $atts
     * @return string
     */
    public function widget_shortcode($atts) {
        // Check if widget should only be shown to logged-in users
        if (!empty($this->settings['logged_in_only']) && !is_user_logged_in()) {
            return '';
        }

        // Parse shortcode attributes
        $atts = shortcode_atts(array(
            'public_key' => $this->settings['public_key'],
            'theme' => $this->settings['theme'],
            'position' => $this->settings['position'],
            'language' => $this->settings['language'],
            'embedded' => $this->settings['embedded'] ? 'true' : 'false',
            'embedded_size' => $this->settings['embedded_size'],
            'embedded_placement' => $this->settings['embedded_placement'],
            'debug' => $this->settings['debug'] ? 'true' : 'false',
            'auto_init' => $this->settings['auto_init'] ? 'true' : 'false',
            'base_url' => $this->settings['base_url'],
            'ai_service_url' => $this->settings['ai_service_url'],
            'is_attachments_disabled' => $this->settings['is_attachments_disabled'] ? 'true' : 'false',
            'class_name' => '',
            'style' => '',
        ), $atts, 'bettercx_widget');

        // Validate public key
        if (empty($atts['public_key'])) {
            if (current_user_can('manage_options')) {
                return '<div class="bettercx-widget-error">' .
                       __('BetterCX Widget: Public key is required. Please configure it in the plugin settings.', 'bettercx-widget') .
                       '</div>';
            }
            return '';
        }

        // Build widget attributes
        $attributes = array();
        foreach ($atts as $key => $value) {
            // Skip empty values, class_name, and style, but include embedded_size and embedded_placement even if they are default values
            if ($key === 'class_name' || $key === 'style') {
                continue;
            }

            // Include embedded_size and embedded_placement if embedded is true
            if (($key === 'embedded_size' || $key === 'embedded_placement') && $atts['embedded'] !== 'true') {
                continue;
            }

            if (!empty($value) || ($key === 'embedded_size' || $key === 'embedded_placement')) {
                $attributes[] = sprintf(
                    '%s="%s"',
                    esc_attr(str_replace('_', '-', $key)),
                    esc_attr($value)
                );
            }
        }

        // Add custom class and style
        $class_attr = !empty($atts['class_name']) ? ' class="' . esc_attr($atts['class_name']) . '"' : '';
        $style_attr = !empty($atts['style']) ? ' style="' . esc_attr($atts['style']) . '"' : '';

        // Render widget
        $html = '<bettercx-widget ' . implode(' ', $attributes) . $class_attr . $style_attr . '></bettercx-widget>';

        // Add debugging script for development
        if ($this->settings['debug']) {
            $this->enqueue_debug_script();
        }

        return $html;
    }

    /**
     * Render widget in footer
     *
     * @since 1.0.0
     */
    public function render_widget() {
        // Only render if widget is enabled and we have a public key
        if (!$this->settings['widget_enabled'] || empty($this->settings['public_key'])) {
            return;
        }

        // Check if widget should only be shown to logged-in users
        if (!empty($this->settings['logged_in_only']) && !is_user_logged_in()) {
            return;
        }

        // Check if widget should be rendered (not already rendered via shortcode)
        if (is_singular() && has_shortcode(get_post()->post_content, 'bettercx_widget')) {
            return;
        }

        // Render global widget
        echo wp_kses_post($this->widget_shortcode(array()));
    }

    /**
     * Allow custom elements in content
     *
     * @since 1.0.0
     * @param array $allowed_html
     * @param string $context
     * @return array
     */
    public function allow_custom_elements($allowed_html, $context) {
        if ($context === 'post') {
            $allowed_html['bettercx-widget'] = array(
                'public-key' => true,
                'theme' => true,
                'position' => true,
                'language' => true,
                'embedded' => true,
                'embedded-size' => true,
                'embedded-placement' => true,
                'debug' => true,
                'auto-init' => true,
                'base-url' => true,
                'ai-service-url' => true,
                'is-attachments-disabled' => true,
                'class' => true,
                'style' => true,
            );
        }
        return $allowed_html;
    }

    /**
     * Allow script tags for event handling
     *
     * @since 1.0.0
     * @param array $allowed_html
     * @param string $context
     * @return array
     */
    public function allow_script_tags($allowed_html, $context) {
        if ($context === 'post') {
            $allowed_html['script'] = array(
                'type' => true,
            );
        }
        return $allowed_html;
    }

    /**
     * Enqueue debug script for development
     *
     * @since 1.0.0
     */
    private function enqueue_debug_script() {
        $debug_script = "
            console.log('BetterCX Widget Debug:');
            console.log('Config:', window.bettercxWidgetConfig);
            console.log('Widget element:', document.querySelector('bettercx-widget'));
            console.log('Custom elements defined:', customElements.get('bettercx-widget'));
        ";

        wp_add_inline_script('bettercx-widget', $debug_script);
    }

    /**
     * Enqueue event handling script
     *
     * @since 1.0.0
     */
    private function enqueue_event_handling_script() {
        $event_script = "
document.addEventListener('DOMContentLoaded', function() {
    const widget = document.querySelector('bettercx-widget');
    if (widget && window.bettercxWidgetConfig) {
        // Set widget attributes from config if not already set
        const config = window.bettercxWidgetConfig;

        if (config.embeddedSize && !widget.hasAttribute('embedded-size')) {
            widget.setAttribute('embedded-size', config.embeddedSize);
        }

        if (config.embeddedPlacement && !widget.hasAttribute('embedded-placement')) {
            widget.setAttribute('embedded-placement', config.embeddedPlacement);
        }

        widget.addEventListener('widgetEvent', function(e) {
            // Trigger custom WordPress action
            if (typeof jQuery !== 'undefined') {
                jQuery(document).trigger('bettercx:widgetEvent', [e.detail]);
            }

            // Trigger global function if defined
            if (typeof window.BetterCXWidgetEvent === 'function') {
                window.BetterCXWidgetEvent(e.detail);
            }

            // Analytics tracking if enabled
            if (config.enableAnalytics) {
                if (typeof gtag !== 'undefined') {
                    gtag('event', 'bettercx_widget_event', {
                        event_type: e.detail.type,
                        event_timestamp: e.detail.timestamp
                    });
                }
            }
        });
    }
});
        ";

        wp_add_inline_script('bettercx-widget', $event_script);
    }

    /**
     * Register widget for widget areas
     *
     * @since 1.0.0
     */
    public function register_widget() {
        register_widget('BetterCX_Widget_Widget');
    }

    /**
     * AJAX handler for saving settings
     *
     * @since 1.0.0
     */
    public function ajax_save_settings() {
        check_ajax_referer('bettercx_widget_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Insufficient permissions.', 'bettercx-widget'));
        }

        // Get and sanitize POST data - WordPress CodeSniffer compliant approach
        $raw_settings = array();

        if ( isset( $_POST['settings'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification
            $unsanitized_settings = wp_unslash( $_POST['settings'] ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
            if ( is_array( $unsanitized_settings ) ) {
                $raw_settings = $this->deep_sanitize_text_array( $unsanitized_settings );
            }
        }



        $settings = $this->sanitize_settings($raw_settings);
        update_option('bettercx_widget_settings', $settings);

        wp_send_json_success(esc_html__('Settings saved successfully!', 'bettercx-widget'));
    }

    /**
     * AJAX handler for testing connection
     *
     * @since 1.0.0
     */
    public function ajax_test_connection() {
        check_ajax_referer('bettercx_widget_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Insufficient permissions.', 'bettercx-widget'));
        }

        $public_key = isset($_POST['public_key']) ? sanitize_text_field(wp_unslash($_POST['public_key'])) : '';

        // Test connection logic here
        $response = wp_remote_get('https://api.bettercx.ai/api/widgets/session/create/', array(
            'timeout' => 10,
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
            'body' => json_encode(array(
                'widget_key' => $public_key,
                'origin' => home_url(),
            )),
        ));

        if (is_wp_error($response)) {
            wp_send_json_error(__('Connection test failed: ', 'bettercx-widget') . $response->get_error_message());
        }

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code === 200) {
            wp_send_json_success(__('Connection test successful!', 'bettercx-widget'));
        } else {
            wp_send_json_error(__('Connection test failed. Please check your public key.', 'bettercx-widget'));
        }
    }

    /**
     * Add privacy policy content
     *
     * @since 1.0.0
     */
    public function add_privacy_policy_content() {
        if (!function_exists('wp_add_privacy_policy_content')) {
            return;
        }

        $content = sprintf(
            '<h2>%s</h2>
            <p>%s</p>
            <p>%s</p>
            <p>%s</p>
            <p>%s</p>',
            __('BetterCX Widget', 'bettercx-widget'),
            __('This plugin integrates the BetterCX chat widget to provide AI-powered customer support on your website.', 'bettercx-widget'),
            __('The widget may collect the following information:', 'bettercx-widget'),
            __('• Messages sent through the chat widget', 'bettercx-widget'),
            __('• Session information for authentication purposes', 'bettercx-widget')
        );

        wp_add_privacy_policy_content(
            __('BetterCX Widget', 'bettercx-widget'),
            wp_kses_post($content)
        );
    }

    // Settings field callbacks
    public function general_section_callback() {
        echo '<p>' . esc_html__('Configure the basic settings for the BetterCX widget.', 'bettercx-widget') . '</p>';
    }

    public function appearance_section_callback() {
        echo '<p>' . esc_html__('Customize the appearance and behavior of the widget.', 'bettercx-widget') . '</p>';
    }

    public function advanced_section_callback() {
        echo '<p>' . esc_html__('Advanced configuration options for developers.', 'bettercx-widget') . '</p>';
    }

    public function public_key_field_callback() {
        $value = isset($this->settings['public_key']) ? $this->settings['public_key'] : '';
        echo '<input type="text" name="bettercx_widget_settings[public_key]" value="' . esc_attr($value) . '" class="regular-text" required />';
        echo '<p class="description">' . esc_html__('Enter your BetterCX widget public key. You can get this from your BetterCX dashboard.', 'bettercx-widget') . '</p>';
    }

    public function widget_enabled_field_callback() {
        $value = isset($this->settings['widget_enabled']) ? $this->settings['widget_enabled'] : true;
        echo '<input type="checkbox" name="bettercx_widget_settings[widget_enabled]" value="1" ' . checked($value, true, false) . ' />';
        echo '<p class="description">' . esc_html__('Enable or disable the widget globally.', 'bettercx-widget') . '</p>';
    }

    public function theme_field_callback() {
        $value = isset($this->settings['theme']) ? $this->settings['theme'] : 'auto';
        $themes = array(
            'auto' => esc_html__('Auto (detect from website)', 'bettercx-widget'),
            'light' => esc_html__('Light', 'bettercx-widget'),
            'dark' => esc_html__('Dark', 'bettercx-widget'),
        );

        echo '<select name="bettercx_widget_settings[theme]">';
        foreach ($themes as $key => $label) {
            echo '<option value="' . esc_attr($key) . '" ' . selected($value, $key, false) . '>' . esc_html($label) . '</option>';
        }
        echo '</select>';
    }

    public function position_field_callback() {
        $value = isset($this->settings['position']) ? $this->settings['position'] : 'right';
        $positions = array(
            'right' => esc_html__('Right', 'bettercx-widget'),
            'left' => esc_html__('Left', 'bettercx-widget'),
        );

        echo '<select name="bettercx_widget_settings[position]">';
        foreach ($positions as $key => $label) {
            echo '<option value="' . esc_attr($key) . '" ' . selected($value, $key, false) . '>' . esc_html($label) . '</option>';
        }
        echo '</select>';
    }

    public function language_field_callback() {
        $value = isset($this->settings['language']) ? $this->settings['language'] : 'auto';
        $languages = array(
            'auto' => esc_html__('Auto (detect from website)', 'bettercx-widget'),
            'pl' => esc_html__('Polish', 'bettercx-widget'),
            'en' => esc_html__('English', 'bettercx-widget'),
        );

        echo '<select name="bettercx_widget_settings[language]">';
        foreach ($languages as $key => $label) {
            echo '<option value="' . esc_attr($key) . '" ' . selected($value, $key, false) . '>' . esc_html($label) . '</option>';
        }
        echo '</select>';
        echo '<p class="description">' . esc_html__('Select the language for the widget interface.', 'bettercx-widget') . '</p>';
    }

    public function show_credits_field_callback() {
        $value = isset($this->settings['show_credits']) ? $this->settings['show_credits'] : false;
        echo '<input type="checkbox" name="bettercx_widget_settings[show_credits]" value="1" ' . checked($value, true, false) . ' />';
        echo '<p class="description">' . esc_html__('Show "Powered by BetterCX" credits (optional).', 'bettercx-widget') . '</p>';
    }

    public function debug_field_callback() {
        $value = isset($this->settings['debug']) ? $this->settings['debug'] : false;
        echo '<input type="checkbox" name="bettercx_widget_settings[debug]" value="1" ' . checked($value, true, false) . ' />';
        echo '<p class="description">' . esc_html__('Enable debug mode for troubleshooting.', 'bettercx-widget') . '</p>';
    }

    public function auto_init_field_callback() {
        $value = isset($this->settings['auto_init']) ? $this->settings['auto_init'] : true;
        echo '<input type="checkbox" name="bettercx_widget_settings[auto_init]" value="1" ' . checked($value, true, false) . ' />';
        echo '<p class="description">' . esc_html__('Automatically initialize the widget when the page loads.', 'bettercx-widget') . '</p>';
    }

    public function embedded_field_callback() {
        $value = isset($this->settings['embedded']) ? $this->settings['embedded'] : false;
        echo '<input type="checkbox" name="bettercx_widget_settings[embedded]" value="1" ' . checked($value, true, false) . ' id="bettercx_embedded" />';
        echo '<p class="description">' . esc_html__('Enable embedded mode for iframe or embedded contexts.', 'bettercx-widget') . '</p>';
    }

    public function embedded_size_field_callback() {
        $value = isset($this->settings['embedded_size']) ? $this->settings['embedded_size'] : 'full';
        $sizes = array(
            'full' => esc_html__('Full Screen', 'bettercx-widget'),
            'medium' => esc_html__('Medium (60% width, 4:3 ratio)', 'bettercx-widget'),
            'small' => esc_html__('Small (40% width, 4:3 ratio)', 'bettercx-widget'),
        );

        echo '<select name="bettercx_widget_settings[embedded_size]" id="bettercx_embedded_size">';
        foreach ($sizes as $key => $label) {
            echo '<option value="' . esc_attr($key) . '" ' . selected($value, $key, false) . '>' . esc_html($label) . '</option>';
        }
        echo '</select>';
        echo '<p class="description">' . esc_html__('Select the size of the embedded widget. Note: On mobile devices, the widget always uses full screen regardless of this setting.', 'bettercx-widget') . '</p>';
    }

    public function embedded_placement_field_callback() {
        $value = isset($this->settings['embedded_placement']) ? $this->settings['embedded_placement'] : 'center';
        $placements = array(
            'top' => esc_html__('Top', 'bettercx-widget'),
            'center' => esc_html__('Center', 'bettercx-widget'),
            'bottom' => esc_html__('Bottom', 'bettercx-widget'),
        );

        echo '<select name="bettercx_widget_settings[embedded_placement]" id="bettercx_embedded_placement">';
        foreach ($placements as $key => $label) {
            echo '<option value="' . esc_attr($key) . '" ' . selected($value, $key, false) . '>' . esc_html($label) . '</option>';
        }
        echo '</select>';
        echo '<p class="description">' . esc_html__('Select the vertical placement of the embedded widget. This setting only applies when Embedded Size is not "Full Screen".', 'bettercx-widget') . '</p>';
    }

    public function enable_analytics_field_callback() {
        $value = isset($this->settings['enable_analytics']) ? $this->settings['enable_analytics'] : false;
        echo '<input type="checkbox" name="bettercx_widget_settings[enable_analytics]" value="1" ' . checked($value, true, false) . ' />';
        echo '<p class="description">' . esc_html__('Enable analytics tracking for widget events (requires Google Analytics).', 'bettercx-widget') . '</p>';
    }

    public function privacy_consent_field_callback() {
        $value = isset($this->settings['privacy_consent']) ? $this->settings['privacy_consent'] : false;
        echo '<input type="checkbox" name="bettercx_widget_settings[privacy_consent]" value="1" ' . checked($value, true, false) . ' />';
        echo '<p class="description">' . esc_html__('Require explicit user consent before enabling analytics and data collection.', 'bettercx-widget') . '</p>';
    }

    public function logged_in_only_field_callback() {
        $value = isset($this->settings['logged_in_only']) ? (bool) $this->settings['logged_in_only'] : false;
        ?>
        <label>
            <input type="checkbox" name="bettercx_widget_settings[logged_in_only]" value="1" <?php checked($value, true); ?> />
            <?php esc_html_e('Show widget only to logged-in users', 'bettercx-widget'); ?>
        </label>
        <p class="description">
            <?php esc_html_e('When enabled, the widget will only be displayed to users who are logged into WordPress. This is useful for member-only sites or applications with authentication.', 'bettercx-widget'); ?>
        </p>
        <?php
    }

    public function is_attachments_disabled_field_callback() {
        $value = isset($this->settings['is_attachments_disabled']) ? (bool) $this->settings['is_attachments_disabled'] : false;
        ?>
        <label>
            <input type="checkbox" name="bettercx_widget_settings[is_attachments_disabled]" value="1" <?php checked($value, true); ?> />
            <?php esc_html_e('Hide attachment uploads in the chat composer', 'bettercx-widget'); ?>
        </label>
        <p class="description">
            <?php esc_html_e('When enabled, users will not see the image upload button in the widget composer.', 'bettercx-widget'); ?>
        </p>
        <?php
    }

    public function custom_css_field_callback() {
        $value = isset($this->settings['custom_css']) ? $this->settings['custom_css'] : '';
        echo '<textarea name="bettercx_widget_settings[custom_css]" rows="10" cols="50" class="large-text code">' . esc_textarea($value) . '</textarea>';
        echo '<p class="description">' . esc_html__('Add custom CSS to style the widget.', 'bettercx-widget') . '</p>';
    }

    public function base_url_field_callback() {
        $value = isset($this->settings['base_url']) ? $this->settings['base_url'] : 'https://api.bettercx.ai';
        echo '<input type="url" name="bettercx_widget_settings[base_url]" value="' . esc_attr($value) . '" class="regular-text" />';
        echo '<p class="description">' . esc_html__('Base URL for the BetterCX API (for testing purposes).', 'bettercx-widget') . '</p>';
    }

    public function ai_service_url_field_callback() {
        $value = isset($this->settings['ai_service_url']) ? $this->settings['ai_service_url'] : 'https://widget.bettercx.ai/prod';
        echo '<input type="url" name="bettercx_widget_settings[ai_service_url]" value="' . esc_attr($value) . '" class="regular-text" />';
        echo '<p class="description">' . esc_html__('AI Service URL for the BetterCX AI service (for testing purposes).', 'bettercx-widget') . '</p>';
    }

    /**
     * Admin page
     *
     * @since 1.0.0
     */
    public function admin_page() {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

            <div class="bettercx-widget-admin-header">
                <div class="bettercx-widget-admin-info">
                    <h2><?php esc_html_e('BetterCX Widget', 'bettercx-widget'); ?></h2>
                    <p><?php esc_html_e('Professional AI-powered chat widget for your WordPress site.', 'bettercx-widget'); ?></p>
                </div>
                <div class="bettercx-widget-admin-actions">
                    <button type="button" id="test-connection" class="button button-secondary">
                        <?php esc_html_e('Test Connection', 'bettercx-widget'); ?>
                    </button>
                </div>
            </div>

            <form method="post" action="options.php" id="bettercx-widget-settings-form">
                <?php
                settings_fields('bettercx_widget_settings');
                do_settings_sections('bettercx-widget');
                submit_button();
                ?>
            </form>

            <div class="bettercx-widget-admin-footer">
                <h3><?php esc_html_e('Usage', 'bettercx-widget'); ?></h3>
                <p><?php esc_html_e('Use the shortcode <code>[bettercx_widget]</code> in your posts, pages, or widgets.', 'bettercx-widget'); ?></p>
                <p><?php esc_html_e('You can override settings per shortcode:', 'bettercx-widget'); ?></p>
                <code><?php echo esc_html('[bettercx_widget public_key="pk_live_xxx" theme="dark" debug="true"]'); ?></code>

                <h3><?php esc_html_e('Event Handling', 'bettercx-widget'); ?></h3>
                <p><?php esc_html_e('To handle widget events in your theme or plugin JavaScript:', 'bettercx-widget'); ?></p>
                <pre><code>// jQuery
jQuery(document).on('bettercx:widgetEvent', function(event, data) {
    console.log('Widget Event:', data);
});

// Vanilla JavaScript
window.BetterCXWidgetEvent = function(event) {
  console.log('Widget Event:', event);
};</code></pre>

                <h3><?php esc_html_e('Support', 'bettercx-widget'); ?></h3>
                <p>
                    <?php esc_html_e('Need help?', 'bettercx-widget'); ?>
                    <a href="https://bettercx.ai/support" target="_blank"><?php esc_html_e('Visit our support center', 'bettercx-widget'); ?></a>
                </p>
            </div>
        </div>
        <?php
    }
}

/**
 * BetterCX Widget Widget Class
 *
 * @since 1.0.0
 */
class BetterCX_Widget_Widget extends WP_Widget {

    /**
     * Constructor
     *
     * @since 1.0.0
     */
    public function __construct() {
        parent::__construct(
            'bettercx_widget_widget',
            esc_html__('BetterCX Widget', 'bettercx-widget'),
            array('description' => esc_html__('Display the BetterCX chat widget in a widget area.', 'bettercx-widget'))
        );
    }

    /**
     * Widget output
     *
     * @since 1.0.0
     * @param array $args
     * @param array $instance
     */
    public function widget($args, $instance) {
        $instance = wp_parse_args($instance, array(
            'title' => '',
            'public_key' => '',
            'theme' => 'auto',
            'position' => 'right',
        ));

        echo wp_kses_post($args['before_widget']);

        if (!empty($instance['title'])) {
            echo wp_kses_post($args['before_title']) . esc_html(apply_filters('widget_title', $instance['title'])) . wp_kses_post($args['after_title']);
        }

        $shortcode_atts = array();
        if (!empty($instance['public_key'])) {
            $shortcode_atts['public_key'] = $instance['public_key'];
        }
        if (!empty($instance['theme'])) {
            $shortcode_atts['theme'] = $instance['theme'];
        }
        if (!empty($instance['position'])) {
            $shortcode_atts['position'] = $instance['position'];
        }

        echo do_shortcode('[bettercx_widget ' . esc_attr(http_build_query($shortcode_atts, '', ' ')) . ']');

        echo wp_kses_post($args['after_widget']);
    }

    /**
     * Widget form
     *
     * @since 1.0.0
     * @param array $instance
     */
    public function form($instance) {
        $instance = wp_parse_args($instance, array(
            'title' => '',
            'public_key' => '',
            'theme' => 'auto',
            'position' => 'right',
        ));

        ?>
        <p>
            <label for="<?php echo esc_attr($this->get_field_id('title')); ?>"><?php esc_html_e('Title:', 'bettercx-widget'); ?></label>
            <input class="widefat" id="<?php echo esc_attr($this->get_field_id('title')); ?>" name="<?php echo esc_attr($this->get_field_name('title')); ?>" type="text" value="<?php echo esc_attr($instance['title']); ?>" />
        </p>
        <p>
            <label for="<?php echo esc_attr($this->get_field_id('public_key')); ?>"><?php esc_html_e('Public Key:', 'bettercx-widget'); ?></label>
            <input class="widefat" id="<?php echo esc_attr($this->get_field_id('public_key')); ?>" name="<?php echo esc_attr($this->get_field_name('public_key')); ?>" type="text" value="<?php echo esc_attr($instance['public_key']); ?>" />
        </p>
        <p>
            <label for="<?php echo esc_attr($this->get_field_id('theme')); ?>"><?php esc_html_e('Theme:', 'bettercx-widget'); ?></label>
            <select class="widefat" id="<?php echo esc_attr($this->get_field_id('theme')); ?>" name="<?php echo esc_attr($this->get_field_name('theme')); ?>">
                <option value="auto" <?php selected($instance['theme'], 'auto'); ?>><?php esc_html_e('Auto', 'bettercx-widget'); ?></option>
                <option value="light" <?php selected($instance['theme'], 'light'); ?>><?php esc_html_e('Light', 'bettercx-widget'); ?></option>
                <option value="dark" <?php selected($instance['theme'], 'dark'); ?>><?php esc_html_e('Dark', 'bettercx-widget'); ?></option>
            </select>
        </p>
        <p>
            <label for="<?php echo esc_attr($this->get_field_id('position')); ?>"><?php esc_html_e('Position:', 'bettercx-widget'); ?></label>
            <select class="widefat" id="<?php echo esc_attr($this->get_field_id('position')); ?>" name="<?php echo esc_attr($this->get_field_name('position')); ?>">
                <option value="right" <?php selected($instance['position'], 'right'); ?>><?php esc_html_e('Right', 'bettercx-widget'); ?></option>
                <option value="left" <?php selected($instance['position'], 'left'); ?>><?php esc_html_e('Left', 'bettercx-widget'); ?></option>
            </select>
        </p>
        <?php
    }

    /**
     * Update widget
     *
     * @since 1.0.0
     * @param array $new_instance
     * @param array $old_instance
     * @return array
     */
    public function update($new_instance, $old_instance) {
        $instance = array();
        $instance['title'] = sanitize_text_field($new_instance['title']);
        $instance['public_key'] = sanitize_text_field($new_instance['public_key']);
        $instance['theme'] = sanitize_text_field($new_instance['theme']);
        $instance['position'] = sanitize_text_field($new_instance['position']);

        return $instance;
    }
}

// Initialize the plugin
BetterCX_Widget::get_instance();
