<?php
/**
 * Plugin Name: Behamics Organic Automation
 * Description: Adds secure REST API endpoints to create, update, and retrieve WordPress page content with token-based authentication. Includes Elementor, ACF, and category support.
 * Version: 1.0.0
 * Author: Behamics
 * Author URI: https://behamics.com/
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: behamics-organic-automation
 */

// Exit if accessed directly
if (!defined('ABSPATH'))
    exit;

// === Generate Token on Activation ===
register_activation_hook(__FILE__, 'behamics_generate_token');
function behamics_generate_token()
{
    if (!get_option('behamics_organic_api_token')) {
        $token = bin2hex(random_bytes(32));
        add_option('behamics_organic_api_token', $token);
    }
}

// === REST API Routes ===
// All endpoints use token-based authentication via behamics_api_permission_callback
add_action('rest_api_init', function () {
    register_rest_route('behamics-organic/v1', '/update-page', [
        'methods' => 'POST',
        'callback' => 'behamics_update_page',
        'permission_callback' => 'behamics_api_permission_callback',
    ]);

    register_rest_route('behamics-organic/v1', '/page/(?P<id>\d+)', [
        'methods' => 'GET',
        'callback' => 'behamics_get_page',
        'permission_callback' => 'behamics_api_permission_callback',
    ]);

    register_rest_route('behamics-organic/v1', '/pages', [
        'methods' => 'GET',
        'callback' => 'behamics_get_all_pages',
        'permission_callback' => 'behamics_api_permission_callback',
    ]);

    register_rest_route('behamics-organic/v1', '/create-or-update-page-by-url', [
        'methods' => 'POST',
        'callback' => 'behamics_create_or_update_page_by_url',
        'permission_callback' => 'behamics_api_permission_callback',
    ]);
});

// === Token Verification ===
function behamics_verify_token($request)
{
    $token = sanitize_text_field($request->get_param('token') ?: ($request->get_json_params()['token'] ?? ''));
    $expected = get_option('behamics_organic_api_token');
    return hash_equals($expected, $token);
}

// === Permission Callback for Token-based Authentication ===
function behamics_api_permission_callback($request)
{
    return behamics_verify_token($request);
}

// === API Callbacks ===
function behamics_update_page($request)
{
    $params = $request->get_json_params();
    $page_id = intval($params['page_id'] ?? 0);
    $title = sanitize_text_field($params['title'] ?? '');
    $content = wp_kses_post($params['content'] ?? '');

    if (!$page_id || (!$title && !$content)) {
        return new WP_REST_Response(['error' => 'Missing page_id or content/title'], 400);
    }

    $update = ['ID' => $page_id];
    if ($title)
        $update['post_title'] = $title;
    if ($content)
        $update['post_content'] = $content;

    $result = wp_update_post($update, true);

    if (is_wp_error($result)) {
        return new WP_REST_Response(['error' => $result->get_error_message()], 500);
    }

    return new WP_REST_Response(['success' => true, 'page_id' => $page_id], 200);
}

function behamics_get_page($request)
{
    $page_id = intval($request['id']);
    $post = get_post($page_id);

    if (!$post || $post->post_type !== 'page') {
        return new WP_REST_Response(['error' => 'Page not found'], 404);
    }

    return new WP_REST_Response([
        'page_id' => $page_id,
        'title' => $post->post_title,
        'content' => apply_filters('the_content', $post->post_content), // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
    ], 200);
}

function behamics_get_all_pages($request)
{
    $pages = get_posts([
        'post_type' => 'page',
        'post_status' => 'any',
        'numberposts' => -1,
    ]);

    $output = array_map(function ($p) {
        $post = get_object_vars($p);
        $post['url'] = get_permalink($p->ID);
        return $post;
    }, $pages);

    return new WP_REST_Response($output, 200);
}


function behamics_create_or_update_page_by_url($request)
{
    $params = $request->get_json_params();
    $html = wp_kses_post($params['html'] ?? '');
    $url_path = trim($params['url_path'] ?? '', '/');
    $title = sanitize_text_field($params['title'] ?? 'Auto Published Page');
    $post_type = sanitize_text_field($params['post_type'] ?? 'page');
    $elementor_data = $params['_elementor_data'] ?? null;
    $elementor_edit_mode = $params['_elementor_edit_mode'] ?? null;
    $sections = $params['sections'] ?? null;

    if (!$url_path || (!$html && !$elementor_data && !$sections)) {
        return new WP_REST_Response(['error' => 'Missing URL path or content'], 400);
    }

    $existing_page = get_page_by_path($url_path, OBJECT, $post_type);
    if ($existing_page) {
        if ($post_type == "page") {
            return new WP_REST_Response([
                'error' => 'A page with this URL path already exists. Please use a different URL path.'
            ], 409);
        }
        // === Update existing post ===
        $post_id = $existing_page->ID;
        $update_data = ['ID' => $post_id];
        if ($title)
            $update_data['post_title'] = $title;
        if ($html)
            $update_data['post_content'] = $html;
        wp_update_post($update_data);

        // Handle ACF Sections
        if (!empty($sections) && is_array($sections)) {
            foreach ($sections as $field_key => $field_value) {
                update_field($field_key, $field_value, $post_id);
            }
            wp_set_post_categories($post_id, [31, 24]);
        }

        // Handle Elementor
        if ($elementor_data) {
            update_post_meta($post_id, '_elementor_edit_mode', $elementor_edit_mode ?: 'builder');
            update_post_meta($post_id, '_elementor_template_type', 'page');
            update_post_meta($post_id, '_elementor_data', $elementor_data);
        }

        // Return success response
        return new WP_REST_Response([
            'success' => true,
            'action' => 'updated',
            'page_id' => $post_id,
            'url' => get_permalink($post_id),
        ], 200);
    }

    // Provide placeholder content if only sections exist
    if (!$html && $sections) {
        $html = '<!-- wp:paragraph --><p> </p><!-- /wp:paragraph -->';
    }

    $post_data = [
        'post_title' => $title,
        'post_name' => $url_path,
        'post_content' => $html,
        'post_status' => 'publish',
        'post_type' => $post_type,
    ];

    $post_id = wp_insert_post($post_data, true);

    if (is_wp_error($post_id)) {
        return new WP_REST_Response(['error' => $post_id->get_error_message()], 500);
    }

    // Handle ACF sections
    if (!empty($sections) && is_array($sections)) {
        foreach ($sections as $field_key => $field_value) {
            update_field($field_key, $field_value, $post_id);
        }
        //  wp_set_post_categories($post_id, [31]);
        wp_set_post_categories($post_id, [31, 24]);

        //  wp_set_post_categories($post_id, [24]);
    }



    // Save Elementor data if provided
    if ($elementor_data) {
        update_post_meta($post_id, '_elementor_edit_mode', $elementor_edit_mode ?: 'builder');
        update_post_meta($post_id, '_elementor_template_type', 'page');
        update_post_meta($post_id, '_elementor_data', $elementor_data);
    }

    return new WP_REST_Response([
        'success' => true,
        'action' => 'created',
        'page_id' => $post_id,
        'url' => get_permalink($post_id),
    ], 200);
}


// add_action('admin_menu', function () {
//     add_menu_page('Custom API Token', 'API Token', 'manage_options', 'behamics-organic-token', function () {
//         $token = get_option('behamics_organic_api_token');
//         echo '<div class="wrap"><h1>Behamics Organic Automation</h1>';
//         echo '<p>Copy the token below and paste it into your dashboard:</p>';
//         echo '<textarea readonly style="width:100%;font-family:monospace;" rows="2">' . esc_textarea($token) . '</textarea>';
//         echo '</div>';
//     });
// });
add_action('admin_menu', function () {
    add_menu_page(
        'Behamics API Token',
        'Behamics Automation',
        'manage_options',
        'behamics-organic-token',
        'behamics_render_token_page',
        plugin_dir_url(__FILE__) . 'assets/behamics-icon.svg', // <-- your logo

        80
    );
});
function behamics_render_token_page()
{
    // Verify nonce for security
    if (isset($_POST['behamics_nonce']) && !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['behamics_nonce'])), 'behamics_action')) {
        wp_die('Security check failed');
    }

    $token = get_option('behamics_organic_api_token');
    ?>
    <div class="wrap">
        <img src="<?php echo esc_url(plugin_dir_url(__FILE__)); ?>assets/behamics-icon.svg" alt="Behamics Logo">
        <div class="behamics-header">

            <h1>Organic Automation</h1>
        </div>
        <p><strong>Copy the token below and paste it into your dashboard:</strong></p>
        <textarea id="behamics-token" readonly rows="2"><?php echo esc_textarea($token); ?></textarea>

        <div class="button-row">
            <button class="copy-button" id="copy-token-btn">Copy Token</button>
            <a class="dashboard-button" href="https://dashboard.behamics.com/seo/integrations" target="_blank">Go to Organic
                Dashboard</a>
        </div>
    </div>
    <?php
}
register_activation_hook(__FILE__, function () {
    if (!get_option('behamics_organic_api_token')) {
        $token = bin2hex(random_bytes(32));
        add_option('behamics_organic_api_token', $token);
    }
    add_option('behamics_activation_redirect', true);
});

add_action('admin_init', function () {
    if (get_option('behamics_activation_redirect', false)) {
        delete_option('behamics_activation_redirect');
        if (!isset($_GET['activate-multi'])) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            wp_safe_redirect(admin_url('admin.php?page=behamics-organic-token'));
            exit;
        }
    }
});
add_action('admin_enqueue_scripts', function ($hook) {
    if ($hook === 'toplevel_page_behamics-organic-token') {
        wp_enqueue_style(
            'behamics-admin-style',
            plugin_dir_url(__FILE__) . 'assets/style.css',
            [],
            filemtime(plugin_dir_path(__FILE__) . 'assets/style.css') // version based on file update
        );

        wp_enqueue_script(
            'behamics-admin-script',
            plugin_dir_url(__FILE__) . 'assets/behamics-admin.js',
            ['jquery'],
            filemtime(plugin_dir_path(__FILE__) . 'assets/behamics-admin.js'),
            true
        );

        // Add inline styles for admin menu icon
        wp_add_inline_style('behamics-admin-style', '
            #adminmenu .toplevel_page_behamics-organic-token img {
                width: 18px !important;
                height: 18px !important;
                object-fit: contain;
            }
        ');
    }
});
add_action('rest_api_init', function () {
    register_rest_route('behamics-organic/v1', '/categories', [
        'methods' => 'GET',
        'callback' => 'behamics_get_all_categories',
        'permission_callback' => 'behamics_api_permission_callback',
    ]);
});


function behamics_get_all_categories()
{
    $categories = get_categories(['hide_empty' => false]);

    $result = array_map(function ($cat) {
        return [
            'id' => $cat->term_id,
            'name' => $cat->name,
            'slug' => $cat->slug,
        ];
    }, $categories);

    return new WP_REST_Response(['categories' => $result], 200);
}
