<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://beepcoder.com
 * @since      1.0.0
 *
 * @package    Beep_Notifier
 * @subpackage Beep_Notifier/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    Beep_Notifier
 * @subpackage Beep_Notifier/public
 * @author     Beepcoder <hello@beepcoder.com>
 */
class Beep_Notifier_Public {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of the plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;

	}

	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

	    /**
	     * Enqueue styles for the public-facing part of the plugin.
	     */
	    wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/beep-notifier-public.css', array(), $this->version, 'all' );

	    // Retrieve settings
	    $beep_settings = get_option( 'beep_notifier_plugin_settings', [] );
	    $blsn_advance_design = isset( $beep_settings['blsn_advance_design'] ) ? absint( $beep_settings['blsn_advance_design'] ) : 0;

	    // Only load additional CSS if advanced design is enabled
	    if ( $blsn_advance_design === 1 ) {
	        
	        $custom_css_file = plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-beep-notifier-settings-css.php';

	        // Ensure the file exists before including it
	        if ( file_exists( $custom_css_file ) ) {
	            require_once $custom_css_file;
	        }

	        // Ensure the CSS variable is set before using it
	        if ( isset( $bc_custom_css ) && ! empty( $bc_custom_css ) ) {
	            wp_add_inline_style( $this->plugin_name, $bc_custom_css );
	        }
	    }
	}

	/**
	 * Register the JavaScript for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

	    /**
	     * Enqueue scripts for the public-facing part of the plugin.
	     */
	    
	    // Ensure jQuery is loaded (some environments disable it)
	    wp_enqueue_script( 'jquery' );

	    // Enqueue the main script
	    wp_enqueue_script(
	        $this->plugin_name,
	        plugin_dir_url( __FILE__ ) . 'js/beep-notifier-public.js',
	        array( 'jquery' ), // Dependencies
	        $this->version,
	        true // Load in the footer for performance
	    );

	    // Retrieve plugin settings
	    $beep_settings = get_option('beep_notifier_plugin_settings', []);

	    $blsn_theme               = isset($beep_settings['blsn_theme']) ? absint($beep_settings['blsn_theme']) : 1;
	    $blsn_mobile_visible      = isset($beep_settings['blsn_mobile_visible']) ? absint($beep_settings['blsn_mobile_visible']) : 0;
	    $blsn_show_animation      = isset($beep_settings['blsn_show_animation']) ? sanitize_text_field($beep_settings['blsn_show_animation']) : 'fade_in';
	    $blsn_hide_animation      = isset($beep_settings['blsn_hide_animation']) ? sanitize_text_field($beep_settings['blsn_hide_animation']) : 'fade_out';
	    $blsn_delay_first         = isset($beep_settings['blsn_delay_first']) ? intval($beep_settings['blsn_delay_first']) : 3;
	    $blsn_display_time        = isset($beep_settings['blsn_display_time']) ? intval($beep_settings['blsn_display_time']) : 5;
	    $blsn_delay_between_time  = isset($beep_settings['blsn_delay_between_time']) ? intval($beep_settings['blsn_delay_between_time']) : 2;
	    $blsn_loop                = isset($beep_settings['blsn_loop']) ? absint($beep_settings['blsn_loop']) : 0;
	    $blsn_open_link_new_tab   = isset($beep_settings['blsn_open_link_new_tab']) ? absint($beep_settings['blsn_open_link_new_tab']) : 0;

	    // Define a mobile visibility class
	    $mobile_class = ($blsn_mobile_visible === 0) ? 'hide-on-mobile' : '';

	    // Pass data to JavaScript
	    wp_localize_script($this->plugin_name, 'beep_ajax', array(
	        'ajax_url'       => admin_url('admin-ajax.php'),
	        'theme'          => $blsn_theme,
	        'mobile_class'   => $mobile_class,
	        'show_animation' => $blsn_show_animation,
	        'hide_animation' => $blsn_hide_animation,
	        'first_delay'    => $blsn_delay_first,
	        'display_time'   => $blsn_display_time,
	        'between_delay'  => $blsn_delay_between_time,
	        'loop'           => $blsn_loop, // 0 = No Loop, 1 = Loop
	        'open_new_tab'   => $blsn_open_link_new_tab, // 0 = Same Tab, 1 = New Tab
	    ));
	}


}
