<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://beepcoder.com
 * @since      1.0.0
 *
 * @package    Beep_Notifier
 * @subpackage Beep_Notifier/public
 */

defined( 'ABSPATH' ) || exit();

class Beep_Notifier_Builder {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param      string    $plugin_name       The name of the plugin.
     * @param      string    $version          The version of this plugin.
     */
    public function __construct( $plugin_name, $version ) {
        $this->plugin_name = sanitize_text_field( $plugin_name );
        $this->version     = sanitize_text_field( $version );
    }

    /**
     * Fetch recent sales from WooCommerce
     *
     * @since 1.0.0
     */
    public function beep_get_recent_sales() {
        if ( ! class_exists( 'WooCommerce' ) ) {
            wp_send_json_error( esc_html__( 'WooCommerce is not active.', 'beep-notifier' ) );
        }

        // Retrieve plugin settings with sanitization
        $beep_settings       = get_option( 'beep_notifier_plugin_settings', [] );
        $blsn_template       = isset( $beep_settings['blsn_template'] ) ? sanitize_text_field( $beep_settings['blsn_template'] ) : 'first_name';
        $blsn_random_order   = isset( $beep_settings['blsn_random_order'] ) ? (bool) $beep_settings['blsn_random_order'] : false;
        $blsn_order_status   = isset( $beep_settings['blsn_order_status'] ) ? array_map( 'sanitize_text_field', (array) $beep_settings['blsn_order_status'] ) : [ 'processing' ];
        $blsn_link_type      = isset( $beep_settings['blsn_link_type'] ) ? sanitize_text_field( $beep_settings['blsn_link_type'] ) : 'product_page';
        $blsn_show_default_image = isset( $beep_settings['blsn_show_defult_image'] ) ? (bool) $beep_settings['blsn_show_defult_image'] : false;
        $blsn_default_image  = isset( $beep_settings['blsn_default_image'] ) ? esc_url( $beep_settings['blsn_default_image'] ) : '';
        $blsn_order_limit    = isset( $beep_settings['blsn_order_limit'] ) ? absint( $beep_settings['blsn_order_limit'] ) : 5;

        // Query WooCommerce orders
        $orders = wc_get_orders([
            'limit'   => $blsn_order_limit,
            'orderby' => 'date',
            'order'   => 'DESC',
            'status'  => $blsn_order_status,
        ]);

        $sales = [];

        foreach ( $orders as $order ) {
            $order_time = strtotime( $order->get_date_created() );
            $time_ago   = human_time_diff( $order_time, current_time( 'timestamp' ) ) . ' ago';

            // Get customer name based on template
            $name = $this->get_customer_name( $order, $blsn_template );

            foreach ( $order->get_items() as $item ) {
                $product_id   = $item->get_product_id();
                $product_link = ( $blsn_link_type === 'none' ) ? '#' : get_permalink( $product_id );
                $product_image = $this->get_product_image( $product_id, $blsn_show_default_image, $blsn_default_image );

                // Prepare sale data
                $sales[] = $this->prepare_sale_data( $name, $order->get_billing_city(), $item->get_name(), $product_image, $product_link, $time_ago );
            }
        }

        // If random order is enabled, shuffle sales
        if ( $blsn_random_order ) {
            shuffle( $sales );
        }

        wp_send_json_success( $sales );
    }

    /**
     * Get customer name based on selected template
     *
     * @param WC_Order $order
     * @param string   $template
     * @return string
     */
    private function get_customer_name( $order, $template ) {
        switch ( $template ) {
            case 'first_name':
                return sanitize_text_field( $order->get_billing_first_name() );
            case 'last_name':
                return sanitize_text_field( $order->get_billing_last_name() );
            case 'full_name':
                return trim( sanitize_text_field( $order->get_billing_first_name() . ' ' . $order->get_billing_last_name() ) );
            default:
                return sanitize_text_field( $order->get_billing_first_name() ); // Default to first name
        }
    }

    /**
     * Get product image or default placeholder
     *
     * @param int    $product_id
     * @param bool   $use_default
     * @param string $default_image
     * @return string
     */
    private function get_product_image( $product_id, $use_default, $default_image ) {
        if ( $use_default && ! empty( $default_image ) ) {
            return esc_url( $default_image );
        }

        $image = wp_get_attachment_image_src( get_post_thumbnail_id( $product_id ), 'thumbnail' );
        return ! empty( $image ) ? esc_url( $image[0] ) : esc_url( wc_placeholder_img_src() );
    }

    /**
     * Prepare sale data array
     *
     * @param string $name
     * @param string $location
     * @param string $product
     * @param string $image
     * @param string $link
     * @param string $time
     * @return array
     */
    private function prepare_sale_data( $name, $location, $product, $image, $link, $time ) {
        return [
            'name'     => esc_html( $name ),
            'location' => esc_html( $location ),
            'product'  => esc_html( $product ),
            'image'    => esc_url( $image ),
            'link'     => esc_url( $link ),
            'time'     => esc_html( $time ),
        ];
    }
}
?>
