<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://beepcoder.com
 * @since      1.0.0
 *
 * @package    Beep_Notifier
 * @subpackage Beep_Notifier/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Beep_Notifier
 * @subpackage Beep_Notifier/includes
 * @author     Beepcoder <hello@beepcoder.com>
 */
class Beep_Notifier {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      Beep_Notifier_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		if ( defined( 'BEEP_NOTIFIER_VERSION' ) ) {
			$this->version = BEEP_NOTIFIER_VERSION;
		} else {
			$this->version = '1.0.0';
		}
		$this->plugin_name = 'beep-notifier';

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_public_hooks();

	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Beep_Notifier_Loader. Orchestrates the hooks of the plugin.
	 * - Beep_Notifier_i18n. Defines internationalization functionality.
	 * - Beep_Notifier_Admin. Defines all hooks for the admin area.
	 * - Beep_Notifier_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {

	    /**
	     * The class responsible for orchestrating the actions and filters of the core plugin.
	     */
	    if (!class_exists('Beep_Notifier_Loader')) {
	        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-beep-notifier-loader.php';
	    }

	    /**
	     * The class responsible for defining internationalization functionality.
	     */
	    if (!class_exists('Beep_Notifier_I18n')) {
	        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-beep-notifier-i18n.php';
	    }

	    /**
	     * The class responsible for defining all actions that occur in the admin area.
	     */
	    if (!class_exists('Beep_Notifier_Admin')) {
	        require_once plugin_dir_path(dirname(__FILE__)) . 'admin/class-beep-notifier-admin.php';
	    }
	    
	    if (!class_exists('Beep_Notifier_Admin_Action')) {
	        require_once plugin_dir_path(dirname(__FILE__)) . 'admin/class-beep-notifier-admin-action.php';
	    }
	    
	    if (!class_exists('Beep_Notifier_Admin_Menu')) {
	        require_once plugin_dir_path(dirname(__FILE__)) . 'admin/class-beep-notifier-admin-menu.php';
	    }

	    /**
	     * The class responsible for defining all views that occur in the admin area.
	     */
	    if (!class_exists('Bc_Admin_Main_Display')) {
	        require_once plugin_dir_path(dirname(__FILE__)) . 'admin/partials/beep-notifier-admin-display.php';
	    }

	    /**
	     * The class responsible for defining all actions that occur in the public-facing side of the site.
	     */
	    if (!class_exists('Beep_Notifier_Public')) {
	        require_once plugin_dir_path(dirname(__FILE__)) . 'public/class-beep-notifier-public.php';
	    }
	    
	    if (!class_exists('Beep_Notifier_Builder')) {
	        require_once plugin_dir_path(dirname(__FILE__)) . 'public/class-beep-notifier-builder.php';
	    }
	    
	    if (!class_exists('Bc_Public_Main_Display')) {
	        require_once plugin_dir_path(dirname(__FILE__)) . 'public/partials/beep-notifier-public-display.php';
	    }

	    // Ensure Beep_Notifier_Loader is loaded before initializing
	    if (class_exists('Beep_Notifier_Loader')) {
	        $this->loader = new Beep_Notifier_Loader();
	    }
	}


	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the Beep_Notifier_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function set_locale() {

		$plugin_i18n = new Beep_Notifier_i18n();

		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );

	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {

		$plugin_admin = new Beep_Notifier_Admin( $this->get_plugin_name(), $this->get_version() );
		
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );
		$this->loader->add_action( 'admin_head', $plugin_admin, 'bc_remove_wpcontent_padding' );
		$this->loader->add_filter( 'plugin_action_links_', $plugin_admin, 'beep_notifier_add_action_links' );


	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_public_hooks() {

		$plugin_public         = new Beep_Notifier_Public( $this->get_plugin_name(), $this->get_version() );
		$plugin_public_display = new Bc_Public_Main_Display( $this->get_plugin_name(), $this->get_version() );
		$beep_notifier_builder = new Beep_Notifier_Builder( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );
		$this->loader->add_action( 'wp_footer', $plugin_public_display, 'output' );
		$this->loader->add_action( 'wp_ajax_beep_get_recent_sales', $beep_notifier_builder, 'beep_get_recent_sales' );
		$this->loader->add_action( 'wp_ajax_nopriv_beep_get_recent_sales', $beep_notifier_builder, 'beep_get_recent_sales' );


	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    Beep_Notifier_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

}
