<?php

defined('ABSPATH') || exit; // Exit if accessed directly

class Bayarcash_EDD_Settings {
	private static $instance = null;
	public const GATEWAY_ID = 'bayarcash';

	public static function get_instance(): self {
		if (null === self::$instance) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		$this->init_hooks();
	}

	private function init_hooks(): void {
		add_filter('edd_settings_sections_gateways', [$this, 'register_gateway_section'], 1);
		add_filter('edd_settings_gateways', [$this, 'register_gateway_settings'], 1);
		add_action('edd_' . self::GATEWAY_ID . '_cc_form', [$this, 'display_payment_options']);
		add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts']);
	}

	public function enqueue_scripts() {
        if (!is_admin() && function_exists('edd_is_checkout') && edd_is_checkout()) {
	        wp_enqueue_script(
		        'bayarcash-edd-payment-options',
		        BAYARCASH_EDD_PLUGIN_URL . 'assets/js/payment-options.js',
		        array( 'jquery' ),
		        BAYARCASH_EDD_VERSION,
		        true
	        );
        }
	}

	public function register_gateway_section(array $gateway_sections): array {
		$gateway_sections[self::GATEWAY_ID] = esc_html__('Bayarcash', 'bayarcash-edd');
		return $gateway_sections;
	}

	public function register_gateway_settings(array $gateway_settings): array {
		$bayarcash_settings = [
			'bayarcash_settings' => [
				'id'   => 'bayarcash_settings',
				'name' => '<h3>' . esc_html__('Bayarcash Settings', 'bayarcash-edd') . '</h3>',
				'type' => 'header',
			],
			'bayarcash_token' => [
				'id'   => 'bayarcash_token',
				'name' => esc_html__('Token', 'bayarcash-edd'),
				'desc' => esc_html__('Enter your Bayarcash token', 'bayarcash-edd'),
				'type' => 'textarea',
				'size' => 'regular',
			],
			'bayarcash_secret_key' => [
				'id'   => 'bayarcash_secret_key',
				'name' => esc_html__('Secret Key', 'bayarcash-edd'),
				'desc' => esc_html__('Secret key can be obtained from Bayarcash Dashboard >> API Keys', 'bayarcash-edd'),
				'type' => 'text',
				'size' => 'regular',
			],
			'bayarcash_portal_key' => [
				'id'   => 'bayarcash_portal_key',
				'name' => esc_html__('Portal Key', 'bayarcash-edd'),
				'desc' => esc_html__('Enter your Bayarcash portal key', 'bayarcash-edd'),
				'type' => 'select',
				'size' => 'regular',
			],
			'bayarcash_payment_methods' => [
				'id'   => 'bayarcash_payment_methods',
				'name' => esc_html__('Payment Methods', 'bayarcash-edd'),
				'desc' => esc_html__('Select available payment methods for your customers', 'bayarcash-edd'),
				'type' => 'multicheck',
				'options' => $this->get_payment_method_options(),
			],
			'bayarcash_debug' => [
				'id'   => 'bayarcash_debug',
				'name' => esc_html__('Debug Mode', 'bayarcash-edd'),
				'desc' => esc_html__('Enable debug logging for troubleshooting', 'bayarcash-edd'),
				'type' => 'checkbox',
			],
		];

		$gateway_settings[self::GATEWAY_ID] = apply_filters('edd_bayarcash_settings', $bayarcash_settings);
		return $gateway_settings;
	}

	private function get_payment_method_options(): array {
		return [
			'1'  => esc_html__('FPX Online Banking', 'bayarcash-edd'),
			'4'  => esc_html__('FPX Line of Credit', 'bayarcash-edd'),
			'5'  => esc_html__('DuitNow Online Banking/Wallets', 'bayarcash-edd'),
			'6'  => esc_html__('DuitNow QR', 'bayarcash-edd'),
			'8'  => esc_html__('Boost PayFlex', 'bayarcash-edd'),
			'9'  => esc_html__('QRIS Online Banking', 'bayarcash-edd'),
			'10' => esc_html__('QRIS Wallet', 'bayarcash-edd'),
			'11' => esc_html__('NETS', 'bayarcash-edd'),
			'12' => esc_html__('Credit Card', 'bayarcash-edd'),
			'13' => esc_html__('Alipay', 'bayarcash-edd'),
			'14' => esc_html__('WeChat Pay', 'bayarcash-edd'),
			'15' => esc_html__('PromptPay', 'bayarcash-edd'),
			'16' => esc_html__('Touch n Go', 'bayarcash-edd'),
			'17' => esc_html__('Boost Wallet', 'bayarcash-edd'),
			'18' => esc_html__('GrabPay', 'bayarcash-edd'),
			'19' => esc_html__('Grab PayLater', 'bayarcash-edd'),
			'21' => esc_html__('ShopeePay', 'bayarcash-edd'),
		];
	}

	public function display_payment_options(): void {
		$selected_methods = edd_get_option('bayarcash_payment_methods', []);
		$all_methods = $this->get_payment_method_options();

		if (empty($selected_methods) || !is_array($selected_methods)) {
			echo '<p>' . esc_html__('No payment methods available.', 'bayarcash-edd') . '</p>';
			return;
		}

		wp_nonce_field('bayarcash_payment_method', 'bayarcash_nonce');

		// Display custom Bayarcash error at top
		$this->display_bayarcash_error();

		echo '<div class="bayarcash-payment-box">';

		$first_option = true;
		foreach ($all_methods as $method_id => $method_name) {
			if (!empty($selected_methods[$method_id]) && $selected_methods[$method_id] !== '0') {
				// Support both old format (stores method name) and new format (stores "1")
				$this->display_payment_option($method_id, $method_name, $first_option);
				$first_option = false;
			}
		}

		echo '</div>';
	}

	private function display_payment_option(string $method_id, string $method_name, bool $is_first): void {
		$image_url = $this->get_payment_method_image($method_id);
		$description = $this->get_payment_method_description($method_id);

		?>
		<div class="bayarcash-payment-option">
			<input
				type="radio"
				id="bayarcash_method_<?php echo esc_attr($method_id); ?>"
				name="bayarcash_payment_method"
				value="<?php echo esc_attr($method_id); ?>"
				required
				<?php checked($is_first); ?>
			>
			<label for="bayarcash_method_<?php echo esc_attr($method_id); ?>" class="bayarcash-payment-option-wrapper">
				<span class="bayarcash-payment-option-label"><?php echo esc_html($method_name); ?></span>
				<img
					class="bayarcash-payment-option-image"
					src="<?php echo esc_url($image_url); ?>"
					alt="<?php echo esc_attr($method_name); ?>"
				>
			</label>
			<div class="bayarcash-payment-details">
				<p><?php echo esc_html($description); ?></p>
			</div>
		</div>
		<?php
	}

	private function display_bayarcash_error(): void {
		$errors = edd_get_errors();
		if (empty($errors) || !isset($errors['bayarcash_error'])) {
			return;
		}

		$error_message = $errors['bayarcash_error'];
		?>
		<div class="bayarcash-error-notice">
			<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
				<circle cx="12" cy="12" r="10"></circle>
				<line x1="12" y1="8" x2="12" y2="12"></line>
				<line x1="12" y1="16" x2="12.01" y2="16"></line>
			</svg>
			<span><?php echo esc_html($error_message); ?></span>
		</div>
		<?php
	}

	private function get_payment_method_description(string $method_id): string {
		$descriptions = [
			'1'  => esc_html__('Pay with online banking Maybank2u, CIMB Clicks, Bank Islam GO and more banks from Malaysia via FPX.', 'bayarcash-edd'),
			'4'  => esc_html__('Pay with FPX Line of Credit for buy now, pay later options.', 'bayarcash-edd'),
			'5'  => esc_html__('Pay with online banking or e-wallets via DuitNow.', 'bayarcash-edd'),
			'6'  => esc_html__('Scan and pay with DuitNow QR using your banking or e-wallet app.', 'bayarcash-edd'),
			'8'  => esc_html__('Pay with Boost PayFlex - buy now, pay later.', 'bayarcash-edd'),
			'9'  => esc_html__('Pay with QRIS Online Banking for Indonesian bank transfers.', 'bayarcash-edd'),
			'10' => esc_html__('Pay with QRIS Wallet for Indonesian e-wallet payments.', 'bayarcash-edd'),
			'11' => esc_html__('Pay with NETS for Singapore bank transfers.', 'bayarcash-edd'),
			'12' => esc_html__('Pay with Visa/Mastercard credit or debit card.', 'bayarcash-edd'),
			'13' => esc_html__('Pay with Alipay.', 'bayarcash-edd'),
			'14' => esc_html__('Pay with WeChat Pay.', 'bayarcash-edd'),
			'15' => esc_html__('Pay with PromptPay for Thailand payments.', 'bayarcash-edd'),
			'16' => esc_html__('Pay with Touch n Go e-wallet.', 'bayarcash-edd'),
			'17' => esc_html__('Pay with Boost e-wallet.', 'bayarcash-edd'),
			'18' => esc_html__('Pay with GrabPay e-wallet.', 'bayarcash-edd'),
			'19' => esc_html__('Pay with Grab PayLater - buy now, pay later.', 'bayarcash-edd'),
			'21' => esc_html__('Pay with ShopeePay e-wallet.', 'bayarcash-edd'),
		];

		return $descriptions[$method_id] ?? esc_html__('Select this payment method to proceed with your purchase.', 'bayarcash-edd');
	}

	private function get_payment_method_image(string $method_id): string {
		$plugin_url = BAYARCASH_EDD_PLUGIN_URL;
		$image_urls = [
			'1'  => $plugin_url . 'assets/img/fpx-online-banking.png',
			'4'  => $plugin_url . 'assets/img/fpx-line-of-credit.png',
			'5'  => $plugin_url . 'assets/img/duitnow-online-banking-wallets.png',
			'6'  => $plugin_url . 'assets/img/duitnow-qr.png',
			'8'  => $plugin_url . 'assets/img/boost-payflex.png',
			'9'  => $plugin_url . 'assets/img/qris-online-banking.png',
			'10' => $plugin_url . 'assets/img/qris-wallet.png',
			'11' => $plugin_url . 'assets/img/nets.svg',
			'12' => $plugin_url . 'assets/img/credit-card.png',
			'13' => $plugin_url . 'assets/img/alipay.png',
			'14' => $plugin_url . 'assets/img/wechatpay.png',
			'15' => $plugin_url . 'assets/img/promptpay.png',
			'16' => $plugin_url . 'assets/img/touchngo.png',
			'17' => $plugin_url . 'assets/img/boost-wallet.png',
			'18' => $plugin_url . 'assets/img/grabpay.png',
			'19' => $plugin_url . 'assets/img/grab-paylater.png',
			'21' => $plugin_url . 'assets/img/shopeepay.svg',
		];

		return $image_urls[$method_id] ?? $plugin_url . 'assets/img/bayarcash-logo.png';
	}
}

Bayarcash_EDD_Settings::get_instance();