<?php
/*
 * Plugin Name: BastionBytes integration for BookStack
 * Description: simple BookStack integration for wordpress
 * Version: 0.4.11
 * Author: Martin Dames (martin@bastionbytes.de)
 * License: GPLv3
 * Text Domain: bastionbytes-integration-for-bookstack
 * Domain Path: /languages
 */

if(!defined('ABSPATH')) {
	// direct access
	die();
}

if(!class_exists('BastionBytesIntegrationForBookStack')) {

	final class BastionBytesIntegrationForBookStack {


		// used to pass messages to the wp backend page
		private $message_error = false;
		private $message_success = false;
		private static $_instance = null;

		public static function instance() {
			if(is_null(self::$_instance)) {
				self::$_instance = new self();
			}
			return(self::$_instance);
		}

		function __construct() {

			register_activation_hook(__FILE__, [$this, 'init']);
			add_shortcode('bb-bookstack', [$this, 'shortcode']);

			add_action('admin_menu', function () {
				$hook = add_submenu_page(
					'options-general.php',
					'BastionBytes integration for BookStack',
					'BastionBytes integration for BookStack',
					'manage_options',
					'bb_integration_for_bookstack_backend_options',
					[$this, 'backend_options']
				);
				add_action('load-' . $hook, [$this, 'backend_options_submit']);
			});

			add_action('plugins_loaded', function () {
				load_plugin_textdomain('bastionbytes-integration-for-bookstack', false, plugin_basename(dirname(__FILE__)) . '/languages');
				$this->upgrade();
			});

			add_action('rest_api_init', function () {
				register_rest_route('bookstack/v1', '/site-update', [
					'methods' => 'POST',
					'callback' => [$this, 'webhook'],
					'permission_callback' => '__return_true'
				]);
			});

			add_action('wp_enqueue_scripts', function () {
				wp_register_style('bb_integration_for_bookstack', plugins_url('style.css',__FILE__ ), [], '0.3');
				wp_enqueue_style('bb_integration_for_bookstack');
			});

			add_action('init', function () {
				global $wp;
				$wp->add_query_var('bb-bookstack-id');
				$wp->add_query_var('bb-bookstack-path');
			});
		}

		// query remote bookstack (returns false on any failure)
		function raw_bookstack_query($url) {

			$bookstack_token_id = get_option('bb_integration_for_bookstack_token_id');
			$bookstack_token_secret = get_option('bb_integration_for_bookstack_token_secret');
			$bookstack_url = get_option('bb_integration_for_bookstack_url');

			$query_options = [
				'timeout' => 3,
				'headers' => 'Authorization: Token ' . $bookstack_token_id . ':' . $bookstack_token_secret
			];

			if(get_option('bb_integration_for_bookstack_ignore_cert') == 1) $query_options['sslverify'] = false;

			$result = wp_remote_get($bookstack_url . '/api/' . $url, $query_options);

			if(is_wp_error($result)) {
				return(false);
			}

			$result = json_decode(wp_remote_retrieve_body($result));

			if(is_null($result)) return(false);

			return($result);
		}

		// query all books from remote bookstack (returns false on any failure)
		function query_all_books() {

			$result = $this->raw_bookstack_query('books');
			if($result === false) return(false);

			$books = [];

			foreach($result->data as $book) {
				$books[] = [
					'title' => $book->name,
					'id' => $book->id,
					'slug' => $book->slug
				];
			}

			return($books);
		}

		function query_all_chapters() {

			$result = $this->raw_bookstack_query('chapters');
			if($result === false) return(false);

			$chapters = [];

			foreach($result->data as $chapter) {
				$chapters[] = [
					'title' => $chapter->name,
					'id' => $chapter->id,
					'slug' => $chapter->slug,
					'book' => $chapter->book_id
				];
			}

			return($chapters);
		}

		function update_db_all_books($books) {
			global $wpdb;
			$first = true;
			$args = [];
			$pos = 1;
			$query = "INSERT INTO `" . $wpdb->prefix . "bb_integration_for_bookstack_books` (`id`, `title`, `slug`) VALUES";
			foreach($books as $book) {
				$args[] = $book['id'];
				$args[] = $book['title'];
				$args[] = $book['slug'];
				if($first) {
					$first = false;
				} else {
					$query .= ",";
				}
				$query .= " (%d, %s, %s)";
			}
			$query .= " ON DUPLICATE KEY UPDATE `title`=VALUES(`title`), `slug`=VALUES(`slug`), `valid`=1, `updated`=NOW()";
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared
			$wpdb->query($wpdb->prepare($query, ...$args));
		}

		function update_db_all_chapters($chapters) {
			global $wpdb;
			$first = true;
			$args = [];
			$pos = 1;
			$query = "INSERT INTO `" . $wpdb->prefix . "bb_integration_for_bookstack_chapters` (`id`, `title`, `slug`, `pos`, `book`) VALUES";
			foreach($chapters as $chapter) {
				$args[] = $chapter['id'];
				$args[] = $chapter['title'];
				$args[] = $chapter['slug'];
				$args[] = $chapter['book'];
				if($first) {
					$first = false;
				} else {
					$query .= ",";
				}
				$query .= " (%d, %s, %s, 1, %s)";
			}
			$query .= " ON DUPLICATE KEY UPDATE `title`=VALUES(`title`), `slug`=VALUES(`slug`), `pos`=VALUES(`pos`), `valid`=0, `book`=VALUES(`book`)";
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared
			$wpdb->query($wpdb->prepare($query, ...$args));
		}

		function get_book_id_by_slug($slug) {
			global $wpdb;
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
			$result = $wpdb->get_row($wpdb->prepare("SELECT `id` FROM `" . $wpdb->prefix . "bb_integration_for_bookstack_books` WHERE `slug`=%s", $slug));
			if(!is_null($result)) {
				return($result->id);
			}
			$books = $this->query_all_books();
			$this->update_db_all_books($books);
			foreach($books as $book) {
				if($book['slug'] == $slug) {
					return($book['id']);
				}
			}
			return(false);
		}

		function get_chapter_id_by_slug($bookId, $slug) {
			global $wpdb;
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
			$result = $wpdb->get_row($wpdb->prepare("SELECT `id` FROM `" . $wpdb->prefix . "bb_integration_for_bookstack_chapters` WHERE `slug`=%s AND `book`=%s", $slug, $bookId));
			if(!is_null($result)) {
				return($result->id);
			}
			$chapters = $this->query_all_chapters();
			$this->update_db_all_chapters($chapters);
			foreach($chapters as $chapter) {
				if(($chapter['slug'] == $slug) && ($chapter['book'] == $bookId)) {
					return($chapter['id']);
				}
			}
			return(false);
		}

		// query page from remote bookstack (returns false on any failure)
		function query_page($id) {

			$result = $this->raw_bookstack_query('pages/' . $id);
			if($result === false) return(false);

			if(!property_exists($result, 'name') || !property_exists($result, 'html')) return(false);

			$r = [
				'title' => $result->name,
				'content' => $result->html,
				'last_changed' => strtotime($result->updated_at),
				'chapter' => $result->chapter_id,
				'book' => $result->book_id,
				'slug' => $result->slug
			];

			return($r);
		}

		// query book from remote bookstack (returns false on any failure)
		function query_book($id) {

			$result = $this->raw_bookstack_query('books/' . $id);
			if($result === false) return(false);

			$r = [
				'title' => $result->name,
				'slug' => $result->slug,
				'book_id' => $result->id,
				'last_changed' => strtotime($result->updated_at),
				'contents' => []
			];

			foreach($result->contents as $rawSection) {

				if($rawSection->type == "chapter") {

					$chapter = [
						'title' => $rawSection->name,
						'id' => $rawSection->id,
						'type' => 'chapter',
						'last_changed' => strtotime($rawSection->updated_at),
						'slug' => $rawSection->slug,
						'pages' => []
					];

					foreach($rawSection->pages as $rawPage) {
						$chapter['pages'][] = [
							'id' => $rawPage->id,
							'title' => $rawPage->name,
							'type' => 'page',
							'slug' => $rawPage->slug,
							'last_changed' => strtotime($rawPage->updated_at)
						];
					}

					if(count($chapter['pages']) > 0) {
						$r['contents'][] = $chapter;
					}

				} else {

					$r['contents'][] = [
						'title' => $rawSection->name,
						'id' => $rawSection->id,
						'type' => 'page',
						'slug' => $rawSection->slug,
						'last_changed' => strtotime($rawSection->updated_at)
					];

				}
			}

			return($r);
		}

		// try to get bookstack page entry from cache
		function query_db_page($id) {
			global $wpdb;
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
			$result = $wpdb->get_row($wpdb->prepare(
				"SELECT `valid`, UNIX_TIMESTAMP(NOW()) - UNIX_TIMESTAMP(`updated`) AS `age`, `title`, `content`, UNIX_TIMESTAMP(`last_changed`) AS `last_changed`, " .
				"`book`, `chapter`, `slug` FROM `" . $wpdb->prefix . "bb_integration_for_bookstack` " .
				"WHERE `id`=%d AND `book` IS NOT NULL AND `chapter` IS NOT NULL AND `content` IS NOT NULL", $id));
			if(is_null($result)) return(false);
			$valid = false;
			if($result->valid == 1) $valid = true;
			$r = [
				'valid' => $valid,
				'age' => $result->age,
				'title' => $result->title,
				'content' => $result->content,
				'last_changed' => $result->last_changed,
				'book' => $result->book,
				'chapter' => $result->chapter,
				'chapter' => $result->slug
			];
			return($r);
		}

		// try to get bookstack book entry from cache
		function query_db_book($bookId) {
			global $wpdb;

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
			$r = $wpdb->get_results($wpdb->prepare("SELECT
					`" . $wpdb->prefix . "bb_integration_for_bookstack`.`id` AS `page_id`,
					`" . $wpdb->prefix . "bb_integration_for_bookstack`.`title` AS `page_title`,
					`last_changed` AS `page_last_changed`,
					`" . $wpdb->prefix . "bb_integration_for_bookstack`.`book` AS `book_id`,
					`chapter` AS `chapter_id`,
					`" . $wpdb->prefix . "bb_integration_for_bookstack_chapters`.`valid` AS `chapter_valid`,
					UNIX_TIMESTAMP(NOW()) - UNIX_TIMESTAMP(`" . $wpdb->prefix . "bb_integration_for_bookstack_chapters`.`updated`) AS `chapter_age`,
					`" . $wpdb->prefix . "bb_integration_for_bookstack_chapters`.`title` AS `chapter_title`,
					`" . $wpdb->prefix . "bb_integration_for_bookstack_chapters`.`slug` AS `chapter_slug`,
					`" . $wpdb->prefix . "bb_integration_for_bookstack_books`.`valid` AS `book_valid`,
					UNIX_TIMESTAMP(NOW()) - UNIX_TIMESTAMP(`" . $wpdb->prefix . "bb_integration_for_bookstack_books`.`updated`) AS `book_age`,
					`" . $wpdb->prefix . "bb_integration_for_bookstack_books`.`title` AS `book_title`,
					`" . $wpdb->prefix . "bb_integration_for_bookstack_books`.`slug` AS `book_slug`
				FROM
					`" . $wpdb->prefix . "bb_integration_for_bookstack`
				LEFT JOIN
					`" . $wpdb->prefix . "bb_integration_for_bookstack_chapters`
				ON
					`" . $wpdb->prefix . "bb_integration_for_bookstack_chapters`.`id`=`chapter`
				LEFT JOIN
					`" . $wpdb->prefix . "bb_integration_for_bookstack_books`
				ON
					`" . $wpdb->prefix . "bb_integration_for_bookstack_books`.`id`=`" . $wpdb->prefix . "bb_integration_for_bookstack`.`book`
				WHERE
					`" . $wpdb->prefix . "bb_integration_for_bookstack`.`book`=%d
				ORDER BY
					`" . $wpdb->prefix . "bb_integration_for_bookstack`.`pos`, `" . $wpdb->prefix . "bb_integration_for_bookstack_chapters`.`pos`", $bookId));

			if(is_null($r)) return(false);

			$result = ['contents' => []];

			$age = 0;
			$valid = true;
			$currentChapterIndex = 0;
			$currentChapterId = 0;

			foreach($r as $row) {

				$age = max($age, $row->chapter_age, $row->book_age);
				if(($row->book_valid == 0) || ($row->chapter_valid == 0)) $valid = false;

				$result['title'] = $row->book_title;
				$result['slug'] = $row->book_slug;
				$result['id'] = $row->book_id;

				if($row->chapter_id == 0) {
					$result['contents'][] = [
						'id' => $row->page_id,
						'type' => 'page'
					];
				} else {
					if($currentChapterId == $row->chapter_id) {
						$result['contents'][$currentChapterIndex]['pages'][] = [
							'id' => $row->page_id,
							'type' => 'page'
						];
					} else {
						$currentChapterIndex = count($result['contents']);
						$currentChapterId = $row->chapter_id;
						$result['contents'][] = [
							'id' => $row->chapter_id,
							'type' => 'chapter',
							'title' => $row->chapter_title,
							'slug' => $row->chapter_slug,
							'pages' => []
						];
					}
				}
			}

			if(count($result['contents']) == 0) return(false);

			$result['age'] = $age;
			$result['valid'] = $valid;

			return($result);
		}

		// format output
		function generate_entry($id, $title, $structure, $content, $lastChanged, $showTitle, $showDate, $showBook, $showChapter) {

			// filter content to insert a paragraph with class "bb-min-width" after floating images
			// this is a fix to avoid very narrow text boxes next to floating images
			$content = trim(preg_replace('/\s+/', ' ', $content));
			$content = preg_replace('/(<p[^>]*><img[^>]*class="align-((left)|(right))"[^>]*>)/', '${1}</p><p class="bb-min-width">', $content);

			$bookStackUrl = str_replace('/', '\\/', get_option('bb_integration_for_bookstack_url'));

			// rewrite links
			$content = preg_replace(
				'/<a [^>]*href="' . $bookStackUrl . '\/books\/([a-z0-9-]+)\/chapter\/([a-z0-9-]+)">/',
				'<a href="?bb-bookstack-path=${1}/${2}/">'
				, $content
			);

			$content = preg_replace(
				'/<a [^>]*href="' . $bookStackUrl . '\/books\/([a-z0-9-]+)\/page\/([a-z0-9-]+)">/',
				'<a href="?bb-bookstack-path=${1}//${2}">'
				, $content
			);
			$content = preg_replace(
				'/<a [^>]*href="' . $bookStackUrl . '\/books\/([a-z0-9-]+)">/',
				'<a href="?bb-bookstack-path=${1}//">'
				, $content
			);
			$content = preg_replace(
				'/<a [^>]*href="' . $bookStackUrl . '\/link\/([0-9]+)[#]*[a-z0-9-]*">/',
				'<a href="?bb-bookstack-id=${1}">'
				, $content
			);

			$r = "";

			if($showBook) {
				$r .= "<div class='bb-bookstack-book'><span class='bb-bookstack-book-label'>" . $structure['title'] . "</span><br/>";
				foreach($structure['contents'] as $bcontent) {
					if($bcontent['type'] == 'chapter') {
						$r .= "<span class='bb-bookstack-chapter'><span class='bb-bookstack-chapter-label'>" . $bcontent['title'] . "</span><br/>";
						foreach($bcontent['pages'] as $page) {
							$active = '';
							if($page['id'] == $id) {
								$active = ' bb-active';
							}
							$r .= "<span class='bb-bookstack-page" . $active . "'><a href='?bb-bookstack-id=" . $page['id'] . "'>" . $page['title'] . "</a></span><br/>";
						}
						$r .= "</span>";
					} else {
						$active = '';
						if($bcontent['id'] == $id) {
							$active = ' bb-active';
						}
						$r .= "<span class='bb-bookstack-page" . $active . "'><a href='?bb-bookstack-id=" . $bcontent['id'] . "'>" . $bcontent['title'] . "</a></span><br/>";
					}
				}
				$r .= "</div>";
			} else if($showChapter) {
				// search to chapter containing page
				$chapter = false;
				foreach($structure['contents'] as $bcontent) {
					if($bcontent['type'] == 'chapter') {
						foreach($bcontent['pages'] as $page) {
							if($page['id'] == $id) {
								$chapter = $bcontent;
								break;
							}
						}
						if($chapter !== false) {
							break;
						}
					}
				}
				if($chapter !== false) {
					$r .= "<div class='bb-bookstack-chapter'><span class='bb-bookstack-chapter-label'>" . $chapter['title'] . "</span><br/>";
					foreach($chapter['pages'] as $page) {
						$active = '';
						if($page['id'] == $id) {
							$active = ' bb-active';
						}
						$r .= "<span class='bb-bookstack-page" . $active . "'><a href='?bb-bookstack-id=" . $page['id'] . "'>" . $page['title'] . "</a></span><br/>";
					}
					$r .= "</div>";
				}
			}

			if($showTitle) {
				$r .= "<div class='bb-bookstack-title'>" . $title . "</div>";
			}
			if($showDate) {
				$r .= "<div class='bb-bookstack-last-changed'>" . gmdate(get_option('date_format'), $lastChanged) . "</div>";
			}
			$r .= "<div class='bb-bookstack-content'>" . $content . "</div>";

			return($r);
		}

		// update or create page cache entry
		function update_db_page($id, $title, $content, $lastChanged, $book, $chapter, $slug) {
			global $wpdb;
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->query($wpdb->prepare("INSERT INTO `" . $wpdb->prefix . "bb_integration_for_bookstack` (`id`, `title`, `content`, `last_changed`, `book`, `chapter`, `slug`) " .
				"VALUES (%d, %s, %s, FROM_UNIXTIME(%d), %d, %d, %s) ON DUPLICATE KEY UPDATE `title`=VALUES(`title`), `content`=VALUES(`content`), " .
				"`last_changed`=VALUES(`last_changed`), `valid`=1, `updated`=NOW(), `book`=VALUES(`book`), `chapter`=VALUES(`chapter`), `slug`=VALUES(`slug`)",
				$id, $title, $content, $lastChanged, $book, $chapter, $slug));
		}

		// update or create book cache entries
		function update_db_book($id, $structure) {
			global $wpdb;

			// update book
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->query($wpdb->prepare("INSERT INTO `" . $wpdb->prefix . "bb_integration_for_bookstack_books` (`id`, `title`, `slug`) " .
				"VALUES (%d, %s, %s) ON DUPLICATE KEY UPDATE `title`=VALUES(`title`), `slug`=VALUES(`slug`)",
				$id, $structure['title'], $structure['slug']));

			// update chapters
			if(count($structure['contents']) > 0) {
				$first = true;
				$args = [];
				$pos = 1;
				$query = "INSERT INTO `" . $wpdb->prefix . "bb_integration_for_bookstack_chapters` (`id`, `title`, `slug`, `pos`, `book`) VALUES";
				foreach($structure['contents'] as $chapter) {
					if($chapter['type'] == 'chapter') {
						$args[] = $chapter['id'];
						$args[] = $chapter['title'];
						$args[] = $chapter['slug'];
						$args[] = $pos++;
						$args[] = $id;
						if($first) {
							$first = false;
						} else {
							$query .= ",";
						}
						$query .= " (%d, %s, %s, %d, %d)";
					}
				}
				$query .= " ON DUPLICATE KEY UPDATE `title`=VALUES(`title`), `slug`=VALUES(`slug`), `pos`=VALUES(`pos`), `updated`=NOW(), `valid`=1, `book`=VALUES(`book`)";
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared
				$wpdb->query($wpdb->prepare($query, ...$args));
			}

			// update pages
			if(count($structure['contents']) > 0) {
				$first = true;
				$args = [];
				$pos = 1;
				$query = "INSERT INTO `" . $wpdb->prefix . "bb_integration_for_bookstack` (`id`, `book`, `chapter`, `pos`, `slug`) VALUES";
				foreach($structure['contents'] as $content) {
					if($content['type'] == 'chapter') {
						foreach($content['pages'] as $page) {
							$args[] = $page['id'];
							$args[] = $id;
							$args[] = $content['id'];
							$args[] = $pos++;
							$args[] = $page['slug'];
							if($first) {
								$first = false;
							} else {
								$query .= ",";
							}
							$query .= " (%d, %d, %d, %d, %s)";
						}
					} else if($content['type'] == 'page') {
						$args[] = $content['id'];
						$args[] = $id;
						$args[] = 0;
						$args[] = $pos++;
						$args[] = $content['slug'];
						if($first) {
							$first = false;
						} else {
							$query .= ",";
						}
						$query .= " (%d, %d, %d, %d, %s)";
					}
				}
				$query .= " ON DUPLICATE KEY UPDATE `book`=VALUES(`book`), `chapter`=VALUES(`chapter`), `pos`=VALUES(`pos`), `slug`=VALUES(`slug`)";
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared
				$wpdb->query($wpdb->prepare($query, ...$args));
			}
		}

		function update_book($id) {
			$resultBook = $this->query_book($id);
			if($resultBook !== false) $this->update_db_book($id, $resultBook);
		}

		function page_id_not_found() {
			return($this->show_error_page(
				'bb_integration_for_bookstack_error_page_not_found',
				__('Content cannot be loaded. BookStack site not found.', 'bastionbytes-integration-for-bookstack'),
				__('Unable to load error post (BookStack site not found) with ulr path "%%URL_PATH%%".', 'bastionbytes-integration-for-bookstack')
			));
		}

		function page_empty() {
			return($this->show_error_page(
				'bb_integration_for_bookstack_error_page_empty',
				__('Content cannot be loaded. BookStack site is empty.', 'bastionbytes-integration-for-bookstack'),
				__('Unable to load error post (BookStack site is empty) with ulr path "%%URL_PATH%%".', 'bastionbytes-integration-for-bookstack')
			));
		}

		function show_error_page($errorPageOption, $defaultMessage, $errorMessage) {
			$url = get_option($errorPageOption);
                        if($url === false) return(htmlentities($defaultMessage));
			$url = trim($url);
			if($url == '') return(htmlentities($defaultMessage));
			$id = url_to_postid($url);
			if($id === 0) {
				return(htmlentities(str_replace('%%URL_PATH%%', $url, $errorMessage)));
			}
			$post = get_post($id);
			$content = $post->post_content;
			// remove bb-bookstack shortcodes to avoid loops
			$content = preg_replace(
				'/\[bb-bookstack[^\]]*\]/',
				__('** bb-bookstack shortcode removed **', 'bastionbytes-integration-for-bookstack'),
				$content
			);
			$content = do_shortcode($content);
			return($content);
		}

		// the registered shortcode function
		function shortcode($atts = [], $content = null) {
			global $wpdb;

			if(get_query_var('bb-bookstack-id')) {
				if(!is_numeric(get_query_var('bb-bookstack-id'))) return(__('Content cannot be loaded.', 'bastionbytes-integration-for-bookstack'));
				$id = get_query_var('bb-bookstack-id');
			} else if(array_key_exists('id', $atts)) {
				if(!is_numeric($atts['id'])) return(__('Content cannot be loaded.', 'bastionbytes-integration-for-bookstack'));
				$id = $atts['id'];
			} else {
				return(__('Content cannot be loaded.', 'bastionbytes-integration-for-bookstack'));
			}

			$id = intval($id); // to be sure it's an integer

			// check if id get overwritten by bb-bookstack-path var
			$forceShowChapter = false;
			$forceShowBook = false;
			if(get_query_var('bb-bookstack-path')) {
				preg_match('/([a-z0-9-]*)\/([a-z0-9-]*)\/([a-z0-9-]*)/', get_query_var('bb-bookstack-path'), $matches);
				$book = $matches[1];
				$chapter = $matches[2];
				$page = $matches[3];
				if($book != '') {
					$query = false;
					$args = false;
					if(($chapter == '') && ($page != '')) {

						// query by book and page
						$bookId = $this->get_book_id_by_slug($book);
						if($bookId !== false) {
							$query = "SELECT `id` from `" . $wpdb->prefix . "bb_integration_for_bookstack` WHERE `book`=%d AND `slug`=%s";
							$args = [$bookId, $page];
						}

					} else if(($chapter != '') && ($page == '')) {

						// query by book and chapter
						$bookId = $this->get_book_id_by_slug($book);
						if($bookId !== false) {
							$chapterId = $this->get_chapter_id_by_slug($bookId, $chapter);
							if($chapterId !== false) {
								$query = "SELECT `id` from `" . $wpdb->prefix . "bb_integration_for_bookstack` WHERE `chapter`=%d ORDER BY `pos` LIMIT 1";
								$args = [$chapterId];
							}
						}

					} else {
						// query by book
						$bookId = $this->get_book_id_by_slug($book);

						if($bookId !== false) {
							$query = "SELECT `id` from `" . $wpdb->prefix . "bb_integration_for_bookstack` WHERE `book`=%d ORDER BY `pos` LIMIT 1";
							$args = [$bookId];
						}
					}

					if($query !== false) {
						$result = null;
						while(is_null($result)) {
							// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared
							$result = $wpdb->get_row($wpdb->prepare($query, ...$args));
							if(is_null($result)) {
								if($fullUpdate) {
									break;
								}
								$this->update_book($bookId);
								$fullUpdate = true;
							}
						}
						if(!is_null($result)) {
							$forceShowBook = true;
							$id = $result->id;
						}
					}
				}
			}


			$showTitle = in_array('show_title', $atts);
			$hideTitle = in_array('hide_title', $atts);
			$showDate = in_array('show_date', $atts);
			$hideDate = in_array('hide_date', $atts);
			if($showTitle && $hideTitle) $hideTitle = false;
			if($showDate && $hideDate) $hideDate = false;
			if(!$showTitle && !$hideTitle) {
				$showTitle = get_option('bb_integration_for_bookstack_show_title') == 1;
			} else if($hideTitle) {
				$showTitle = false;
			}
			if(!$showDate && !$hideDate) {
				$showDate = get_option('bb_integration_for_bookstack_show_date') == 1;
			} else if($hideDate) {
				$showDate = false;
			}

			$showBook = in_array('book', $atts) || $forceShowBook;
			$showChapter = $showBook || in_array('chapter', $atts) || $forceShowChapter;

			$mode = get_option('bb_integration_for_bookstack_mode');
			$maxCacheAge = get_option('bb_integration_for_bookstack_max_cache_age');

			$resultPage = false;
			$resultBook = false;

			$writePageToCache = false;
			$writeBookToCache = false;

			if($mode != 'no_cache') {
				$resultPage = $this->query_db_page($id);
			}

			if(($resultPage === false) || ($resultPage['valid'] === false) || ($resultPage['age'] > $maxCacheAge * 3600)) {
				// cache miss: page
				if($mode != 'no_bookstack') {
					$resultPage2 = $this->query_page($id);
					if($resultPage2 !== false) {
						$resultPage = $resultPage2;
						$writePageToCache = true;
					}
				}
			}

			if($resultPage === false) {
				return($this->page_id_not_found());
			} else if(trim($resultPage['content']) === '') {
				return($this->page_empty());
			}

			if($showChapter && ($resultPage !== false)) {

				if($mode != 'no_cache') {
					$resultBook = $this->query_db_book($resultPage['book']);
				}

				if(($resultBook === false) || ($resultBook['valid'] === false) || ($resultBook['age'] > $maxCacheAge * 3600)) {
					// cache miss: book
					if($mode != 'no_bookstack') {
						$resultBook2 = $this->query_book($resultPage['book']);
						if($resultBook2 !== false) {
							$resultBook = $resultBook2;
							$writeBookToCache = true;
						}
					}
				}
			}

			if($writePageToCache) {
				$this->update_db_page($id, $resultPage['title'], $resultPage['content'], $resultPage['last_changed'], $resultPage['book'], $resultPage['chapter'], $resultPage['slug']);
			}

			if($writeBookToCache) {
				$this->update_db_book($resultPage['book'], $resultBook);
			}

			return($this->generate_entry($id, $resultPage['title'], $resultBook, $resultPage['content'], $resultPage['last_changed'], $showTitle, $showDate, $showBook, $showChapter));

		}

		function generate_webhook_secret() {
			$cs = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
			$csLen = strlen($cs);
			$r = '';
			for($i = 0; $i < 32; $i++) {
				$r .= $cs[random_int(0, $csLen - 1)];
			}
			return($r);
		}

		// registered plugin init function
		function upgrade() {
			global $wpdb;

			$version = get_option('bb_integration_for_bookstack_db_version');
			if($version === false) $version = 0;
			$version = intval($version);

			if($version < 1) {
				$this->init(); // update schema

				// delete unusable cache entries
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
				$wpdb->query("DELETE FROM `". $wpdb->prefix . "bb_integration_for_bookstack` WHERE `book` IS NULL");

				$version = 1;
				update_option('bb_integration_for_bookstack_db_version', $version);
			}

			if($version < 2) {
				$this->init(); // update schema

				// delete unusable cache entries
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
				$wpdb->query("DELETE FROM `". $wpdb->prefix . "bb_integration_for_bookstack`");
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
				$wpdb->query("DELETE FROM `". $wpdb->prefix . "bb_integration_for_bookstack_chapters`");
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
				$wpdb->query("DELETE FROM `". $wpdb->prefix . "bb_integration_for_bookstack_books`");

				$version = 2;
				update_option('bb_integration_for_bookstack_db_version', $version);
			}

			if($version < 3) {
				$this->init(); // update schema
				$version = 3;
				update_option('bb_integration_for_bookstack_db_version', $version);
			}
		}

		function init() {
			global $wpdb;

			add_option('bb_integration_for_bookstack_url', 'https://url.to.your.bookstack');
			add_option('bb_integration_for_bookstack_token_id', 'your_token_id');
			add_option('bb_integration_for_bookstack_token_secret', 'your_token_secret');
			add_option('bb_integration_for_bookstack_webhook_secret', $this->generate_webhook_secret());
			add_option('bb_integration_for_bookstack_mode', 'normal');
			add_option('bb_integration_for_bookstack_max_cache_age', 24);
			add_option('bb_integration_for_bookstack_irgnore_cert', 0);
			add_option('bb_integration_for_bookstack_show_title', 1);
			add_option('bb_integration_for_bookstack_show_date', 1);
			add_option('bb_integration_for_bookstack_error_page_not_found', '');
			add_option('bb_integration_for_bookstack_error_page_empty', '');

			require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );

			// id: the bookstack page id
			// valid: if the cache entry is valid
			// updated: last renew timestamp of the entry
			// title: the page title
			// content: the page content
			// last_changed: last chnage of the bookstack entry
			// book: book id of this page
			// chapter: chapter id of this page
			$query = "CREATE TABLE `" . $wpdb->prefix . "bb_integration_for_bookstack` (
				`id` INT NOT NULL UNIQUE,
				`valid` INT(1) NOT NULL DEFAULT '1',
				`updated` TIMESTAMP NOT NULL DEFAULT NOW(),
				`title` TEXT NOT NULL,
				`content` MEDIUMTEXT,
				`last_changed` TIMESTAMP NOT NULL DEFAULT NOW(),
				`book` INT,
				`chapter` INT,
				`pos` int(11) NOT NULL DEFAULT 1,
				`slug` varchar(255) NOT NULL
				) " . $wpdb->get_charset_collate() . ";";
			dbDelta($query);

			// id: the bookstack chapter id
			// valid: if the cache entry is valid
			// updated: last renew timestamp of the entry
			// title: the chapter title
			$query = "CREATE TABLE `" . $wpdb->prefix . "bb_integration_for_bookstack_chapters` (
				`id` int(11) NOT NULL UNIQUE,
				`valid` int(1) NOT NULL DEFAULT 1,
				`updated` timestamp NOT NULL DEFAULT current_timestamp(),
				`title` text NOT NULL,
				`slug` varchar(255) NOT NULL,
				`pos` int(11) NOT NULL DEFAULT 1,
				`book` int(11)
				) " . $wpdb->get_charset_collate() . ";";
			dbDelta($query);

			// id: the bookstack book id
			// valid: if the cache entry is valid
			// updated: last renew timestamp of the entry
			// title: the book title
			$query = "CREATE TABLE `" . $wpdb->prefix . "bb_integration_for_bookstack_books` (
				`id` int(11) NOT NULL UNIQUE,
				`valid` int(1) NOT NULL DEFAULT 1,
				`updated` timestamp NOT NULL DEFAULT current_timestamp(),
				`title` text NOT NULL,
				`slug` varchar(255) NOT NULL
				) " . $wpdb->get_charset_collate() . ";";
			dbDelta($query);

		}

		// the backend configuration page
		function backend_options() {
?>
	<div class='wrap'>
		<h1><?php esc_html_e('Settings > BastionBytes integration for BookStack', 'bastionbytes-integration-for-bookstack'); ?></h1>
		<?php if($this->message_error !== false) { ?>
			<div id="bookstack-error" class="notice notice-error settings-error">
				<p>
					<strong><?php echo esc_html($this->message_error); ?></strong>
				</p>
			</div>
		<?php } ?>
		<?php if($this->message_success !== false) { ?>
			<div id="bookstack-error-settings_updated" class="notice notice-success settings-error">
				<p>
					<strong><?php echo esc_html($this->message_success); ?></strong>
				</p>
			</div>
		<?php } ?>
		<form action="<?php menu_page_url('bb_integration_for_bookstack_backend_options'); ?>" method="post">
			<?php wp_nonce_field('bb_integration_for_bookstack_backend_options'); ?>
			<table class="form-table" role="presentation">
				<tr>
					<th scope="row">
						<label><?php esc_html_e('BookStack URL', 'bastionbytes-integration-for-bookstack'); ?></label>
					</th>
					<td>
						<input type="text" name="bb_integration_for_bookstack_option_url" class="regular-text" value="<?php echo esc_html(get_option('bb_integration_for_bookstack_url')); ?>">
					</td>
				</tr>
				<tr>
					<th scope="row">
						<label><?php esc_html_e('BookStack Token ID', 'bastionbytes-integration-for-bookstack'); ?></label>
					</th>
					<td>
						<input type="text" name="bb_integration_for_bookstack_option_token_id" class="regular-text" value="<?php echo esc_html(get_option('bb_integration_for_bookstack_token_id')); ?>">
					</td>
				</tr>
				<tr>
					<th scope="row">
						<label><?php esc_html_e('BookStack Token Secret', 'bastionbytes-integration-for-bookstack'); ?></label>
					</th>
					<td>
						<input type="text" name="bb_integration_for_bookstack_option_token_secret" class="regular-text" value="" placeholder="<?php esc_html_e('hidden', 'bastionbytes-integration-for-bookstack'); ?>">
						<p class="description">
							<?php esc_html_e('If this field is left empty, no changes will be made', 'bastionbytes-integration-for-bookstack'); ?>
						</p>
					</td>
				</tr>
				<tr>
					<th scope="row">
						<label><?php esc_html_e('Show BookStack page title', 'bastionbytes-integration-for-bookstack'); ?></label>
					</th>
					<td>
						<?php
							$checked = "";
							if(get_option('bb_integration_for_bookstack_show_title') == 1) $checked = "checked";
						?>
						<input name="bb_integration_for_bookstack_option_show_title" type="checkbox" <?php echo esc_html($checked); ?>>
					</td>
				</tr>
				<tr>
					<th scope="row">
						<label><?php esc_html_e('Show BookStack last change date', 'bastionbytes-integration-for-bookstack'); ?></label>
					</th>
					<td>
						<?php
							$checked = "";
							if(get_option('bb_integration_for_bookstack_show_date') == 1) $checked = "checked";
						?>
						<input name="bb_integration_for_bookstack_option_show_date" type="checkbox" <?php echo esc_html($checked); ?>>
					</td>
				</tr>
				<tr>
					<th scope="row">
						<label><?php esc_html_e('Cache-Timeout (hours)', 'bastionbytes-integration-for-bookstack'); ?></label>
					</th>
					<td>
						<input type="number" name="bb_integration_for_bookstack_option_max_cache_age" class="regular-text" value="<?php echo esc_html(get_option('bb_integration_for_bookstack_max_cache_age')); ?>">
						<p class="description">
							<?php esc_html_e('The value &quot;0&quot; means that cache entries never expire', 'bastionbytes-integration-for-bookstack'); ?>
						</p>
					</td>
				</tr>
				<tr>
					<th scope="row">
						<label><?php esc_html_e('Mode', 'bastionbytes-integration-for-bookstack'); ?></label>
					</th>
					<td>
						<?php
							$mode = get_option('bb_integration_for_bookstack_mode');
							$modes = ['normal', 'no_bookstack', 'no_cache'];
							$mode_labels = [
								__('normal', 'bastionbytes-integration-for-bookstack'),
								__('never connect to BookStack (use cache only)', 'bastionbytes-integration-for-bookstack'),
								__('always connect to BookStack (turn off cache)', 'bastionbytes-integration-for-bookstack')
							];
							if(!in_array($mode, $modes)) $mode = 'normal'
						?>
						<select name="bb_integration_for_bookstack_option_mode">
							<?php
								for($i = 0; $i < count($modes); $i++) {
									echo '<option value="' . esc_html($modes[$i]) . '"';
									if($modes[$i] == $mode) echo ' selected';
									echo '>' . esc_html($mode_labels[$i]) . '</option>';
								}
							?>
						</select>
					</td>
				</tr>
				<tr>
					<th scope="row">
						<label><?php esc_html_e('Ignore BookStack certificate', 'bastionbytes-integration-for-bookstack'); ?></label>
					</th>
					<td>
						<?php
							$checked = "";
							if(get_option('bb_integration_for_bookstack_ignore_cert') == 1) $checked = "checked";
						?>
						<input name="bb_integration_for_bookstack_option_ignore_cert" type="checkbox" <?php echo esc_html($checked); ?>>
						<p class="description">
							<?php esc_html_e('A faulty BookStack certificate will throw no error', 'bastionbytes-integration-for-bookstack'); ?>
						</p>
					</td>
				</tr>
				<tr>
					<th scope="row">
						<label><?php esc_html_e('Site not found: post URL', 'bastionbytes-integration-for-bookstack'); ?></label>
					</th>
					<td>
						<input type="text" name="bb_integration_for_bookstack_option_error_page_not_found" class="regular-text" value="<?php echo esc_html(get_option('bb_integration_for_bookstack_error_page_not_found')); ?>">
						<p class="description">
							<?php esc_html_e('If a BookStack site was not found, display the post with the given URL. If empty a generic message will we shown.', 'bastionbytes-integration-for-bookstack'); ?>
						</p>
					</td>
				</tr>
				<tr>
					<th scope="row">
						<label><?php esc_html_e('Site empty: post URL', 'bastionbytes-integration-for-bookstack'); ?></label>
					</th>
					<td>
						<input type="text" name="bb_integration_for_bookstack_option_error_page_empty" class="regular-text" value="<?php echo esc_html(get_option('bb_integration_for_bookstack_error_page_empty')); ?>">
						<p class="description">
							<?php esc_html_e('If a BookStack site is empty, display the post with the given URL. If empty a generic message will we shown.', 'bastionbytes-integration-for-bookstack'); ?>
						</p>
					</td>
				</tr>
				<tr>
					<th scope="row">
						<label><?php esc_html_e('BookStack Webhook URL', 'bastionbytes-integration-for-bookstack'); ?></label>
					</th>
					<td><?php echo esc_html(get_site_url(null, '', 'rpc')); ?>/wp-json/bookstack/v1/site-update?<?php echo esc_html(get_option('bb_integration_for_bookstack_webhook_secret')); ?></td>
				</tr>
			</table>
			<p class="submit">
				<input type="submit" name="bb_integration_for_bookstack_update_config" class="button button-primary" value="<?php esc_html_e('Save changes', 'bastionbytes-integration-for-bookstack'); ?>">
			</p>

			<hr style='border-bottom-width: 2px;border-top-width: 2px;border-top-style: solid;'>

			<h2><?php esc_html_e('Administrativ action', 'bastionbytes-integration-for-bookstack'); ?></h2>
			<table class="form-table" role="presentation">
				<tr>
					<th scope="row">
						<label><?php esc_html_e('Regenerate Webhook URL', 'bastionbytes-integration-for-bookstack'); ?></label>
					</th>
					<td>
						<input type="submit" name="bb_integration_for_bockstack_generate_webhook_secret" class="button button-primary" value="<?php esc_html_e('Execute action', 'bastionbytes-integration-for-bookstack'); ?>">
					</td>
				</tr>
				<tr>
					<th scope="row">
						<label><?php esc_html_e('Invalidate all cache entries', 'bastionbytes-integration-for-bookstack'); ?></label>
					</th>
					<td>
						<input type="submit" name="bb_integration_for_bookstack_invalidate_cache" class="button button-primary" value="<?php esc_html_e('Execute action', 'bastionbytes-integration-for-bookstack'); ?>">
					</td>
				</tr>
				<tr>
					<th scope="row">
						<label><?php esc_html_e('Delete all cache entries', 'bastionbytes-integration-for-bookstack'); ?></label>
					</th>
					<td>
						<input type="submit" name="bb_integration_for_bookstack_drop_cache" class="button button-primary" value="<?php esc_html_e('Execute action', 'bastionbytes-integration-for-bookstack'); ?>">
					</td>
				</tr>
			</table>
		</form>
		<hr style='border-bottom-width: 2px;border-top-width: 2px;border-top-style: solid;'>
		<?php
			$helpFile = realpath(dirname(__FILE__)) . '/languages/help-' . get_locale() . '.html';
			if(!file_exists($helpFile)) {
				$helpFile = realpath(dirname(__FILE__)) . '/languages/help.html';
			}
			include($helpFile);
		?>
	</div>
<?php
		}

		// the missing php startsWith function for strings
		function starts_with($x, $prefix) {
			if(strlen($x) < strlen($prefix)) {
				return(true);
			}
			return(substr($x, 0, strlen($prefix)) == $prefix);
		}

		// the form submit function registered in options_menu()
		function backend_options_submit() {
			global $wpdb;

			if(!isset($_SERVER['REQUEST_METHOD'])) return;
			if($_SERVER['REQUEST_METHOD'] !== 'POST') return;

			// check nonce
			check_admin_referer('bb_integration_for_bookstack_backend_options');

			if(isset($_POST['bb_integration_for_bookstack_generate_webhook_secret'])) {

				update_option('bb_integration_for_bookstack_webhook_secret', $this->generate_webhook_secret());
				$this->message_success = __('New webhook URL created', 'bastionbytes-integration-for-bookstack');

			} else if(isset($_POST['bb_integration_for_bookstack_invalidate_cache'])) {

				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
				$wpdb->query("UPDATE `" . $wpdb->prefix . "bb_integration_for_bookstack` SET `valid`=0");

				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
				$wpdb->query("UPDATE `" . $wpdb->prefix . "bb_integration_for_bookstack_chapters` SET `valid`=0");

				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
				$wpdb->query("UPDATE `" . $wpdb->prefix . "bb_integration_for_bookstack_books` SET `valid`=0");

				$this->message_success = __('All cache entries invalidated', 'bastionbytes-integration-for-bookstack');

			} else if(isset($_POST['bb_integration_for_bookstack_drop_cache'])) {

				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
				$wpdb->query("DELETE FROM `" . $wpdb->prefix . "bb_integration_for_bookstack`");

				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
				$wpdb->query("DELETE FROM `" . $wpdb->prefix . "bb_integration_for_bookstack_chapters`");

				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
				$wpdb->query("DELETE FROM `" . $wpdb->prefix . "bb_integration_for_bookstack_books`");

				$this->message_success = __('All cache entries deleted', 'bastionbytes-integration-for-bookstack');

			} else if(isset($_POST['bb_integration_for_bookstack_update_config'])) {

				$url = '';
				$token_id = '';
				$token_secret = '';
				$max_cache_age = '';
				if(isset($_POST['bb_integration_for_bookstack_option_url'])) $url = sanitize_url(wp_unslash($_POST['bb_integration_for_bookstack_option_url']));
				if(isset($_POST['bb_integration_for_bookstack_option_token_id'])) $token_id = trim(sanitize_text_field(wp_unslash($_POST['bb_integration_for_bookstack_option_token_id'])));
				if(isset($_POST['bb_integration_for_bookstack_option_token_secret'])) $token_secret = trim(sanitize_text_field(wp_unslash($_POST['bb_integration_for_bookstack_option_token_secret'])));
				if(isset($_POST['bb_integration_for_bookstack_option_max_cache_age'])) $max_cache_age = trim(sanitize_text_field(wp_unslash($_POST['bb_integration_for_bookstack_option_max_cache_age'])));
				if(isset($_POST['bb_integration_for_bookstack_option_error_page_not_found'])) $errorPageNotFoundUrl = trim(sanitize_text_field(wp_unslash($_POST['bb_integration_for_bookstack_option_error_page_not_found'])));
				if(isset($_POST['bb_integration_for_bookstack_option_error_page_empty'])) $errorPageEmptyUrl = trim(sanitize_text_field(wp_unslash($_POST['bb_integration_for_bookstack_option_error_page_empty'])));

				$mode = 'normal';
				$modes = ['normal', 'no_bookstack', 'no_cache'];
				if(isset($_POST['bb_integration_for_bookstack_option_mode'])) $mode = trim(sanitize_text_field(wp_unslash($_POST['bb_integration_for_bookstack_option_mode'])));
				if(!in_array($mode, $modes)) $mode = 'normal';

				// some input validation checks
				if(!$this->starts_with($url, "https://") && !$this->starts_with($url, "http://")) {
					$this->message_error = __('The BookStack URL has to begin with http:// oder https://', 'bastionbytes-integration-for-bookstack');
				}
				$host = substr($url, strpos($url, '//') + 2);
				if(strlen($host) < 4 || strlen($host) > 100) {
					$this->message_error = __('The BookStack URL is too short or too long', 'bastionbytes-integration-for-bookstack');
				}
				if(!preg_match('/^[a-zA-Z0-9\/.-]*$/', $host)) {
					$this->message_error = __('The BookStack URL contains invalid characters', 'bastionbytes-integration-for-bookstack');
				}
				if(!preg_match('/^[a-zA-Z0-9]*$/', $token_id)) {
					$this->message_error = __('The token ID contains invalid characters', 'bastionbytes-integration-for-bookstack');
				}
				if(strlen($token_id) < 4 || strlen($token_id) > 100) {
					$this->message_error = __('The token ID is too short or too long', 'bastionbytes-integration-for-bookstack');
				}
				if($token_secret != "") {
					if(!preg_match('/^[a-zA-Z0-9]*$/', $token_secret)) {
						$this->message_error = __('The token secret contains invalid characters', 'bastionbytes-integration-for-bookstack');
					}
					if(strlen($token_secret) < 4 || strlen($token_secret) > 100) {
						$his->message_error = __('The token secret is too short or too long', 'bastionbytes-integration-for-bookstack');
					}
				}
				if(!is_numeric($max_cache_age)) {
					$this->message_error = __('Cache-Timeout is not a number', 'bastionbytes-integration-for-bookstack');
				}
				$max_cache_age = intval($max_cache_age);
				if(($max_cache_age < 0) || ($max_cache_age > 336)) {
					$this->message_error = __('Cache-Timeout must be at least 0 and maximum of 336', 'bastionbytes-integration-for-bookstack');
				}



				// if no error occured update the values
				if($this->message_error === false) {

					while(substr($url, -1) == '/') {
						$url = substr($url, 0, -1);
					}

					update_option('bb_integration_for_bookstack_url', $url);
					update_option('bb_integration_for_bookstack_token_id', $token_id);
					update_option('bb_integration_for_bookstack_max_cache_age', $max_cache_age);
					update_option('bb_integration_for_bookstack_error_page_not_found', $errorPageNotFoundUrl);
					update_option('bb_integration_for_bookstack_error_page_empty', $errorPageEmptyUrl);

					$show_title = 0;
					if(isset($_POST['bb_integration_for_bookstack_option_show_title'])) $show_title = 1;
					update_option('bb_integration_for_bookstack_show_title', $show_title);

					$show_date = 0;
					if(isset($_POST['bb_integration_for_bookstack_option_show_date'])) $show_date = 1;
					update_option('bb_integration_for_bookstack_show_date', $show_date);

					$ignore_cert = 0;
					if(isset($_POST['bb_integration_for_bookstack_option_ignore_cert'])) $ignore_cert = 1;
					update_option('bb_integration_for_bookstack_ignore_cert', $ignore_cert);

					update_option('bb_integration_for_bookstack_mode', $mode);

					if($token_secret != "") {
						update_option('bb_integration_for_bookstack_token_secret', $token_secret);
					}
					$this->message_success = __('Changes saved', 'bastionbytes-integration-for-bookstack');
				}
			}
		}

		function webhook(WP_REST_Request $request) {
			global $wpdb;

			if(!isset($request[get_option('bb_integration_for_bookstack_webhook_secret')])) {
				http_response_code(401);
				die('permission denied');
			}
			$json = $request->get_json_params();

			// log webhook requests from bookstack
			/*
			$log = fopen(dirname(__FILE__) . '/webhook.json.log', 'a');
			fwrite($log, '*** ' . gmdate(get_option('date_format')) . ' ***' . "\n");
			fwrite($log, json_encode($json) . "\n");
			fclose($log);
			*/

			if(!isset($json['event'])) {
				http_response_code(400);
				die('bad request');
			}
			$event = $json['event'];

			$booksToInvalidate = [];
			$chaptersToInvalidate = [];
			$pagesToInvalidate = [];

			if($event == 'page_move') {
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
				$result = $wpdb->get_row($wpdb->prepare('SELECT `book` FROM `' . $wpdb->prefix . 'bb_integration_for_bookstack` WHERE `id`=%d', $json['related_item']['id']));
				if(!is_null($result) && !is_null($result->book)) {
					$booksToInvalidate[] = intval($result->book);
				}
			}

			if($event == 'chapter_move') {
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
				$result = $wpdb->get_row($wpdb->prepare('SELECT `chapter` FROM `' . $wpdb->prefix . 'bb_integration_for_bookstack` WHERE `id`=%d', $json['related_item']['id']));
				if(!is_null($result) && !is_null($result->chapter)) {
					$booksToInvalidate[] = intval($result->chapter);
				}
			}

			if(in_array($event, ['page_update', 'page_delete', 'page_move'])) {
				$pagesToInvalidate[] = intval($json['related_item']['id']);
			}

			if(in_array($event, ['chapter_create', 'chapter_update', 'chapter_delete', 'page_delete', 'page_move', 'chapter_move'])) {
				$booksToInvalidate[] = intval($json['related_item']['book_id']);
			}

			if(in_array($event, ['book_update', 'book_delete', 'book_sort', 'page_create'])) {
				$booksToInvalidate[] = intval($json['related_item']['id']);
			}

			if(in_array($event, ['chapter_create', 'chapter_update', 'chapter_delete'])) {
				$chaptersToInvalidate[] = intval($json['related_item']['id']);
			}

			if(count($booksToInvalidate) + count($chaptersToInvalidate) + count($pagesToInvalidate) == 0) {
				http_response_code(400);
				die('bad request');
			}

			foreach($pagesToInvalidate as $page) {
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
				$wpdb->query($wpdb->prepare('UPDATE `' . $wpdb->prefix . 'bb_integration_for_bookstack` SET `valid`=0 WHERE `id`=%d', $page));
			}

			foreach($chaptersToInvalidate as $chapter) {
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
				$wpdb->query($wpdb->prepare('UPDATE `' . $wpdb->prefix . 'bb_integration_for_bookstack_chapters` SET `valid`=0 WHERE `id`=%d', $chapter));
			}

			foreach($booksToInvalidate as $book) {
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
				$wpdb->query($wpdb->prepare('UPDATE `' . $wpdb->prefix . 'bb_integration_for_bookstack_books` SET `valid`=0 WHERE `id`=%d', $book));
			}
		}

	}
}

BastionBytesIntegrationForBookStack::instance();

