<?php

/**
 * Plugin Name: Barilo Light – Top Bar Message
 * Description: Display a custom message in a top bar. Perfect for announcements, alerts, greetings, or promotions.
 * Version: 1.1.0
 * Author: Radislav Nahaba 
 * Author URI: https://profiles.wordpress.org/nahabagroup/
 * Requires at least: 6.7
 * Tested up to: 6.9
 * Requires PHP: 8.2
 * License: GPL2+
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: barilo-light-top-bar-message
 */

// enqueue barilo CSS
function barilo_enqueue_styles()
{
   wp_enqueue_style(
      'barilo-style',
      plugin_dir_url(__FILE__) . 'assets/css/barilo.css',
      [],
      '1.1.0'
   );
}
add_action('wp_enqueue_scripts', 'barilo_enqueue_styles');

// admin enqueue
add_action('admin_enqueue_scripts', 'barilo_enqueue_color_picker');

function barilo_enqueue_color_picker($hook)
{
   // Load only on Barilo Light settings page.
   if ($hook !== 'settings_page_barilo-light-top-bar-message-settings') {
      return;
   }

   // Built-in WP color picker styles & script.
   wp_enqueue_style('wp-color-picker');
   wp_enqueue_script(
      'barilo-color-picker',
      plugin_dir_url(__FILE__) . 'assets/barilo-color-picker.js',
      ['wp-color-picker', 'jquery'],
      '1.1.0',
      true
   );
}


// message in top bar settings
function barilo_display_top_bar()
{
   $hide_bar = get_option('barilo_hide_bar', 0);
   if ($hide_bar) {
      return;
   }

   $hide_after_date = get_option('barilo_hide_after_date', '');
   if ($hide_after_date !== '') {
      $today = current_time('Y-m-d');
      if ($today > $hide_after_date) {
         return;
      }
   }

   $message = get_option('barilo_top_message', '');
   $phone = trim(get_option('barilo_phone', ''));
   $email = trim(get_option('barilo_email', ''));
   $bg_color = get_option('barilo_bg_color', 'lightgray');
   $text_color = get_option('barilo_text_color', 'black');

   $color_map = [
      'lightblue'    => '#e6f4ff',
      'lightgreen'   => '#e6ffe6',
      'beige'        => '#fff5e6',
      'lavender'     => '#f3e6ff',
      'lightgray'    => '#f2f2f2',
      'darkgray'     => '#333333',
      'white'        => '#ffffff',
      'black'        => '#000000',
      'navy'         => '#1a1a80',
      'teal'         => '#005f5f',
      'crimson'      => '#cc0033',
      'orange'       => '#ffd699',
      'purple'       => '#e0ccff',
   ];

   // Background color: hex or legacy name.
   if (is_string($bg_color) && strpos($bg_color, '#') === 0) {
      $bg_hex = sanitize_hex_color($bg_color) ?: '#f2f2f2';
   } else {
      $bg_hex = $color_map[$bg_color] ?? '#f2f2f2';
   }

   // Text color: hex or legacy name.
   if (is_string($text_color) && strpos($text_color, '#') === 0) {
      $text_hex = sanitize_hex_color($text_color) ?: '#000000';
   } else {
      $text_hex = $color_map[$text_color] ?? '#000000';
   }


   // If nothing at all is set, don't output the bar.
   if ($message === '' && $phone === '' && $email === '') {
      return;
   }


   $inner_html = '';

   // Message
   if ($message !== '') {
      $inner_html .= '<span class="barilo-message">' . wp_kses_post($message) . '</span>';
   }

   // Phone / email block.
   if ($phone !== '' || $email !== '') {
      $inner_html .= '<span class="barilo-contact">';

      if ($phone !== '') {

         $tel_href = 'tel:' . preg_replace('/[^0-9+]/', '', $phone);

         $inner_html .= '<a href="' . esc_url($tel_href) . '" class="barilo-phone">📞 ' . esc_html($phone) . '</a>';
      }

      if ($email !== '') {
         $safe_email = sanitize_email($email);
         if ($safe_email) {

            if ($phone !== '') {
               $inner_html .= '<span class="barilo-contact-sep"></span>';
            }

            $inner_html .= '<a href="mailto:' . esc_attr($safe_email) . '" class="barilo-email">✉️ ' . esc_html($safe_email) . '</a>';
         }
      }

      $inner_html .= '</span>';
   }

   $allowed_barilo_html = [
      'span' => ['class' => true],
      'a'    => [
         'href'   => true,
         'class'  => true,
         'title'  => true,
         'target' => true,
         'rel'    => true,
      ],
      'strong' => [],
      'em'     => [],
      'br'     => [],
   ];

   echo '<div class="barilo-top-bar" style="background-color:' . esc_attr($bg_hex) . '; color:' . esc_attr($text_hex) . ';">' .
      wp_kses($inner_html, $allowed_barilo_html) .
      '</div>';
}

add_action('wp_body_open', 'barilo_display_top_bar');

// barilo settings page
function barilo_add_settings_page()
{
   add_options_page(
      __('Barilo Light Settings', 'barilo-light-top-bar-message'),
      __('Barilo Light', 'barilo-light-top-bar-message'),
      'manage_options',
      'barilo-light-top-bar-message-settings',
      'barilo_render_settings_page',
   );
}
add_action('admin_menu', 'barilo_add_settings_page');

//render settings page content
function barilo_render_settings_page()
{
   echo '<div class="wrap">';
   echo '<h1> <strong>' . esc_html__('⚙️ Barilo Light Settings', 'barilo-light-top-bar-message') . '</strong> </h1>';
   echo '<form method="post" action="options.php">';
   settings_fields('barilo_settings_group');
   do_settings_sections('barilo-light-top-bar-message-settings');
   submit_button();
   echo '<p style="margin-top:10px; color:#3c434a; font-size:13px;"> 🔒 ' . esc_html__('This plugin does not collect or store any user data.', 'barilo-light-top-bar-message') . '</p>';
   echo '<p style="margin-top:10px; color:#3c434a; font-size:13px;"> 💡 ' . esc_html__('Have an idea or suggestion?', 'barilo-light-top-bar-message') . ' <a href="mailto:wordpress@nahabagroup.com?subject=Barilo%20Light%20Suggestion">' . esc_html__('Send it here', 'barilo-light-top-bar-message') . '</a></p>';
   echo '<p style="margin-top:10px; color:#4c434; font: size 13px;"> ☕ ' . esc_html__('Enjoying Barilo Light?', 'barilo-light-top-bar-message') . ' ' . '<a href="https://buymeacoffee.com/rnahaba" target="_blank" rel="noopener noreferrer">' . esc_html__('Buy me a coffee', 'barilo-light-top-bar-message') . '</a></p>';
   echo '</form>';
   echo '</div>';
}


// register settings
function barilo_register_settings()
{
   register_setting('barilo_settings_group', 'barilo_top_message', [
      'sanitize_callback' => 'barilo_sanitize_message'
   ]);

   register_setting('barilo_settings_group', 'barilo_bg_color', [
      'sanitize_callback' => 'barilo_sanitize_bg_color'
   ]);

   register_setting('barilo_settings_group', 'barilo_text_color', [
      'sanitize_callback' => 'barilo_sanitize_text_color'
   ]);

   register_setting('barilo_settings_group', 'barilo_hide_bar', [
      'type' => 'boolean',
      'sanitize_callback' => 'barilo_sanitize_hide_bar',
      'default' => 0,
   ]);

   register_setting(
      'barilo_settings_group',
      'barilo_phone',
      [
         'type' => 'string',
         'sanitize_callback' => 'barilo_sanitize_phone',
         'default' => '',
      ]
   );

   register_setting(
      'barilo_settings_group',
      'barilo_email',
      [
         'type' => 'string',
         'sanitize_callback' => 'barilo_sanitize_email_option',
         'default' => '',
      ]
   );

   register_setting(
      'barilo_settings_group',
      'barilo_hide_after_date',
      [
         'type' => 'string',
         'sanitize_callback' => 'barilo_sanitize_hide_after_date',
         'default' => '',
      ]
   );

   add_settings_section(
      'barilo_main_section',
      '',
      'barilo_light_main_section_callback',
      'barilo-light-top-bar-message-settings'
   );

   add_settings_field(
      'barilo_top_message_field',
      __('📝 Message Text', 'barilo-light-top-bar-message'),
      'barilo_message_field_html',
      'barilo-light-top-bar-message-settings',
      'barilo_main_section'
   );

   add_settings_field(
      'barilo_bg_color_field',
      __('🎨 Background Color', 'barilo-light-top-bar-message'),
      'barilo_bg_color_field_html',
      'barilo-light-top-bar-message-settings',
      'barilo_main_section'
   );

   add_settings_field(
      'barilo_text_color_field',
      __('✍️ Text Color', 'barilo-light-top-bar-message'),
      'barilo_text_color_field_html',
      'barilo-light-top-bar-message-settings',
      'barilo_main_section'
   );

   add_settings_field(
      'barilo_phone_field',
      __('📞 Phone', 'barilo-light-top-bar-message'),
      'barilo_phone_field_html',
      'barilo-light-top-bar-message-settings',
      'barilo_main_section'
   );

   add_settings_field(
      'barilo_email_field',
      __('✉️ Email', 'barilo-light-top-bar-message'),
      'barilo_email_field_html',
      'barilo-light-top-bar-message-settings',
      'barilo_main_section'
   );

   add_settings_field(
      'barilo_hide_after_date_field',
      __('⏰ Hide after date', 'barilo-light-top-bar-message'),
      'barilo_hide_after_date_field_html',
      'barilo-light-top-bar-message-settings',
      'barilo_main_section'
   );

   add_settings_field(
      'barilo_hide_bar_field',
      __('🙈 Hide message', 'barilo-light-top-bar-message'),
      'barilo_hide_bar_field_callback',
      'barilo-light-top-bar-message-settings',
      'barilo_main_section'
   );
}

add_action('admin_init', 'barilo_register_settings',);


function barilo_light_main_section_callback()
{
   echo '<p>' . esc_html__('💬 Thank you for installing Barilo Light! Please adjust settings for your top bar below:', 'barilo-light-top-bar-message') . '</p>';
}

// hide barilo
function barilo_hide_bar_field_callback()
{
   $hide_bar = get_option('barilo_hide_bar', 0);
?>
   <label>
      <input type="checkbox" name="barilo_hide_bar" value="1" <?php checked(1, $hide_bar); ?> />
      <?php esc_html_e('Temporarily hide the top bar message', 'barilo-light-top-bar-message'); ?>
   </label>
<?php
}

// display input field admin message characters length 
function barilo_message_field_html()
{
   $value = get_option('barilo_top_message', '');
?>
   <input
      type="text"
      name="barilo_top_message"
      value="<?php echo esc_attr($value); ?>"
      class="large-text"
      maxlength="100"
      placeholder="Welcome! You can use links like &lt;a href=&quot;https://example.com&quot;&gt;Click here&lt;/a&gt;" />
<?php
}


// color values fallback functions sanitize
function barilo_sanitize_bg_color($value)
{
   $hex = sanitize_hex_color($value);
   return $hex ? $hex : '#f2f2f2';
}

function barilo_sanitize_text_color($value)
{
   $hex = sanitize_hex_color($value);
   return $hex ? $hex : '#000000';
}


// sanitize message
function barilo_sanitize_message($value)
{
   $allowed_tags = [
      'a'      => [
         'href'  => true,
         'title' => true,
         'target' => true,
         'rel'   => true,
      ],
      'strong' => [],
      'em'     => [],
      'br'     => [],
   ];

   return wp_kses($value, $allowed_tags);
}


function barilo_sanitize_hide_bar($value)
{
   return $value ? 1 : 0;
}

function barilo_sanitize_phone($value)
{
   $value = sanitize_text_field($value);
   $value = preg_replace('/[^0-9+\s()\-]/', '', $value);
   return $value;
}

function barilo_sanitize_email_option($value)
{
   $value = sanitize_email($value);
   return $value ? $value : '';
}

function barilo_sanitize_hide_after_date($value)
{
   $value = sanitize_text_field($value);
   if ($value === '') {
      return '';
   }
   $dt = DateTime::createFromFormat('Y-m-d', $value);
   if (!$dt) {
      return '';
   }
   return $dt->format('Y-m-d');
}


// background color function
function barilo_bg_color_field_html()
{
   $value = get_option('barilo_bg_color', 'lightgray');

   // If the saved value looks like a hex, keep it.
   if (is_string($value) && strpos($value, '#') === 0) {
      $bg_hex = sanitize_hex_color($value);
   } else {
      // Map old named colors to hex for backwards compatibility.
      $color_map = [
         'lightblue'  => '#e6f4ff',
         'lightgreen' => '#e6ffe6',
         'beige'      => '#fff5e6',
         'lavender'   => '#f3e6ff',
         'lightgray'  => '#f2f2f2',
         'darkgray'   => '#333333',
      ];
      $bg_hex = $color_map[$value] ?? '#f2f2f2';
   }

   if (!$bg_hex) {
      $bg_hex = '#f2f2f2';
   }
?>
   <input
      type="text"
      name="barilo_bg_color"
      value="<?php echo esc_attr($bg_hex); ?>"
      class="barilo-color-field"
      data-default-color="#f2f2f2" />
   <p class="description">
      <?php esc_html_e('Choose the background color for the top bar.', 'barilo-light-top-bar-message'); ?>
   </p>
<?php
}


// text color function
function barilo_text_color_field_html()
{
   $value = get_option('barilo_text_color', 'black');

   if (is_string($value) && strpos($value, '#') === 0) {
      $text_hex = sanitize_hex_color($value);
   } else {
      $color_map = [
         'black'    => '#000000',
         'darkgray' => '#333333',
         'white'    => '#ffffff',
         'navy'     => '#1a1a80',
         'teal'     => '#005f5f',
         'crimson'  => '#cc0033',
      ];
      $text_hex = $color_map[$value] ?? '#000000';
   }

   if (!$text_hex) {
      $text_hex = '#000000';
   }
?>
   <input
      type="text"
      name="barilo_text_color"
      value="<?php echo esc_attr($text_hex); ?>"
      class="barilo-color-field"
      data-default-color="#000000" />
   <p class="description">
      <?php esc_html_e('Choose the text color for the top bar.', 'barilo-light-top-bar-message'); ?>
   </p>
<?php
}

function barilo_phone_field_html()
{
   $value = get_option('barilo_phone', '');
?>
   <input
      type="text"
      name="barilo_phone"
      value="<?php echo esc_attr($value); ?>"
      class="regular-text"
      placeholder="123 456 789" />
   <p class="description">
      <?php esc_html_e('Optional. If set, this will become a tappable phone link in the top bar.', 'barilo-light-top-bar-message'); ?>
   </p>
<?php
}

function barilo_email_field_html()
{
   $value = get_option('barilo_email', '');
?>
   <input
      type="email"
      name="barilo_email"
      value="<?php echo esc_attr($value); ?>"
      class="regular-text"
      placeholder="hello@example.com" />
   <p class="description">
      <?php esc_html_e('Optional. If set, this will become a clickable email link in the top bar.', 'barilo-light-top-bar-message'); ?>
   </p>
<?php
}

function barilo_hide_after_date_field_html()
{
   $value = get_option('barilo_hide_after_date', '');
?>
   <input
      type="date"
      name="barilo_hide_after_date"
      value="<?php echo esc_attr($value); ?>" />
   <p class="description">
      <?php esc_html_e('Optional. If set, the bar will stop showing after this date. Leave empty to show it indefinitely.', 'barilo-light-top-bar-message'); ?>
   </p>
<?php
}
