<?php
    /**
     * Tax Calculator Shortcode Class
     *
     * @package BangladeshTaxCalculator
     */

    if ( ! defined( 'ABSPATH' ) ) {
        exit; // Exit if accessed directly.
    }

    class BDIncTax_Calculator_Shortcode {

        /**
         * Constructor
         */
        public function __construct() {
            add_shortcode( 'bangladesh_income_tax_calculator', array( $this, 'bdinctax_render_shortcode' ) );
            add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_frontend_assets' ) );
            add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
        }

        /**
         * Render the tax calculator shortcode
         *
         * @param array $atts Shortcode attributes
         * @return string HTML output
         */
        public function bdinctax_render_shortcode( $atts ) {
            // Parse shortcode attributes
            $atts = shortcode_atts( array(
                'theme' => 'default',
                'title' => ''
            ), $atts, 'bangladesh_income_tax_calculator' );

            // Sanitize attributes
            $theme          = sanitize_text_field( $atts[ 'theme' ] );
            $title          = sanitize_text_field( $atts[ 'title' ] );
            $allowed_themes = array( 'default', 'dark', 'light' );
            if ( ! in_array( $theme, $allowed_themes ) ) {
                $theme = 'default';
            }

            // Enqueue assets
            $this->enqueue_shortcode_assets();

            // Generate unique ID for this instance
            $instance_id = 'tax-calc-' . uniqid();

            // Build the HTML
            ob_start();
        ?>
        <div class="bd-tax-calculator-wrapper tax-calc-<?php echo esc_attr( $theme ); ?>" id="<?php echo esc_attr( $instance_id ); ?>">
            <?php
                $allowed_html = $this->kses_allowed_html();
                        echo wp_kses( $this->get_form_html( $title ), $allowed_html );
                        echo wp_kses( $this->get_result_html(), $allowed_html );
                    ?>
        </div>
        <?php
            return ob_get_clean();
                }

                /**
                 * Get the form HTML
                 *
                 * @param string $title The title to display
                 * @return string Form HTML
                 */
                private function get_form_html( $title = 'Tax Calculator' ) {
                    ob_start();
                ?>
        <form class="bd-tax-form" method="post">
            <?php if ( ! empty( $title ) ): ?>
            <div class="bd-tax-form-header">
                <h3><?php echo esc_html( $title ); ?></h3>
            </div>
            <?php endif; ?>

            <div class="bd-tax-form-body">
                <div class="bd-tax-field">
                    <label for="taxYear">
                        <?php esc_html_e( 'Tax Year', 'bangladesh-income-tax-calculator' ); ?>
                        <span class="required">*</span>
                    </label>
                    <select name="taxYear" id="taxYear" required>
                        <option value=""><?php esc_html_e( '-- Select Tax Year --', 'bangladesh-income-tax-calculator' ); ?></option>
                        <option value="2024-25"><?php esc_html_e( '2024-25 or Before', 'bangladesh-income-tax-calculator' ); ?></option>
                        <option value="2025-26"><?php esc_html_e( '2025-26 or After', 'bangladesh-income-tax-calculator' ); ?></option>
                    </select>
                    <div class="bd-tax-error" style="display: none;"></div>
                </div>

                <div class="bd-tax-field">
                    <label for="area">
                        <?php esc_html_e( 'Area', 'bangladesh-income-tax-calculator' ); ?>
                        <span class="required">*</span>
                    </label>
                    <select name="area" id="area" required>
                        <option value=""><?php esc_html_e( '-- Select Area --', 'bangladesh-income-tax-calculator' ); ?></option>
                        <option value="dhaka_chattogram"><?php esc_html_e( 'Dhaka/Chattagram City Corporation', 'bangladesh-income-tax-calculator' ); ?></option>
                        <option value="other_city"><?php esc_html_e( 'Other City Corporation', 'bangladesh-income-tax-calculator' ); ?></option>
                        <option value="outside_city"><?php esc_html_e( 'Out of the City Corporation', 'bangladesh-income-tax-calculator' ); ?></option>
                    </select>
                    <div class="bd-tax-error" style="display: none;"></div>
                </div>

                <div class="bd-tax-field">
                    <label for="gender">
                        <?php esc_html_e( 'Gender', 'bangladesh-income-tax-calculator' ); ?>
                        <span class="required">*</span>
                    </label>
                    <select name="gender" id="gender" required>
                        <option value=""><?php esc_html_e( '-- Select --', 'bangladesh-income-tax-calculator' ); ?></option>
                        <option value="male"><?php esc_html_e( 'Male', 'bangladesh-income-tax-calculator' ); ?></option>
                        <option value="female"><?php esc_html_e( 'Female', 'bangladesh-income-tax-calculator' ); ?></option>
                        <option value="third_gender"><?php esc_html_e( 'Third Gender', 'bangladesh-income-tax-calculator' ); ?></option>
                    </select>
                    <div class="bd-tax-error" style="display: none;"></div>
                </div>

                <div class="bd-tax-checkboxes">
                    <label class="bd-tax-checkbox">
                        <input type="checkbox" name="above65" />
                        <?php esc_html_e( 'Above 65 Age', 'bangladesh-income-tax-calculator' ); ?>
                    </label>
                    <label class="bd-tax-checkbox">
                        <input type="checkbox" name="freedomFighter" />
                        <?php esc_html_e( 'Is Freedom Fighter', 'bangladesh-income-tax-calculator' ); ?>
                    </label>
                    <label class="bd-tax-checkbox">
                        <input type="checkbox" name="disabled" />
                        <?php esc_html_e( 'Is Disabled', 'bangladesh-income-tax-calculator' ); ?>
                    </label>
                </div>

                <div class="bd-tax-field">
                    <label for="grossIncome">
                        <?php esc_html_e( 'Gross Income (Yearly, Taka)', 'bangladesh-income-tax-calculator' ); ?>
                        <span class="required">*</span>
                    </label>
                    <input type="number" name="grossIncome" id="grossIncome" min="0" step="0.01" required />
                    <div class="bd-tax-error" style="display: none;"></div>
                </div>

                <div class="bd-tax-field">
                    <label for="totalInvestment">
                        <?php esc_html_e( 'Total Investment (Yearly, Taka)', 'bangladesh-income-tax-calculator' ); ?>
                        <span class="required">*</span>
                    </label>
                    <input type="number" name="totalInvestment" id="totalInvestment" min="0" step="0.01" required />
                    <div class="bd-tax-error" style="display: none;"></div>
                </div>

                <div class="bd-tax-field">
                    <label for="paidTax">
                        <?php esc_html_e( 'Paid Tax (Optional, Taka)', 'bangladesh-income-tax-calculator' ); ?>
                    </label>
                    <input type="number" name="paidTax" id="paidTax" min="0" step="0.01" placeholder="0" />
                    <div class="bd-tax-error" style="display: none;"></div>
                </div>
            </div>

            <div class="bd-tax-form-footer">
                <button type="submit" class="bd-tax-submit-btn">
                    <?php esc_html_e( 'Calculate Tax', 'bangladesh-income-tax-calculator' ); ?>
                </button>
            </div>
        </form>
        <?php
            return ob_get_clean();
                }

                /**
                 * Get the result HTML template
                 *
                 * @return string Result HTML
                 */
                private function get_result_html() {
                    ob_start();
                ?>
        <div class="bd-tax-result" style="display: none;">
            <div class="bd-tax-result-header">
                <h4><?php esc_html_e( 'Tax Calculation Result', 'bangladesh-income-tax-calculator' ); ?></h4>
            </div>

            <div class="bd-tax-result-body">
                <div class="bd-tax-summary-section">
                    <table class="bd-tax-summary-table">
                        <tbody>
                            <tr>
                                <th><?php esc_html_e( 'Gross Income', 'bangladesh-income-tax-calculator' ); ?></th>
                                <td class="gross-income-display">৳0</td>
                            </tr>
                            <tr>
                                <th><?php esc_html_e( 'Deduction (Gross Income/3 or ৳4,50,000)', 'bangladesh-income-tax-calculator' ); ?></th>
                                <td class="deduction-display">৳0</td>
                            </tr>
                            <tr>
                                <th><?php esc_html_e( 'Taxable Income (After Deduction)', 'bangladesh-income-tax-calculator' ); ?></th>
                                <td class="taxable-income-display">৳0</td>
                            </tr>
                        </tbody>
                    </table>
                </div>

                <div class="bd-tax-slab-section">
                    <h5><?php esc_html_e( 'Tax Calculation (Slab by Slab)', 'bangladesh-income-tax-calculator' ); ?></h5>
                    <table class="bd-tax-slab-table">
                        <thead>
                            <tr>
                                <th><?php esc_html_e( 'Slab Range (৳)', 'bangladesh-income-tax-calculator' ); ?></th>
                                <th><?php esc_html_e( 'Rate (%)', 'bangladesh-income-tax-calculator' ); ?></th>
                                <th><?php esc_html_e( 'Taxable Amount', 'bangladesh-income-tax-calculator' ); ?></th>
                                <th><?php esc_html_e( 'Tax', 'bangladesh-income-tax-calculator' ); ?></th>
                            </tr>
                        </thead>
                        <tbody class="slab-breakdown">
                            <!-- Dynamic content will be inserted here -->
                        </tbody>
                    </table>
                </div>

                <div class="bd-tax-final-section">
                    <table class="bd-tax-final-table">
                        <tbody>
                            <tr>
                                <th><?php esc_html_e( 'Total Tax (Before Rebate)', 'bangladesh-income-tax-calculator' ); ?></th>
                                <td class="total-tax-display">৳0</td>
                            </tr>
                            <tr>
                                <th><?php esc_html_e( 'Allowable Investment Used', 'bangladesh-income-tax-calculator' ); ?></th>
                                <td class="allowable-investment-display">৳0</td>
                            </tr>
                            <tr>
                                <th><?php esc_html_e( 'Rebate A (3% of taxable income)', 'bangladesh-income-tax-calculator' ); ?></th>
                                <td class="rebate-a-display">৳0</td>
                            </tr>
                            <tr>
                                <th><?php esc_html_e( 'Rebate B (15% of applicable investment)', 'bangladesh-income-tax-calculator' ); ?></th>
                                <td class="rebate-b-display">৳0</td>
                            </tr>
                            <tr>
                                <th><?php esc_html_e( 'Applied Rebate', 'bangladesh-income-tax-calculator' ); ?></th>
                                <td class="applied-rebate-display">৳0</td>
                            </tr>
                            <tr class="final-tax-row">
                                <th><?php esc_html_e( 'Final Tax After Rebate', 'bangladesh-income-tax-calculator' ); ?></th>
                                <td class="final-tax-display">৳0</td>
                            </tr>
                            <tr class="minimum-tax-row" style="display: none;">
                                <th><?php esc_html_e( 'Minimum Tax Applied', 'bangladesh-income-tax-calculator' ); ?></th>
                                <td class="minimum-tax-display">৳0</td>
                            </tr>
                            <tr class="minimum-tax-message-row" style="display: none;">
                                <td colspan="2" class="minimum-tax-message" style="font-style: italic; color: #666; text-align: center; padding: 10px;">
                                    <!-- Minimum tax message will be inserted here -->
                                </td>
                            </tr>
                            <tr>
                                <th><?php esc_html_e( 'Paid Tax', 'bangladesh-income-tax-calculator' ); ?></th>
                                <td class="paid-tax-display">৳0</td>
                            </tr>
                            <tr class="payable-tax-row">
                                <th><?php esc_html_e( 'Payable Tax', 'bangladesh-income-tax-calculator' ); ?></th>
                                <td class="payable-tax-display">৳0</td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        <?php
            return ob_get_clean();
                }

                /**
                 * Enqueue shortcode-specific assets
                 */
                private function enqueue_shortcode_assets() {
                    // This will be called when shortcode is rendered
                    wp_enqueue_script( 'bd-tax-calculator-frontend' );
                    wp_enqueue_style( 'bd-tax-calculator-frontend' );
                }

                /**
                 * Enqueue frontend assets
                 */
                public function enqueue_frontend_assets() {
                    // Only register assets, enqueue when shortcode is used
                    wp_register_script(
                        'bd-tax-calculator-frontend',
                        BDINCTAX_CALC_PLUGIN_URL . 'assets/js/frontend.js',
                        array(),
                        BDINCTAX_CALC_VERSION,
                        true
                    );

                    wp_register_style(
                        'bd-tax-calculator-frontend',
                        BDINCTAX_CALC_PLUGIN_URL . 'assets/css/frontend.css',
                        array(),
                        BDINCTAX_CALC_VERSION
                    );
                }

                /**
                 * Enqueue admin assets
                 */
                public function enqueue_admin_assets() {
                    // Register for admin use (Gutenberg editor)
                    wp_register_script(
                        'bd-tax-calculator-frontend',
                        BDINCTAX_CALC_PLUGIN_URL . 'assets/js/frontend.js',
                        array(),
                        BDINCTAX_CALC_VERSION,
                        true
                    );

                    wp_register_style(
                        'bd-tax-calculator-frontend',
                        BDINCTAX_CALC_PLUGIN_URL . 'assets/css/frontend.css',
                        array(),
                        BDINCTAX_CALC_VERSION
                    );
                }

                /**
                 * Custom function for KSES HTML List
                 * Allows all necessary HTML elements for the tax calculator form
                 *
                 * @return array Allowed HTML Array
                 */
                private function kses_allowed_html() {
                    return array(
                        // Basic HTML elements
                        'div' => array(
                            'class' => array(),
                            'id' => array(),
                            'style' => array()
                        ),
                        'span' => array(
                            'class' => array(),
                            'id' => array()
                        ),
                        'p' => array(
                            'class' => array()
                        ),
                        'h1' => array(
                            'class' => array()
                        ),
                        'h2' => array(
                            'class' => array()
                        ),
                        'h3' => array(
                            'class' => array()
                        ),
                        'h4' => array(
                            'class' => array()
                        ),
                        'h5' => array(
                            'class' => array()
                        ),
                        'h6' => array(
                            'class' => array()
                        ),
                        'strong' => array(),
                        'em' => array(),
                        'br' => array(),
                        'hr' => array(),

                        // Form elements
                        'form' => array(
                            'class' => array(),
                            'id' => array(),
                            'method' => array(),
                            'action' => array()
                        ),
                        'input' => array(
                            'type' => array(),
                            'name' => array(),
                            'id' => array(),
                            'class' => array(),
                            'value' => array(),
                            'placeholder' => array(),
                            'required' => array(),
                            'min' => array(),
                            'max' => array(),
                            'step' => array(),
                            'checked' => array(),
                            'disabled' => array(),
                            'readonly' => array()
                        ),
                        'select' => array(
                            'name' => array(),
                            'id' => array(),
                            'class' => array(),
                            'required' => array(),
                            'disabled' => array()
                        ),
                        'option' => array(
                            'value' => array(),
                            'selected' => array()
                        ),
                        'textarea' => array(
                            'name' => array(),
                            'id' => array(),
                            'class' => array(),
                            'rows' => array(),
                            'cols' => array(),
                            'placeholder' => array(),
                            'required' => array(),
                            'disabled' => array(),
                            'readonly' => array()
                        ),
                        'label' => array(
                            'for' => array(),
                            'class' => array()
                        ),
                        'button' => array(
                            'type' => array(),
                            'class' => array(),
                            'id' => array(),
                            'disabled' => array()
                        ),
                        'fieldset' => array(
                            'class' => array()
                        ),
                        'legend' => array(
                            'class' => array()
                        ),

                        // List elements
                        'ul' => array(
                            'class' => array()
                        ),
                        'ol' => array(
                            'class' => array()
                        ),
                        'li' => array(
                            'class' => array()
                        ),

                        // Table elements (for result display)
                        'table' => array(
                            'class' => array()
                        ),
                        'thead' => array(
                            'class' => array()
                        ),
                        'tbody' => array(
                            'class' => array()
                        ),
                        'tfoot' => array(),
                        'tr' => array(
                            'class' => array()
                        ),
                        'th' => array(
                            'class' => array(),
                            'colspan' => array(),
                            'rowspan' => array()
                        ),
                        'td' => array(
                            'class' => array(),
                            'colspan' => array(),
                            'rowspan' => array()
                        ),

                        // Links
                        'a' => array(
                            'href' => array(),
                            'class' => array(),
                            'id' => array(),
                            'target' => array(),
                            'rel' => array()
                        ),

                        // Images
                        'img' => array(
                            'src' => array(),
                            'alt' => array(),
                            'class' => array(),
                            'width' => array(),
                            'height' => array()
                        )
                    );
                }
        }
