<?php

namespace BisnNotify\Hooks;

if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

/**
 * Admin class
 */
class Admin {
	public function __construct() {
		add_action('admin_menu', [$this, 'back_in_stock_notify_add_admin_menu']);
		add_action('woocommerce_product_set_stock_status', [$this, 'clear_notifications_on_instock'], 10, 2);
	}

	/**
	 * Add the admin menu
	 */
	public function back_in_stock_notify_add_admin_menu() {
		add_submenu_page(
			'woocommerce',
			__('Stock Notify', 'back-in-stock-notify'),
			__('Stock Notify', 'back-in-stock-notify'),
			'manage_woocommerce',
			'stock-notify',
			[$this, 'back_in_stock_notify_render_admin_page']
		);
	}

	/**
	 * Render the admin page
	 */
	public function back_in_stock_notify_render_admin_page() {
		$products = wc_get_products(['status' => 'publish', 'limit' => -1]);
		$all_notifications = []; // Array to collect all notifications for export
	
		?>
		<div class="wrap">
			<h1><?php esc_html_e('Back In Stock Notifications', 'back-in-stock-notify'); ?></h1>
			<table class="wp-list-table widefat striped">
				<thead>
					<tr>
						<th><?php esc_html_e('Product', 'back-in-stock-notify'); ?></th>
						<th><?php esc_html_e('Emails', 'back-in-stock-notify'); ?></th>
					</tr>
				</thead>
				<tbody>
					<?php 
					$found = false; // Track if any product has notifications
					foreach ($products as $product): 
						$notifications = get_post_meta($product->get_id(), '_back_in_stock_notify_notifications', true);
						$notifications = is_array($notifications) ? $notifications : [];
	
						if (!empty($notifications)): 
							$found = true;
							$all_notifications = array_merge($all_notifications, $notifications); // Add to export list
							?>
							<tr>
								<td><?php echo esc_html($product->get_name()); ?></td>
								<td><?php echo esc_html(implode(', ', $notifications)); ?></td>
							</tr>
						<?php 
						endif;
					endforeach;
	
					if (!$found): // If no products have notifications, show a message
					?>
						<tr>
							<td colspan="2"><?php esc_html_e('No products with notifications found.', 'back-in-stock-notify'); ?></td>
						</tr>
					<?php endif; ?>
				</tbody>
			</table>
			<p class="notice-email" style="font-weight:500"> <?php esc_html_e('All the Emails will cleared when the outstock product is set again in stock.', 'back-in-stock-notify') ?></p>
			<form method="post" style="margin-top: 10px">
				<?php wp_nonce_field('back_in_stock_notify_export_emails_action', 'back_in_stock_notify_export_emails_nonce'); ?>
				<input type="submit" name="export_emails" class="button-primary" value="<?php esc_attr_e('Export Emails', 'back-in-stock-notify'); ?>">
			</form>
		</div>
		<?php
	
		// Handle the export action
		if (isset($_POST['export_emails'])) {
			if (isset($_POST['back_in_stock_notify_export_emails_nonce'])) {
				$nonce = sanitize_text_field(wp_unslash($_POST['back_in_stock_notify_export_emails_nonce']));
				if (!wp_verify_nonce($nonce, 'back_in_stock_notify_export_emails_action')) {
					wp_die(esc_html__('Security check failed. Please try again.', 'back-in-stock-notify'));
				}
			} else {
				wp_die(esc_html__('Security check failed. Please try again.', 'back-in-stock-notify'));
			}
			$this->export_emails_to_csv($all_notifications);
		}
	}
	
	/**
	 * Export Emails to CSV
	 *
	 * @param array $emails Array of emails to export.
	 */
	private function export_emails_to_csv($emails) {
		if (empty($emails)) {
			wp_die(esc_html__('No emails to export.', 'back-in-stock-notify'));
		}
	
		// Clear output buffer to prevent unexpected output in the CSV
		if (ob_get_contents()) {
			ob_end_clean();
		}
	
		// Set CSV headers
		header('Content-Type: text/csv; charset=utf-8');
		header('Content-Disposition: attachment; filename=back-in-stock-emails.csv');
	
		// Open output stream
		$output = fopen('php://output', 'w');
	
		// Write the column headers
		fputcsv($output, ['Email']);
	
		// Write the email data
		foreach (array_unique($emails) as $email) {
			fputcsv($output, [$email]);
		}
	
		// Using direct file operation for browser output as WP_Filesystem is not applicable for streaming output.
		fclose($output); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose

		exit; // Stop further execution
	}

	/**
	 * Clear notifications when the product goes back in stock.
	 *
	 * @param int    $product_id The ID of the product.
	 * @param string $stock_status The new stock status.
	 */
	public function clear_notifications_on_instock($product_id, $stock_status) {
		if ($stock_status === 'instock') {
			delete_post_meta($product_id, '_back_in_stock_notify_notifications');
		}
	}
}