<?php
/**
 * Plugin Name: Back In Stock Notify
 * Description: Notify customers when out-of-stock products are back in stock. Collect email addresses and manage them in the admin panel.
 * Requires at least: 6.1
 * Requires PHP:      7.0
 * Version:           1.1.7
 * Author:            Iqbal Hossain
 * License: GPL-3.0-or-later
 * License URI: https://www.gnu.org/licenses/gpl-3.0.html
 * Text Domain: 	  back-in-stock-notify
 * Domain Path:       /languages
 */

if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

/**
 * Final class for the plugin
 */

final class Back_In_Stock_Notify {
	/**
	 * Plugin version
	*
	* @var string
	*/
	const VERSION = '1.1.7';

	/**
	 * Plugin instance.
	 *
	 * @var Back_In_Stock_Notify|null
	 */
	private static $instance = null;

	/**
	 * Get the singleton instance of the plugin.
	 *
	 * @return Back_In_Stock_Notify
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Check if WooCommerce is active.
	 *
	 * @return bool True if WooCommerce is active, false otherwise.
	 */
	private function is_woocommerce_active() {
		return in_array( 'woocommerce/woocommerce.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) || ( is_multisite() && array_key_exists( 'woocommerce/woocommerce.php', get_site_option( 'active_sitewide_plugins', [] ) ) );
	}

	/**
	 * Display an admin notice if WooCommerce is not active.
	 */
	public function check_woocommerce_dependency() {
		if ( ! $this->is_woocommerce_active() ) {
			add_action( 'admin_notices', [ $this, 'woocommerce_missing_notice' ] );
			deactivate_plugins( plugin_basename( __FILE__ ) );
		}
	}

	/**
	 * Admin notice for missing WooCommerce.
	 */
	public function woocommerce_missing_notice() {
		$install_url = admin_url( 'plugin-install.php?s=woocommerce&tab=search&type=term' );
		?>
		<div class="notice notice-error">
			<p>
				<?php esc_html_e( 'Back In Stock Notify requires WooCommerce to be installed and active.', 'back-in-stock-notify' ); ?>
			</p>
			<p>
				<a href="<?php echo esc_url( $install_url ); ?>" class="button button-primary"><?php esc_html_e( 'Install WooCommerce', 'back-in-stock-notify' ); ?></a>
			</p>
		</div>
		<?php
	}

	/**
	 * Plugin constructor
	*/
	private function __construct() {
		$this->define_constants();
		$this->includes();
		$this->register_activation_hooks();

		// Check WooCommerce dependency
		add_action( 'admin_init', [ $this, 'check_woocommerce_dependency' ] );

		// Initialize the plugin after all plugins are loaded.
		add_action( 'plugins_loaded', [ $this, 'init_plugin' ] );
		// Load the plugin text domain
		//add_action( 'init', [ $this, 'back_in_stock_notify_load_textdomain' ] );
		// Add custom row meta for plugin description.
		add_filter( 'plugin_row_meta', [ $this, 'back_in_stock_notify_plugin_row_meta' ], 10, 2 );
		// Enqueue assets for the plugin
		add_action('wp_enqueue_scripts', [$this, 'back_in_stock_notify_enqueue_assets']);
		// Hook the notification form into the single product page
		add_action('woocommerce_single_product_summary', [$this, 'back_in_stock_notify_display_notify_form'], 35);
	}

	/**
	 * Define the required constants
	*/
	private function define_constants() {
		define( 'BISN_VERSION', self::VERSION );
		define( 'BISN_FILE', plugin_dir_url( __FILE__ ) );
		define( 'BISN_DIR', plugin_dir_path( __FILE__ ) );
	}

	/**
	 * Register activation hook.
	 */
	private function register_activation_hooks() {
		register_activation_hook( __FILE__, [ $this, 'activate' ] );
	}

	/**
	 * Include required files.
	 */
	private function includes() {
		require_once BISN_DIR . '/vendor/autoload.php';
	}

	/**
	 * Initialize the plugin functionality.
	 */
	public function init_plugin() {
		/**
		 * Fires before the initialization of the Back_In_Stock_Notify plugin.
		 *
		 * This action hook allows developers to perform additional tasks before the Back_In_Stock_Notify plugin has been initialized.
		 * @since 1.1.1
		 */
		do_action( 'back_in_stock_notify/before_init' );


		/**
		 * Register hooks for the plugin.
		 *
		 * @since 1.1.1
		 */
		new BisnNotify\Hooks\Admin();

		/**
		 * Register hooks for the plugin.
		 *
		 * @since 1.1.1
		 */
		new BisnNotify\Hooks\Ajax();

		/**
		 * Fires after the initialization of the Back_In_Stock_Notify plugin.
		 *
		 * This action hook allows developers to perform additional tasks after the Back_In_Stock_Notify plugin has been initialized.
		 * @since 1.0.0
		 */
		do_action( 'back_in_stock_notify/after_init' );
	}

	/**
	 * Perform actions upon plugin activation.
	 */
	public function activate() {

		if ( ! $this->is_woocommerce_active() ) {
			deactivate_plugins( plugin_basename( __FILE__ ) );
			wp_die(
				esc_html__( 'Back In Stock Notify requires WooCommerce to be installed and active.', 'back-in-stock-notify' ) .
				'<br><a href="' . esc_url( admin_url( 'plugin-install.php?s=woocommerce&tab=search&type=term' ) ) . '">' . esc_html__( 'Install WooCommerce', 'back-in-stock-notify' ) . '</a>',
				esc_html__( 'Plugin Activation Error', 'back-in-stock-notify' ),
				[ 'back_link' => true ]
			);
		}

		$installed = get_option( 'back_in_stock_notify_installed' );

		if ( ! $installed ) {
			update_option( 'back_in_stock_notify_installed', time() );
		}

		update_option( 'back_in_stock_notify_version', BISN_VERSION );
	}

	/**
	 * Add custom row meta to the plugin description in the Plugins page.
	 *
	 * @param array  $plugin_meta Meta information about the plugin.
	 * @param string $plugin_file Plugin file path.
	 * @return array Modified plugin meta.
	 */
	public function back_in_stock_notify_plugin_row_meta( $plugin_meta, $plugin_file ) {
		if ( plugin_basename( __FILE__ ) === $plugin_file ) {
			$row_meta = [
				'video' => '<a href="https://youtu.be/-s0bqT6sz3M" aria-label="' . esc_attr__( 'View Video Tutorials', 'back-in-stock-notify' ) . '" target="_blank">' . esc_html__( 'Video Tutorials', 'back-in-stock-notify' ) . '</a>',
			];

			// Merge the custom meta with existing plugin meta.
			$plugin_meta = array_merge( $plugin_meta, $row_meta );
		}

		return $plugin_meta;
	}

	/**
	 * Enqueue assets for the plugin.
	 */
	public function back_in_stock_notify_enqueue_assets() {
		// Check if we are on a single product page or other relevant pages
		if (is_product()) {
			global $product;
			$product_id = $product ? $product->get_id() : 0;

			$back_in_stock_notify_version = BISN_VERSION;
				wp_enqueue_script('back-in-stock-notify-form', BISN_FILE . 'assets/js/notify-form.js', array(), '1.0', true);
				wp_enqueue_style('back-in-stock-notify-form', BISN_FILE . 'assets/css/notify-form.css', [], '1.0');
				wp_localize_script('back-in-stock-notify-form', 'BISN_Ajax', [
					'ajax_url' => admin_url('admin-ajax.php'),
					'nonce'    => wp_create_nonce('back_in_stock_notify_nonce'),
					'product_id' => $product_id, // Pass the product ID here
				]);
		}
	}

	/**
	 * Display the notification form on the single product page.
	 */
	public function back_in_stock_notify_display_notify_form() {
		global $product;

		// Check if the product is out of stock
		if ($product->is_in_stock()) {
			return;
		}

		// Display the notification form
		?>
		<div class="back-in-stock-notify-form">
			<p><?php esc_html_e('This product is out of stock. Enter your email to get notified when it’s back in stock.', 'back-in-stock-notify'); ?></p>
			<input type="email" id="bisn-email" placeholder="<?php esc_attr_e('Enter your email', 'back-in-stock-notify'); ?>" required>
			<button id="bisn-submit"><?php esc_html_e('Notify Me', 'back-in-stock-notify'); ?></button>
			<div id="bisn-message"></div>
		</div>
		<?php
	}

	/**
	 * Load the plugin text domain for translation.
	 */
	// public function back_in_stock_notify_load_textdomain() {
	// 	/**
	// 	 * Loads the plugin text domain for the Back_In_Stock_Notify.
	// 	 *
	// 	 * This function is responsible for loading the translation files for the plugin.
	// 	 * It sets the text domain to 'back-in-stock-notify' and specifies the directory
	// 	 * where the translation files are located.
	// 	 *
	// 	 * @param string $domain   The text domain for the plugin.
	// 	 * @param bool   $network  Whether the plugin is network activated.
	// 	 * @param string $directory The directory where the translation files are located.
	// 	 * @return bool True on success, false on failure.
	// 	 * @since 1.0.0
	// 	 */
	// 	load_plugin_textdomain( 'back-in-stock-notify', false, BISN_DIR . 'languages/' );
	// }
}

/**
 * Initialize the main plugin instance.
 *
 * @return Back_In_Stock_Notify
 */
function back_in_stock_notify() {
	return Back_In_Stock_Notify::instance();
}

// Kick off the plugin.
back_in_stock_notify();

