<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * تتبع زيارات البوتات وتخزينها في قاعدة البيانات فقط
 */
class BT_Bots_Tracker_Tracker {

    /**
     * ربط الهوكات الأساسية
     */
    public function init_hooks() {

        // التقاط كود الاستجابة HTTP قبل الإرسال (في الواجهة فقط)
        if ( ! is_admin() ) {
            add_filter( 'status_header', array( $this, 'capture_status_code' ), 10, 4 );
        }

        // تتبع الزيارات في الواجهة فقط
        add_action( 'shutdown', array( $this, 'maybe_track_bot_visit' ) );
    }

    /**
     * التقاط كود الاستجابة (200, 404, 301…)
     *
     * @param string $status_header النص الكامل للهيدر
     * @param int    $code          كود الاستجابة
     * @param string $description   الوصف النصي
     * @param string $protocol      البروتوكول
     *
     * @return string
     */
    public function capture_status_code( $status_header, $code, $description, $protocol ) {

        // نخزّنه في متغير جلوبال مؤقت (مش في قاعدة البيانات مباشرة)
        $GLOBALS['bt_tracker_status_code'] = (int) $code;

        return $status_header; // نرجّع الهيدر كما هو
    }

    /**
     * التأكد إن الزيارة من بوت، ثم تسجيلها
     */
    public function maybe_track_bot_visit() {

        // لا نسجّل داخل لوحة التحكم أو في طلبات AJAX/REST
        if ( is_admin() || wp_doing_ajax() || ( defined( 'REST_REQUEST' ) && REST_REQUEST ) ) {
            return;
        }

        // User Agent (يُستخدم فقط للتحقق من البوت واستخراج الاسم — لا يتم تخزينه كما هو)
        $ua_raw = isset( $_SERVER['HTTP_USER_AGENT'] ) ? wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) : '';
        $ua     = sanitize_text_field( $ua_raw );

        if ( '' === $ua ) {
            return;
        }

        // استخراج اسم البوت من الـ UA (بنظام سيجنشرز + فحص عام)
        $bot_name = $this->extract_bot_name( $ua );

        // لو مش بوت مؤكد → لا نسجل الزيارة
        if ( '' === $bot_name ) {
            return;
        }

        // الرابط الذي زاره البوت (Path فقط مع دعم العربي)
        $url_visited = $this->get_current_url();

        // IP مع تفضيل IPv4 ثم IPv6
        $ip_address = $this->get_client_ip_prefer_ipv4();

        // وقت الزيارة
        $visit_time = current_time( 'mysql' );

        // جلب كود الاستجابة لو موجود
        $status_code = isset( $GLOBALS['bt_tracker_status_code'] )
            ? (int) $GLOBALS['bt_tracker_status_code']
            : null;

        // تحضير البيانات لإرسالها للداتا بيز
        // (لا نخزن User Agent نفسه، فقط اسم البوت + IP + الرابط + الوقت + كود الاستجابة)
        $visit_data = array(
            'bot_name'    => $bot_name,
            'url_visited' => $url_visited,
            'visit_time'  => $visit_time,
            'status_code' => $status_code,
            'ip_address'  => $ip_address,
        );

        // تخزين في قاعدة البيانات فقط
        BT_Bots_Tracker_Database::log_visit( $visit_data );
    }

    /**
     * محاولة استخراج اسم البوت من اليوزر أجنت بشكل أدق
     *
     * - أولاً: سيجنشرز ثابتة لبوتات معروفة (Googlebot, Bingbot, Ahrefs, Semrush, Screaming Frog, Moz, …)
     * - ثانيًا: فحص عام لأي UA يحتوي على bot/crawler/spider… إلخ
     * - لو لم ينطبق أي شيء → نرجع سلسلة فارغة (مش بوت)
     *
     * @param string $ua
     * @return string اسم البوت أو "" لو مش بوت
     */
    protected function extract_bot_name( $ua ) {
        $ua = trim( (string) $ua );
        if ( '' === $ua ) {
            return '';
        }

        $ua_lc = strtolower( $ua );

        // 1) سيجنشرز ثابتة لأهم محركات البحث + أدوات السيو + البوتات الشهيرة
        $signatures = array(
            // محركات بحث رئيسية
            'Googlebot'                           => array( 'googlebot' ),
            'Google AdsBot'                       => array( 'adsbot-google' ),
            'Googlebot-Image'                     => array( 'googlebot-image' ),
            'Google Other'                        => array( 'googleother' ),
            'Bingbot'                             => array( 'bingbot' ),
            'BingPreview'                         => array( 'bingpreview' ),
            'DuckDuckBot'                         => array( 'duckduckbot' ),
            'Yahoo Slurp'                         => array( 'slurp' ),
            'YandexBot'                           => array( 'yandexbot', 'yandeximages', 'yandexmobilebot' ),
            'BaiduSpider'                         => array( 'baiduspider' ),
            'Sogou Spider'                        => array( 'sogou spider', 'sogou web spider' ),
            'Applebot'                            => array( 'applebot' ),
            'PetalBot'                            => array( 'petalbot' ),

            // بوتات OpenAI / LLMs
            'GPTBot'                              => array( 'gptbot' ),
            'OAI-SearchBot'                       => array( 'oai-searchbot' ),
            'ChatGPT-User'                        => array( 'chatgpt-user' ),
            'CCBot'                               => array( 'ccbot' ),
            'ClaudeBot'                           => array( 'claudebot', 'anthropic-ai' ),

            // أدوات سيو / زواحف تحليلية
            'AhrefsBot'                           => array( 'ahrefsbot' ),
            'SemrushBot'                          => array( 'semrushbot' ),
            'Semrush'                             => array( 'semrush ' ), // مع مسافة علشان ما نمسكش أي كلمة فيها semrush جوه دومين كبير
            'Moz'                                 => array( 'rogerbot', 'moz.com', 'mozbot' ),
            'MajesticSEO'                         => array( 'mj12bot', 'majestic-12', 'majesticseo' ),
            'Screaming Frog'                      => array( 'screaming frog seo spider', 'screaming frog seo' ),
            'Sitebulb'                            => array( 'sitebulb' ),
            'SEOkicks-Robot'                      => array( 'seokicks-robot' ),
            'SerpstatBot'                         => array( 'serpstatbot' ),
            'SEMrushBot'                          => array( 'semrushbot' ),
            'SE Ranking Bot'                      => array( 'seranking', 'se ranking' ),
            'SEO PowerSuite / LinkAssistant'      => array( 'linkassistant', 'seo powersuite' ),

            // Uptime / Monitoring
            'UptimeRobot'                         => array( 'uptimerobot' ),
            'Pingdom'                             => array( 'pingdom' ),

            // منصات اجتماعية (للبريفيو)
            'Facebook Bot'                        => array( 'facebookexternalhit', 'facebookbot' ),
            'Twitter Bot'                         => array( 'twitterbot' ),
            'LinkedIn Bot'                        => array( 'linkedinbot' ),
            'Instagram Bot'                       => array( 'instagram' ),
        );

        foreach ( $signatures as $bot_name => $patterns ) {
            foreach ( $patterns as $pattern ) {
                $pattern = trim( strtolower( $pattern ) );
                if ( '' === $pattern ) {
                    continue;
                }

                if ( false !== strpos( $ua_lc, $pattern ) ) {
                    // نرجّع اسم البوت نفسه (مش الـ UA) مع حد 191 حرف
                    return mb_substr( $bot_name, 0, 191 );
                }
            }
        }

        // 2) فحص عام لأي UA يحتوي على bot/crawler/spider/crawl/preview/analyzer/seo ككلمة تقريباً
        if ( preg_match( '/\b(bot|crawler|spider|crawl|preview|analyzer|seo)\b/i', $ua ) ) {
            // نرجّع جزء من الـ UA نفسه، بحيث تقدر تراجع الأنواع لاحقاً من التقرير
            return mb_substr( $ua, 0, 191 );
        }

        // مش بوت مؤكد
        return '';
    }

    /**
     * الحصول على IP مع تفضيل IPv4 ثم IPv6
     *
     * نحاول قراءة هيدرات البروكسي (إن وجدت) ثم REMOTE_ADDR،
     * نختار أول IPv4 صالح، وإن لم يوجد نختار أول IPv6 صالح.
     *
     * @return string
     */
    protected function get_client_ip_prefer_ipv4() {

        $candidates = array();

        // بعض الاستضافات والبروكسيات تضع IP الأصلي في هذه الهيدرات
        if ( ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
            // قد تحتوي على قائمة IPs مفصولة بفواصل
            $forwarded_raw = wp_unslash( $_SERVER['HTTP_X_FORWARDED_FOR'] );
            $forwarded_ips = explode( ',', $forwarded_raw );
            foreach ( $forwarded_ips as $ip ) {
                $candidates[] = trim( $ip );
            }
        }

        if ( ! empty( $_SERVER['HTTP_CLIENT_IP'] ) ) {
            $client_ip_raw = wp_unslash( $_SERVER['HTTP_CLIENT_IP'] );
            $candidates[]  = trim( $client_ip_raw );
        }

        if ( ! empty( $_SERVER['HTTP_X_REAL_IP'] ) ) {
            $real_ip_raw  = wp_unslash( $_SERVER['HTTP_X_REAL_IP'] );
            $candidates[] = trim( $real_ip_raw );
        }

        if ( ! empty( $_SERVER['REMOTE_ADDR'] ) ) {
            $remote_ip_raw = wp_unslash( $_SERVER['REMOTE_ADDR'] );
            $candidates[]  = trim( $remote_ip_raw );
        }

        // أولاً: نبحث عن أي IPv4 صالح
        foreach ( $candidates as $ip ) {
            if ( filter_var( $ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 ) ) {
                return sanitize_text_field( $ip );
            }
        }

        // ثانيًا: لو مفيش IPv4، نبحث عن أي IPv6 صالح
        foreach ( $candidates as $ip ) {
            if ( filter_var( $ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6 ) ) {
                return sanitize_text_field( $ip );
            }
        }

        // لو لم نجد أي IP صالح
        return '';
    }

    /**
     * إرجاع مسار الصفحة الحالي فقط (Path) مع دعم العربي
     *
     * مثال:
     *   /product-category/خدمات-صيانة/
     *
     * @return string
     */
    protected function get_current_url() {

        // REQUEST_URI زي ما البوت طلبها فعلاً (مثلاً: /my-page?x=1)
        $raw_request_uri = isset( $_SERVER['REQUEST_URI'] )
            ? wp_unslash( $_SERVER['REQUEST_URI'] )
            : '/';

        if ( $raw_request_uri === '' ) {
            $raw_request_uri = '/';
        }

        // نحللها علشان نفك ترميز العربي في الـ path فقط
        $parsed = wp_parse_url( $raw_request_uri );

        if ( ! is_array( $parsed ) ) {
            // لو حصل خطأ في التحليل نرجّع مجرد "/"
            return '/';
        }

        $path  = isset( $parsed['path'] ) && $parsed['path'] !== '' ? $parsed['path'] : '/';
        $query = isset( $parsed['query'] ) ? $parsed['query'] : '';

        // نضمن إن فيه "/" في البداية فقط بدون تعديل شكل الرابط النهائي
        if ( $path[0] !== '/' ) {
            $path = '/' . $path;
        }

        // فك ترميز العربي في الـ path فقط
        $path = rawurldecode( $path );

        // بناء الرابط النهائي من غير توحيد ولا trailing slash
        $url = $path;
        if ( $query !== '' ) {
            $url .= '?' . $query;
        }

        // فلتر لأي تعديل مستقبلي
        $url = apply_filters( 'bt_bots_tracker_current_url', $url );

        return $url;
    }

}
