<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * BT_Bots_Tracker_Loader
 *
 * كلاس مسؤول عن:
 * - تحميل باقي ملفات الإضافة (الكلاسات)
 * - تهيئة الإضافة (Database + Tracker + Admin Pages)
 * - التعامل مع التفعيل/التهيئة
 */
class BT_Bots_Tracker_Loader {

    /**
     * يتم استدعاؤها مع هوك plugins_loaded
     */
    public static function init() {
        $self = new self();
        $self->include_files();
        $self->bootstrap();
    }

    /**
     * يتم استدعاؤها عند تفعيل الإضافة لأول مرة
     */
    public static function activate() {

        // تحميل كلاس قاعدة البيانات لو مش متحمل
        if ( ! class_exists( 'BT_Bots_Tracker_Database' ) ) {
            require_once BT_BOTS_TRACKER_PATH . 'includes/class-bt-database.php';
        }

        // إنشاء الجدول في قاعدة البيانات
        BT_Bots_Tracker_Database::create_tables();

        /**
         * إنشاء كرون يومي للتنظيف (إن لم يكن موجودًا)
         * الحدث: bt_bots_tracker_cleanup_event
         */
        if ( ! wp_next_scheduled( 'bt_bots_tracker_cleanup_event' ) ) {
            wp_schedule_event( time(), 'daily', 'bt_bots_tracker_cleanup_event' );
        }
    }

    /**
     * يتم استدعاؤها عند إلغاء تفعيل الإضافة (إن تم ربطها في ملف الإضافة الرئيسي)
     * مهمتها إلغاء حدث الكرون الخاص بالتنظيف.
     */
    public static function deactivate() {
        $timestamp = wp_next_scheduled( 'bt_bots_tracker_cleanup_event' );
        if ( $timestamp ) {
            wp_unschedule_event( $timestamp, 'bt_bots_tracker_cleanup_event' );
        }
    }

    /**
     * تحميل باقي الملفات المطلوبة
     */
    protected function include_files() {

        // قاعدة البيانات
        require_once BT_BOTS_TRACKER_PATH . 'includes/class-bt-database.php';

        // كلاس التتبع الأساسي
        require_once BT_BOTS_TRACKER_PATH . 'includes/class-bt-tracker.php';

        // دوال مساعدة إن وُجدت
        if ( file_exists( BT_BOTS_TRACKER_PATH . 'includes/helpers.php' ) ) {
            require_once BT_BOTS_TRACKER_PATH . 'includes/helpers.php';
        }

        // كلاس المنيو في لوحة التحكم
        require_once BT_BOTS_TRACKER_PATH . 'includes/admin/class-bt-admin-menu.php';

        // صفحات لوحة التحكم
        require_once BT_BOTS_TRACKER_PATH . 'includes/admin/pages/class-bt-page-main.php';
        require_once BT_BOTS_TRACKER_PATH . 'includes/admin/pages/class-bt-page-crawl.php';

        // تحميل صفحة الإعدادات الجديدة (Setting)
        require_once BT_BOTS_TRACKER_PATH . 'includes/admin/pages/class-bt-page-setting.php';
    }

    /**
     * تهيئة الكلاسات وربط الهوكات
     */
    protected function bootstrap() {

        // تهيئة أي إعدادات في الداتا بيز
        if ( method_exists( 'BT_Bots_Tracker_Database', 'init' ) ) {
            BT_Bots_Tracker_Database::init();
        }

        // تشغيل التتبع
        $tracker = new BT_Bots_Tracker_Tracker();
        $tracker->init_hooks();

        // إنشاء الصفحات
        $page_main    = new BT_Bots_Tracker_Page_Main();
        $page_crawl   = new BT_Bots_Tracker_Page_Crawl();
        $page_setting = new BT_Bots_Tracker_Page_Setting(); // صفحة الإعدادات الجديدة

        // تمرير الصفحات كلها للمنيو
        $admin_menu = new BT_Bots_Tracker_Admin_Menu(
            $page_main,
            $page_crawl,
            $page_setting
        );

        // تسجيل المنيو
        $admin_menu->init_menu();
    }

    /**
     * دالة التنظيف التي يستدعيها الكرون اليومي
     *
     * - تقرأ عدد الأيام من الإعداد bt_bots_retention_days
     * - لو 0 أو أقل → لا تقوم بأي حذف (احتفاظ دائم)
     * - غير ذلك → تحذف الزيارات الأقدم من هذا العدد من الأيام
     * - بعد الحذف يتم عمل OPTIMIZE TABLE لتقليل المساحة
     */
    public static function run_cleanup() {

        // التأكد من وجود كلاس قاعدة البيانات
        if ( ! class_exists( 'BT_Bots_Tracker_Database' ) ) {
            require_once BT_BOTS_TRACKER_PATH . 'includes/class-bt-database.php';
        }

        // قراءة عدد الأيام من الإعدادات (افتراضي 30 يوم لو الخيار غير موجود)
        $days = (int) get_option( 'bt_bots_retention_days', 30 );

        // لو القيمة 0 أو أقل → لا نقوم بحذف أي بيانات أو أوبتمايز
        if ( $days <= 0 ) {
            return;
        }

        // تنفيذ الحذف فقط لو الدالة موجودة في كلاس الداتابيز
        if ( method_exists( 'BT_Bots_Tracker_Database', 'delete_older_than' ) ) {
            BT_Bots_Tracker_Database::delete_older_than( $days );
        }

        // بعد الحذف نعمل OPTIMIZE TABLE مرة واحدة يوميًا لتقليل المساحة
        if ( method_exists( 'BT_Bots_Tracker_Database', 'optimize_table' ) ) {
            BT_Bots_Tracker_Database::optimize_table();
        }
    }
}

/**
 * ربط حدث الكرون bt_bots_tracker_cleanup_event
 * بدالة التنظيف داخل اللودر.
 *
 * هذا يضمن أن ووردبريس يستدعي run_cleanup يومياً
 * حسب جدول الكرون المسجَّل في activate().
 */
add_action(
    'bt_bots_tracker_cleanup_event',
    array( 'BT_Bots_Tracker_Loader', 'run_cleanup' )
);
