<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * إدارة قاعدة البيانات لإضافة Bots Tracker
 */
class BT_Bots_Tracker_Database {

    /**
     * اسم الجدول بدون البادئة
     */
    const TABLE_SLUG = 'bots_visits';

    /**
     * إصدار بنية قاعدة البيانات التي يتوقعها الكود.
     * (مربوط بالثابت BT_BOTS_TRACKER_DB_VERSION في الملف الرئيسي للبلاجن)
     */
    const DB_VERSION = BT_BOTS_TRACKER_DB_VERSION;

    /**
     * جلب اسم الجدول كاملًا مع البادئة
     *
     * @return string
     */
    public static function get_table_name() {
        global $wpdb;

        return $wpdb->prefix . self::TABLE_SLUG;
    }

    /**
     * نقطة دخول نظام الـ Migration
     *
     * يتم استدعاؤها عند Bootstrap للإضافة (ليست مرتبطة بالتفعيل فقط).
     * هدفها:
     * - التأكد من أن جدول قاعدة البيانات موجود.
     * - تطبيق أي ترقيات لازمة بناءً على bt_bots_tracker_db_version.
     */
    public static function init() {
        // الإصدار الحالي المحفوظ في قاعدة البيانات
        $installed_version = get_option( 'bt_bots_tracker_db_version', '' );

        // نتأكد دائمًا أن الجدول الأساسي موجود
        self::create_tables();

        // لو مفيش نسخة محفوظة، نعتبرها "0" حتى نسمح بتطبيق الترقيات
        if ( empty( $installed_version ) ) {
            $installed_version = '0';
        }

        /**
         * 🔁 ترقيات حسب الإصدارات
         * هنا نطبق كل ترقيات الداتا بيز واحدة واحدة بالترتيب
         */

        // ترقية إلى 1.0.0 (حذف عمود user_agent لو كان موجود)
        if ( version_compare( $installed_version, '1.0.0', '<' ) ) {
            self::upgrade_to_1_0_0();

            // تحديث النسخة بعد إتمام الترقية
            $installed_version = '1.0.0';
            update_option( 'bt_bots_tracker_db_version', $installed_version );
        }

        // لو في ترقيات مستقبلية، تضاف هنا:
        // if ( version_compare( $installed_version, '1.2.0', '<' ) ) {
        //     self::upgrade_to_1_2_0();
        //     $installed_version = '1.2.0';
        //     update_option( 'bt_bots_tracker_db_version', $installed_version );
        // }

        // في النهاية، تأكد أن النسخة المحفوظة تساوي النسخة الحالية في الكود
        if ( version_compare( $installed_version, self::DB_VERSION, '<' ) ) {
            update_option( 'bt_bots_tracker_db_version', self::DB_VERSION );
        }
    }

    /**
     * ترقية بنية قاعدة البيانات إلى الإصدار 1.0.0
     *
     * مثال عملي:
     * - حذف أعمدة user_agent إن وُجدت لتقليل حجم الجدول.
     */
    protected static function upgrade_to_1_0_0() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'bots_visits';

        // تأكد أولاً أن الجدول موجود
        $table_exists = $wpdb->get_var(
            $wpdb->prepare(
                "SHOW TABLES LIKE %s",
                $table_name
            )
        );

        if ( $table_exists !== $table_name ) {
            // لو مفيش جدول، نسوي إنشاء جديد ونخرج
            self::create_tables();
            return;
        }

        // حذف عمود user_agent إن وجد (للاحتياط لو كان موجود في نسخ تجريبية)
        $ua_column = $wpdb->get_var(
            $wpdb->prepare(
                "SHOW COLUMNS FROM {$table_name} LIKE %s",
                'user_agent'
            )
        );

        if ( $ua_column ) {
            $wpdb->query( "ALTER TABLE {$table_name} DROP COLUMN user_agent" );
        }

        // يمكن هنا أيضاً إعادة بناء فهارس إن احتجت
        // مثال:
        // $wpdb->query( "ALTER TABLE {$table_name} ADD INDEX idx_visit_time (visit_time)" );
    }

    /**
     * إنشاء الجدول في قاعدة البيانات – تستدعى عند تفعيل الإضافة
     */
    public static function create_tables() {
        global $wpdb;

        $table_name      = self::get_table_name();
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$table_name} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            bot_name VARCHAR(45) NOT NULL,
            url_visited TEXT NOT NULL,
            visit_time DATETIME NOT NULL,
            status_code INT(3) DEFAULT NULL,
            ip_address VARCHAR(45) DEFAULT NULL,
            PRIMARY KEY  (id),
            INDEX idx_bot_name (bot_name),
            INDEX idx_visit_time (visit_time),
            INDEX idx_url_visited (url_visited(191))
        ) {$charset_collate};";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta( $sql );
    }

    /**
     * تسجيل زيارة بوت واحدة في الجدول
     *
     * @param array $data
     *   [
     *     'bot_name'    => string,
     *     'url_visited' => string,
     *     'visit_time'  => string (Y-m-d H:i:s),
     *     'status_code' => int|null (اختياري),
     *     'ip_address'  => string (اختياري)
     *   ]
     *
     * @return int|false ID السجل المُضاف أو false عند الفشل
     */
    public static function log_visit( $data ) {
        global $wpdb;

        $table_name = self::get_table_name();

        $defaults = array(
            'bot_name'    => '',
            'url_visited' => '',
            'visit_time'  => current_time( 'mysql' ),
            'status_code' => null,
            'ip_address'  => '',
        );

        $row = wp_parse_args( $data, $defaults );

        // تحديد كود الاستجابة النهائي: إما القادم من البيانات أو من الجلوبال الذي يلتقطه الفلتر
        if ( isset( $row['status_code'] ) && $row['status_code'] !== null && $row['status_code'] !== '' ) {
            $status_code = (int) $row['status_code'];
        } elseif ( isset( $GLOBALS['bt_tracker_status_code'] ) ) {
            $status_code = (int) $GLOBALS['bt_tracker_status_code'];
        } else {
            $status_code = null;
        }

        // تنظيف المدخلات
        // تمت إزالة تخزين user_agent من الجدول، ولا يزال يتم تخزين IP عند توفره
        $bot_name    = sanitize_text_field( $row['bot_name'] );
        $url_visited = $row['url_visited']; // نخزن المسار كما هو (يمكن أن يحتوي عربي)
        $visit_time  = sanitize_text_field( $row['visit_time'] );
        $ip_address  = sanitize_text_field( $row['ip_address'] );

        /**
         * تطبيق إعدادات الفلترة قبل التخزين في الداتا بيز
         *
         * - لو الفلتر معطّل ➜ نخزن كل البوتات (السلوك الافتراضي).
         * - لو الفلتر مفعّل:
         *    - وضع "exclude": نستبعد البوتات المطابقة للقائمة.
         *    - وضع "include": لا نخزن إلا البوتات المطابقة للقائمة.
         */
        $filter_enabled = (int) get_option( 'bt_bots_filter_enabled', 0 ) ? 1 : 0;

        if ( $filter_enabled ) {

            $mode = get_option( 'bt_bots_filter_mode', 'exclude' );
            if ( ! in_array( $mode, array( 'exclude', 'include' ), true ) ) {
                $mode = 'exclude';
            }

            $source = get_option( 'bt_bots_filter_source', 'stored' );
            if ( ! in_array( $source, array( 'stored', 'custom' ), true ) ) {
                $source = 'stored';
            }

            $bot_name_for_match = trim( (string) $bot_name );
            $bot_name_l         = strtolower( $bot_name_for_match );

            $match = false;

            if ( 'stored' === $source ) {
                // الاختيار من البوتات المخزنة
                $selected_bots = get_option( 'bt_bots_filter_selected_bots', array() );
                if ( is_array( $selected_bots ) && ! empty( $selected_bots ) && '' !== $bot_name_for_match ) {
                    foreach ( $selected_bots as $selected_name ) {
                        $selected_name = trim( (string) $selected_name );
                        if ( '' === $selected_name ) {
                            continue;
                        }

                        // مقارنة غير حساسة لحالة الأحرف
                        if ( 0 === strcasecmp( $bot_name_for_match, $selected_name ) ) {
                            $match = true;
                            break;
                        }
                    }
                }
            } else {
                // اختيار بالأسماء اليدوية – في طور التحسين
                $custom_names_raw = get_option( 'bt_bots_filter_custom_names', '' );
                if ( '' !== $custom_names_raw && '' !== $bot_name_for_match ) {
                    $lines = preg_split( '/\R+/', $custom_names_raw );
                    if ( is_array( $lines ) ) {
                        foreach ( $lines as $pattern ) {
                            $pattern = trim( (string) $pattern );
                            if ( '' === $pattern ) {
                                continue;
                            }
                            $pattern_l = strtolower( $pattern );

                            // نعتبر البوت مطابق لو الاسم فيه الباترن ده (substring)
                            if ( false !== strpos( $bot_name_l, $pattern_l ) ) {
                                $match = true;
                                break;
                            }
                        }
                    }
                }
            }

            // تطبيق منطق الوضع (استثناء / تضمين)
            if ( 'exclude' === $mode ) {
                // استثناء البوتات المطابقة للقائمة
                if ( $match ) {
                    // لا نقوم بتخزين هذه الزيارة
                    return false;
                }
            } else {
                // include = تخزين فقط البوتات المطابقة للقائمة
                if ( ! $match ) {
                    return false;
                }
            }
        }

        $inserted = $wpdb->insert(
            $table_name,
            array(
                'bot_name'    => $bot_name,
                'url_visited' => $url_visited,
                'visit_time'  => $visit_time,
                'status_code' => $status_code,
                'ip_address'  => $ip_address,
            ),
            array( '%s', '%s', '%s', '%d', '%s' )
        );

        if ( ! $inserted ) {
            return false;
        }

        return (int) $wpdb->insert_id;
    }

    /**
     * حذف السجلات الموجودة بالفعل بناءً على إعدادات الفلتر الحالية
     *
     * - لو الفلتر غير مفعّل ➜ لا يتم حذف شيء.
     * - مصدر "stored":
     *      - exclude: حذف السجلات الخاصة بالبـوتات المحددة فقط.
     *      - include: حذف كل السجلات ما عدا هذه البوتات.
     * - مصدر "custom":
     *      - exclude: حذف السجلات التي يحتوي اسم البوت فيها على أي من الأنماط.
     *      - include: حذف السجلات التي لا يحتوي اسم البوت فيها على أي من الأنماط.
     *
     * @return int عدد السجلات المحذوفة
     */
    public static function delete_by_current_filter() {
        global $wpdb;

        $table_name = self::get_table_name();

        $filter_enabled = (int) get_option( 'bt_bots_filter_enabled', 0 ) ? 1 : 0;
        if ( ! $filter_enabled ) {
            return 0;
        }

        $mode = get_option( 'bt_bots_filter_mode', 'exclude' );
        if ( ! in_array( $mode, array( 'exclude', 'include' ), true ) ) {
            $mode = 'exclude';
        }

        $source = get_option( 'bt_bots_filter_source', 'stored' );
        if ( ! in_array( $source, array( 'stored', 'custom' ), true ) ) {
            $source = 'stored';
        }

        $bots_or_patterns = array();

        if ( 'stored' === $source ) {
            $bots_or_patterns = get_option( 'bt_bots_filter_selected_bots', array() );
            if ( ! is_array( $bots_or_patterns ) ) {
                $bots_or_patterns = array();
            }
            $bots_or_patterns = array_filter(
                array_map(
                    'trim',
                    array_map( 'strval', $bots_or_patterns )
                ),
                static function( $v ) {
                    return '' !== $v;
                }
            );
        } else {
            $custom = get_option( 'bt_bots_filter_custom_names', '' );
            $lines  = preg_split( '/\R+/', (string) $custom );
            if ( is_array( $lines ) ) {
                foreach ( $lines as $line ) {
                    $line = trim( (string) $line );
                    if ( '' !== $line ) {
                        $bots_or_patterns[] = $line;
                    }
                }
            }
        }

        // لا توجد قيم صالحة ➜ لا داعي للحذف
        if ( empty( $bots_or_patterns ) ) {
            return 0;
        }

        // بناء الاستعلام
        $deleted = 0;

        if ( 'stored' === $source ) {
            // استخدام IN / NOT IN مع أسماء البوتات المحددة
            $placeholders = array_fill( 0, count( $bots_or_patterns ), '%s' );
            $in_clause    = implode( ',', $placeholders );

            if ( 'exclude' === $mode ) {
                // حذف السجلات المطابقة فقط
                $sql = "DELETE FROM {$table_name} WHERE bot_name IN ({$in_clause})";
                $wpdb->query( $wpdb->prepare( $sql, $bots_or_patterns ) );
            } else {
                // include: حذف أي شيء غير هذه البوتات
                $sql = "DELETE FROM {$table_name} WHERE bot_name NOT IN ({$in_clause})";
                $wpdb->query( $wpdb->prepare( $sql, $bots_or_patterns ) );
            }

            $deleted = (int) $wpdb->rows_affected;

        } else {
            // مصدر custom: نستخدم LIKE / NOT LIKE مع patterns
            $where_parts = array();
            $params      = array();

            if ( 'exclude' === $mode ) {
                // حذف السجلات التي تحتوي أسماؤها على أي pattern
                foreach ( $bots_or_patterns as $pattern ) {
                    $where_parts[] = 'bot_name LIKE %s';
                    $params[]      = '%' . $wpdb->esc_like( $pattern ) . '%';
                }

                $where_sql = implode( ' OR ', $where_parts );
                $sql       = "DELETE FROM {$table_name} WHERE {$where_sql}";
                $wpdb->query( $wpdb->prepare( $sql, $params ) );
            } else {
                // include: حذف السجلات التي لا تحتوي على أي من الأنماط
                foreach ( $bots_or_patterns as $pattern ) {
                    $where_parts[] = 'bot_name NOT LIKE %s';
                    $params[]      = '%' . $wpdb->esc_like( $pattern ) . '%';
                }

                $where_sql = implode( ' AND ', $where_parts );
                $sql       = "DELETE FROM {$table_name} WHERE {$where_sql}";
                $wpdb->query( $wpdb->prepare( $sql, $params ) );
            }

            $deleted = (int) $wpdb->rows_affected;
        }

        return $deleted;
    }

    /**
     * جلب سجلات الزيارات مع فلاتر وباجيناشن
     *
     * @param array $args
     *   [
     *     'limit'     => int,
     *     'offset'    => int,
     *     'bot_name'  => string|'',
     *     'url'       => string|'',
     *     'date_from' => string (Y-m-d) أو ''،
     *     'date_to'   => string (Y-m-d) أو ''،
     *     'order_by'  => 'visit_time' أو 'bot_name' أو 'url_visited'،
     *     'order'     => 'DESC' أو 'ASC'
     *   ]
     *
     * @return array من كائنات النتائج
     */
    public static function get_visits( $args = array() ) {
        global $wpdb;

        $table_name = self::get_table_name();

        $defaults = array(
            'limit'     => 50,
            'offset'    => 0,
            'bot_name'  => '',
            'url'       => '',
            'date_from' => '',
            'date_to'   => '',
            'order_by'  => 'visit_time',
            'order'     => 'DESC',
        );

        $args = wp_parse_args( $args, $defaults );

        $where   = array();
        $params  = array();

        $where[] = '1=1';

        if ( $args['bot_name'] !== '' ) {
            $where[]  = 'bot_name = %s';
            $params[] = $args['bot_name'];
        }

        if ( $args['url'] !== '' ) {
            $where[]  = 'url_visited LIKE %s';
            $params[] = '%' . $wpdb->esc_like( $args['url'] ) . '%';
        }

        if ( $args['date_from'] !== '' ) {
            $where[]  = 'visit_time >= %s';
            $params[] = $args['date_from'] . ' 00:00:00';
        }

        if ( $args['date_to'] !== '' ) {
            $where[]  = 'visit_time <= %s';
            $params[] = $args['date_to'] . ' 23:59:59';
        }

        $where_sql = implode( ' AND ', $where );

        $allowed_order_by = array( 'visit_time', 'bot_name', 'url_visited', 'id' );
        $order_by         = in_array( $args['order_by'], $allowed_order_by, true ) ? $args['order_by'] : 'visit_time';

        $order = strtoupper( $args['order'] ) === 'ASC' ? 'ASC' : 'DESC';

        $sql = "SELECT *
                FROM {$table_name}
                WHERE {$where_sql}
                ORDER BY {$order_by} {$order}
                LIMIT %d OFFSET %d";

        $params[] = (int) $args['limit'];
        $params[] = (int) $args['offset'];

        $prepared = $wpdb->prepare( $sql, $params );

        return $wpdb->get_results( $prepared );
    }

    /**
     * عدّ السجلات بناءً على نفس فلاتر get_visits – لاستخدامها في الباجيناشن
     *
     * @param array $args
     *
     * @return int
     */
    public static function count_visits( $args = array() ) {
        global $wpdb;

        $table_name = self::get_table_name();

        $defaults = array(
            'bot_name'  => '',
            'url'       => '',
            'date_from' => '',
            'date_to'   => '',
        );

        $args = wp_parse_args( $args, $defaults );

        $where   = array();
        $params  = array();

        $where[] = '1=1';

        if ( $args['bot_name'] !== '' ) {
            $where[]  = 'bot_name = %s';
            $params[] = $args['bot_name'];
        }

        if ( $args['url'] !== '' ) {
            $where[]  = 'url_visited LIKE %s';
            $params[] = '%' . $wpdb->esc_like( $args['url'] ) . '%';
        }

        if ( $args['date_from'] !== '' ) {
            $where[]  = 'visit_time >= %s';
            $params[] = $args['date_from'] . ' 00:00:00';
        }

        if ( $args['date_to'] !== '' ) {
            $where[]  = 'visit_time <= %s';
            $params[] = $args['date_to'] . ' 23:59:59';
        }

        $where_sql = implode( ' AND ', $where );

        $sql      = "SELECT COUNT(*) FROM {$table_name} WHERE {$where_sql}";
        $prepared = $wpdb->prepare( $sql, $params );

        $count = (int) $wpdb->get_var( $prepared );

        return $count;
    }

    /**
     * حذف السجلات الأقدم من عدد معيّن من الأيام
     *
     * @param int $days
     *
     * @return int عدد الصفوف المحذوفة
     */
    public static function delete_older_than( $days ) {
        global $wpdb;

        $table_name = self::get_table_name();

        $days = (int) $days;
        if ( $days <= 0 ) {
            return 0;
        }

        $cutoff = gmdate(
            'Y-m-d H:i:s',
            time() - ( $days * DAY_IN_SECONDS )
        );

        $sql = $wpdb->prepare(
            "DELETE FROM {$table_name} WHERE visit_time < %s",
            $cutoff
        );

        $wpdb->query( $sql );

        return (int) $wpdb->rows_affected;
    }

    /**
     * عمل OPTIMIZE TABLE للجدول لتقليل المساحة بعد الحذف
     */
    public static function optimize_table() {
        global $wpdb;

        $table_name = self::get_table_name();

        // أمر الأوبتمايز – يعيد تنظيم الجدول ويقلل الحجم على الديسك
        $wpdb->query( "OPTIMIZE TABLE {$table_name}" );
    }

    /**
     * جلب قائمة بكل أسماء البوتات الموجودة في الجدول
     *
     * @return array
     */
    public static function get_all_bots() {
        global $wpdb;

        $table_name = self::get_table_name();

        $sql = "SELECT DISTINCT bot_name FROM {$table_name} ORDER BY bot_name ASC";

        $results = $wpdb->get_col( $sql );

        if ( ! is_array( $results ) ) {
            return array();
        }

        return $results;
    }
}
