<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * صفحة إعدادات Bots Tracker (مدة الاحتفاظ + إدارة/تنظيف قاعدة البيانات)
 */
class BT_Bots_Tracker_Page_Setting {

    /**
     * Slug الخاص بالصفحة في لوحة التحكم
     */
    const PAGE_SLUG = 'bt-bots-setting';

    /**
     * عرض الصفحة داخل لوحة التحكم
     */
    public function render() {

        // صلاحيات الأدمن فقط
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        // ربط ملفات الـ CSS والـ JS الخاصة بالإضافة بهذه الصفحة فقط
        if ( function_exists( 'wp_enqueue_style' ) && defined( 'BT_BOTS_TRACKER_URL' ) ) {

            wp_enqueue_style(
                'bt-bots-tracker-admin',
                BT_BOTS_TRACKER_URL . 'includes/assets/css/style.css',
                array(),
                defined( 'BT_BOTS_TRACKER_VERSION' ) ? BT_BOTS_TRACKER_VERSION : time()
            );

            wp_enqueue_script(
                'bt-bots-tracker-admin-js',
                BT_BOTS_TRACKER_URL . 'includes/assets/js/admin.js',
                array( 'jquery' ),
                defined( 'BT_BOTS_TRACKER_VERSION' ) ? BT_BOTS_TRACKER_VERSION : time(),
                true
            );
        }

        // -----------------------------
        // 1) إعدادات مدة الاحتفاظ
        // -----------------------------

        // القيمة الحالية المحفوظة في الداتا بيز (افتراضي 30 يوم)
        $current_retention_days = (int) get_option( 'bt_bots_retention_days', 30 );
        if ( $current_retention_days < 1 ) {
            $current_retention_days = 1;
        }

        // حفظ الإعدادات عند إرسال فورم الاحتفاظ
        if ( isset( $_POST['bt_bots_retention_submit'] ) ) {

            check_admin_referer( 'bt_bots_retention_save', 'bt_bots_retention_nonce' );

            // قراءة القيمة القادمة من الفورم
            $days = isset( $_POST['bt_bots_retention_days'] )
                ? (int) $_POST['bt_bots_retention_days']
                : $current_retention_days;

            if ( $days < 1 ) {
                $days = 1;
            }

            // حفظ الخيار في الداتا بيز
            update_option( 'bt_bots_retention_days', $days );
            $current_retention_days = $days;

            // تحميل كلاس الداتا بيز
            $this->ensure_db_class();

            // تنفيذ حذف فوري للسجلات الأقدم من عدد الأيام المحدد
            if ( class_exists( 'BT_Bots_Tracker_Database' ) && method_exists( 'BT_Bots_Tracker_Database', 'delete_older_than' ) ) {
                BT_Bots_Tracker_Database::delete_older_than( $days );

                if ( method_exists( 'BT_Bots_Tracker_Database', 'optimize_table' ) ) {
                    BT_Bots_Tracker_Database::optimize_table();
                }
            }

            echo '<div class="notice notice-success is-dismissible bt-bots-notice"><p>'
                 . sprintf(
                     /* translators: %d: number of days. */
                     esc_html__( 'Retention settings saved. Bot logs older than %d days have been cleaned up.', 'azayem-bots-tracker' ),
                     (int) $days
                 )
                 . '</p></div>';
        }

        // -----------------------------
        // 1.1) إعدادات استهداف / تصفية البوتات قبل تخزينها
        // -----------------------------

        // قراءة الإعدادات الحالية من الداتا بيز
        $bt_filter_enabled = (int) get_option( 'bt_bots_filter_enabled', 0 ) ? 1 : 0;

        $bt_filter_mode = get_option( 'bt_bots_filter_mode', 'exclude' );
        if ( ! in_array( $bt_filter_mode, array( 'exclude', 'include' ), true ) ) {
            $bt_filter_mode = 'exclude';
        }

        $bt_filter_source = get_option( 'bt_bots_filter_source', 'stored' );
        if ( ! in_array( $bt_filter_source, array( 'stored', 'custom' ), true ) ) {
            $bt_filter_source = 'stored';
        }

        $bt_filter_selected_bots = get_option( 'bt_bots_filter_selected_bots', array() );
        if ( ! is_array( $bt_filter_selected_bots ) ) {
            $bt_filter_selected_bots = array();
        }

        $bt_filter_custom_names = get_option( 'bt_bots_filter_custom_names', '' );

        // حفظ إعدادات الفلترة عند إرسال الفورم
        if ( isset( $_POST['bt_bots_filter_submit'] ) ) {

            check_admin_referer( 'bt_bots_filter_save', 'bt_bots_filter_nonce' );

            $bt_filter_enabled = isset( $_POST['bt_bots_filter_enabled'] ) ? 1 : 0;

            $bt_filter_mode = isset( $_POST['bt_bots_filter_mode'] )
                ? sanitize_text_field( wp_unslash( $_POST['bt_bots_filter_mode'] ) )
                : 'exclude';

            if ( ! in_array( $bt_filter_mode, array( 'exclude', 'include' ), true ) ) {
                $bt_filter_mode = 'exclude';
            }

            $bt_filter_source = isset( $_POST['bt_bots_filter_source'] )
                ? sanitize_text_field( wp_unslash( $_POST['bt_bots_filter_source'] ) )
                : 'stored';

            if ( ! in_array( $bt_filter_source, array( 'stored', 'custom' ), true ) ) {
                $bt_filter_source = 'stored';
            }

            $bt_filter_selected_bots = array();
            if ( isset( $_POST['bt_bots_filter_selected_bots'] ) && is_array( $_POST['bt_bots_filter_selected_bots'] ) ) {
                foreach ( $_POST['bt_bots_filter_selected_bots'] as $bot_name ) {
                    $bot_name = sanitize_text_field( wp_unslash( $bot_name ) );
                    if ( '' !== $bot_name ) {
                        $bt_filter_selected_bots[] = $bot_name;
                    }
                }

                if ( ! empty( $bt_filter_selected_bots ) ) {
                    $bt_filter_selected_bots = array_values( array_unique( $bt_filter_selected_bots, SORT_STRING ) );
                }
            }

            $bt_filter_custom_names = isset( $_POST['bt_bots_filter_custom_names'] )
                ? sanitize_textarea_field( wp_unslash( $_POST['bt_bots_filter_custom_names'] ) )
                : '';

            update_option( 'bt_bots_filter_enabled', $bt_filter_enabled ? 1 : 0 );
            update_option( 'bt_bots_filter_mode', $bt_filter_mode );
            update_option( 'bt_bots_filter_source', $bt_filter_source );
            update_option( 'bt_bots_filter_selected_bots', $bt_filter_selected_bots );
            update_option( 'bt_bots_filter_custom_names', $bt_filter_custom_names );

            // تحميل كلاس الداتا بيز
            $this->ensure_db_class();

            $deleted_rows = 0;
            if ( class_exists( 'BT_Bots_Tracker_Database' ) && method_exists( 'BT_Bots_Tracker_Database', 'delete_by_current_filter' ) ) {
                $deleted_rows = BT_Bots_Tracker_Database::delete_by_current_filter();

                if ( method_exists( 'BT_Bots_Tracker_Database', 'optimize_table' ) ) {
                    BT_Bots_Tracker_Database::optimize_table();
                }
            }

            $notice_message = esc_html__( 'Bot filter settings saved successfully.', 'azayem-bots-tracker' );
            if ( $bt_filter_enabled ) {
                if ( $deleted_rows > 0 ) {
                    $notice_message .= ' ' . sprintf(
                        /* translators: %d: number of deleted rows. */
                        esc_html__( '%d existing records were deleted based on the new filter.', 'azayem-bots-tracker' ),
                        (int) $deleted_rows
                    );
                } else {
                    $notice_message .= ' ' . esc_html__( 'No existing records matched the filter for deletion.', 'azayem-bots-tracker' );
                }
            } else {
                $notice_message .= ' ' . esc_html__( 'Filter is currently disabled, so no existing records were deleted.', 'azayem-bots-tracker' );
            }

            echo '<div class="notice notice-success is-dismissible bt-bots-notice"><p>'
                 . esc_html( $notice_message )
                 . '</p></div>';
        }

        // ==================================================
        // 2) Database Optimization – الأكشنات + الإحصائيات
        // ==================================================

        global $wpdb;

        // قيم مبدئية لإحصائيات الجدول
        $bt_db_size_mb     = '';
        $bt_db_rows_count  = '';
        $bt_db_growth_text = '';

        // تأكد أن كلاس الداتا بيز متحمل
        $this->ensure_db_class();

        // اسم جدول الزيارات
        $bt_table_name = $this->get_table_name();

        // -----------------------------------------
        // 2.1) حذف السجلات الأقدم من X يوم (زر Delete)
        // -----------------------------------------
        if ( isset( $_POST['bt_bots_delete_old_submit'] ) ) {

            check_admin_referer( 'bt_bots_optimize_delete_old', 'bt_bots_optimize_delete_old_nonce' );

            $days_delete = isset( $_POST['bt_bots_delete_older_than'] )
                ? (int) wp_unslash( $_POST['bt_bots_delete_older_than'] )
                : 0;

            if ( $days_delete > 0 && class_exists( 'BT_Bots_Tracker_Database' ) && method_exists( 'BT_Bots_Tracker_Database', 'delete_older_than' ) ) {

                $deleted_rows = BT_Bots_Tracker_Database::delete_older_than( $days_delete );

                if ( method_exists( 'BT_Bots_Tracker_Database', 'optimize_table' ) ) {
                    BT_Bots_Tracker_Database::optimize_table();
                }

                echo '<div class="notice notice-success is-dismissible bt-bots-notice"><p>'
                     . sprintf(
                         /* translators: 1: number of deleted rows, 2: number of days. */
                         esc_html__( 'Deleted %1$d records older than %2$d days.', 'azayem-bots-tracker' ),
                         (int) $deleted_rows,
                         (int) $days_delete
                     )
                     . '</p></div>';

            } else {
                echo '<div class="notice notice-error is-dismissible bt-bots-notice"><p>'
                     . esc_html__( 'Could not delete old records. Please check the database class.', 'azayem-bots-tracker' )
                     . '</p></div>';
            }
        }

        // -----------------------------------------
        // 2.2) تفريغ الجدول بالكامل (زر Clear All)
        // -----------------------------------------
        if ( isset( $_POST['bt_bots_clear_all_submit'] ) ) {

            check_admin_referer( 'bt_bots_optimize_clear_all', 'bt_bots_optimize_clear_all_nonce' );

            if ( $bt_table_name && $wpdb instanceof wpdb ) {

                // عدد السجلات قبل التفريغ (لعرضه في الرسالة فقط)
                $before_count = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$bt_table_name}" );

                // تفريغ الجدول
                $wpdb->query( "TRUNCATE TABLE {$bt_table_name}" );

                // أوبتمايز بعد التفريغ (اختياري)
                if ( class_exists( 'BT_Bots_Tracker_Database' ) && method_exists( 'BT_Bots_Tracker_Database', 'optimize_table' ) ) {
                    BT_Bots_Tracker_Database::optimize_table();
                }

                echo '<div class="notice notice-success is-dismissible bt-bots-notice"><p>'
                     . sprintf(
                         /* translators: %d: number of removed records. */
                         esc_html__( 'Table truncated successfully. %d records have been removed.', 'azayem-bots-tracker' ),
                         (int) $before_count
                     )
                     . '</p></div>';

            } else {
                echo '<div class="notice notice-error is-dismissible bt-bots-notice"><p>'
                    . esc_html__( 'Could not truncate table. Table name or $wpdb not available.', 'azayem-bots-tracker' )
                    . '</p></div>';

            }
        }

        // -----------------------------------------
        // 2.3) قراءة حجم الجدول + عدد السجلات + معدل النمو
        // -----------------------------------------
        if ( $bt_table_name && $wpdb instanceof wpdb ) {

            // حجم الجدول وعدد السجلات
            $status = $wpdb->get_row(
                $wpdb->prepare( 'SHOW TABLE STATUS LIKE %s', $bt_table_name ),
                ARRAY_A
            );

            if ( ! empty( $status ) && isset( $status['Data_length'], $status['Index_length'], $status['Rows'] ) ) {

                $bytes            = (float) $status['Data_length'] + (float) $status['Index_length'];
                $bt_db_size_mb    = number_format_i18n( $bytes / 1024 / 1024, 2 );
                $bt_db_rows_count = number_format_i18n( (int) $status['Rows'] );
            }

            // معدل النمو اليومي (متوسط آخر 14 يوم)
            $days_window = 14;

            $sql_growth = "
                SELECT DATE(visit_time) AS visit_date, COUNT(*) AS total
                FROM {$bt_table_name}
                WHERE visit_time >= DATE_SUB(UTC_TIMESTAMP(), INTERVAL %d DAY)
                GROUP BY DATE(visit_time)
                ORDER BY visit_date DESC
            ";

            $growth_rows = $wpdb->get_results(
                $wpdb->prepare( $sql_growth, $days_window ),
                ARRAY_A
            );

            if ( ! empty( $growth_rows ) ) {
                $sum   = 0;
                $count = 0;

                foreach ( $growth_rows as $row ) {
                    $sum   += (int) $row['total'];
                    $count++;
                }

                if ( $count > 0 ) {
                    $avg_per_day = (int) round( $sum / $count );

                    $bt_db_growth_text = sprintf(
                        /* translators: 1: average rows, 2: days window. */
                        esc_html__( 'The table is growing by about %1$d records per day (average of last %2$d days).', 'azayem-bots-tracker' ),
                        $avg_per_day,
                        $days_window
                    );
                }
            }

            if ( '' === $bt_db_growth_text ) {
                $bt_db_growth_text = esc_html__( 'Not enough data to calculate daily growth yet.', 'azayem-bots-tracker' );
            }

        } else {
            $bt_db_growth_text = esc_html__( 'Visits table not found.', 'azayem-bots-tracker' );
        }

        // =========================
        // 3) واجهة صفحة الإعدادات
        // =========================
        ?>
        <div class="wrap bt-bots-tracker-wrap">

            <h1><?php esc_html_e( 'Settings', 'azayem-bots-tracker' ); ?></h1>

            <?php
            // إظهار التابات (Bot visits – Crawl budget – Settings)
            if ( function_exists( 'bt_bots_render_tabs' ) ) {
                bt_bots_render_tabs( self::PAGE_SLUG );
            }
            ?>

            <p class="description">
                <?php esc_html_e( 'Control how long to keep bot visits and manage the database size.', 'azayem-bots-tracker' ); ?>
            </p>

            <!-- إعدادات مدة الاحتفاظ -->
            <form method="post" style="margin-top: 20px; max-width: 600px;">

                <?php wp_nonce_field( 'bt_bots_retention_save', 'bt_bots_retention_nonce' ); ?>

                <table class="form-table" role="presentation">
                    <tbody>
                    <tr>
                        <th scope="row">
                            <label for="bt_bots_retention_days">
                                <?php esc_html_e( 'Retention period (days)', 'azayem-bots-tracker' ); ?>
                            </label>
                        </th>
                        <td>
                            <input
                                name="bt_bots_retention_days"
                                type="number"
                                id="bt_bots_retention_days"
                                value="<?php echo esc_attr( $current_retention_days ); ?>"
                                class="small-text"
                                min="1"
                            />
                            <p class="description">
                                <?php esc_html_e( 'Bot visits older than this number of days will be deleted automatically.', 'azayem-bots-tracker' ); ?>
                            </p>
                        </td>
                    </tr>
                    </tbody>
                </table>

                <?php
                submit_button(
                    esc_html__( 'Save retention settings', 'azayem-bots-tracker' ),
                    'primary',
                    'bt_bots_retention_submit'
                );
                ?>
            </form>

            <!-- ===================================================== -->
            <!-- Database Optimization — إدارة / تنظيف قاعدة البيانات -->
            <!-- ===================================================== -->

            <hr style="margin: 30px 0;">

            <div class="bt-bots-db-optimization">

                <h2>
                    <?php esc_html_e( 'Database Optimization', 'azayem-bots-tracker' ); ?>
                </h2>

                <p class="description">
                    <?php esc_html_e( 'Clean old bot visits, clear the whole table, and monitor the database size & growth.', 'azayem-bots-tracker' ); ?>
                </p>

                <!-- الصف العلوي: حالة الجدول + حذف السجلات القديمة جنب بعض -->
                <div class="bt-bots-db-top-row" style="display:flex; gap:20px; align-items:flex-start;">

                    <!-- صندوق الإحصائيات -->
                    <div class="bt-bots-db-stats">
                        <h3><?php esc_html_e( 'Current table status', 'azayem-bots-tracker' ); ?></h3>
                        <ul>
                            <li>
                                <strong><?php esc_html_e( 'Table size:', 'azayem-bots-tracker' ); ?></strong>
                                <?php
                                if ( '' !== $bt_db_size_mb ) {
                                    echo ' ' . esc_html(
                                        sprintf(
                                            /* translators: %s: table size in megabytes. */
                                            __( '%s MB', 'azayem-bots-tracker' ),
                                            $bt_db_size_mb
                                        )
                                    );
                                } else {
                                    echo ' ' . esc_html__( 'N/A', 'azayem-bots-tracker' );
                                }
                                ?>
                            </li>
                            <li>
                                <strong><?php esc_html_e( 'Rows count:', 'azayem-bots-tracker' ); ?></strong>
                                <?php
                                if ( '' !== $bt_db_rows_count ) {
                                    echo ' ' . esc_html( $bt_db_rows_count );
                                } else {
                                    echo ' ' . esc_html__( 'N/A', 'azayem-bots-tracker' );
                                }
                                ?>
                            </li>
                            <li>
                                <strong><?php esc_html_e( 'Daily growth:', 'azayem-bots-tracker' ); ?></strong>
                                <?php echo ' ' . esc_html( $bt_db_growth_text ); ?>
                            </li>
                        </ul>
                    </div>

                    <!-- صندوق حذف السجلات القديمة (بنفس شكل الكارت) -->
                    <div class="bt-bots-db-stats">
                        <h3><?php esc_html_e( 'Delete old records', 'azayem-bots-tracker' ); ?></h3>
                        <p class="description">
                            <?php esc_html_e( 'Delete bot visits that are older than a specific number of days.', 'azayem-bots-tracker' ); ?>
                        </p>

                        <form method="post" class="bt-bots-db-form">
                            <?php wp_nonce_field( 'bt_bots_optimize_delete_old', 'bt_bots_optimize_delete_old_nonce' ); ?>

                            <p>
                                <label>
                                    <input type="radio" name="bt_bots_delete_older_than" value="7" checked="checked" />
                                    <?php
                                    printf(
                                        /* translators: %d: number of days. */
                                        esc_html__( 'Older than %d days', 'azayem-bots-tracker' ),
                                        7
                                    );
                                    ?>
                                </label><br />
                                <label>
                                    <input type="radio" name="bt_bots_delete_older_than" value="30" />
                                    <?php
                                    printf(
                                        /* translators: %d: number of days. */
                                        esc_html__( 'Older than %d days', 'azayem-bots-tracker' ),
                                        30
                                    );
                                    ?>
                                </label><br />
                                <label>
                                    <input type="radio" name="bt_bots_delete_older_than" value="90" />
                                    <?php
                                    printf(
                                        /* translators: %d: number of days. */
                                        esc_html__( 'Older than %d days', 'azayem-bots-tracker' ),
                                        90
                                    );
                                    ?>
                                </label><br />
                                <label>
                                    <input type="radio" name="bt_bots_delete_older_than" value="180" />
                                    <?php
                                    printf(
                                        /* translators: %d: number of days. */
                                        esc_html__( 'Older than %d days', 'azayem-bots-tracker' ),
                                        180
                                    );
                                    ?>
                                </label>
                            </p>

                            <p>
                                <button
                                    type="submit"
                                    class="button button-secondary"
                                    id="bt-bots-delete-old-submit"
                                    data-confirm="<?php esc_attr_e( 'Are you sure you want to delete old records?', 'azayem-bots-tracker' ); ?>"
                                    name="bt_bots_delete_old_submit">
                                    <?php esc_html_e( 'Delete old records', 'azayem-bots-tracker' ); ?>
                                </button>
                            </p>
                        </form>
                    </div>

                </div><!-- .bt-bots-db-top-row -->

                <!-- تفريغ الجدول بالكامل -->
                <div class="bt-bots-db-box bt-bots-db-box-danger">
                    <h3><?php esc_html_e( 'Clear All Bot Visits', 'azayem-bots-tracker' ); ?></h3>
                    <p class="description">
                        <?php esc_html_e( 'This will TRUNCATE the visits table and remove ALL records. Use with caution.', 'azayem-bots-tracker' ); ?>
                    </p>

                    <form method="post" class="bt-bots-db-form">
                        <?php wp_nonce_field( 'bt_bots_optimize_clear_all', 'bt_bots_optimize_clear_all_nonce' ); ?>

                        <p>
                            <button
                                type="submit"
                                class="button button-secondary bt-bots-button-danger"
                                id="bt-bots-clear-all-submit"
                                data-confirm="<?php esc_attr_e( 'Are you absolutely sure? This will delete ALL bot visits and cannot be undone.', 'azayem-bots-tracker' ); ?>"
                                name="bt_bots_clear_all_submit">
                                <?php esc_html_e( 'Clear All Bot Visits', 'azayem-bots-tracker' ); ?>
                            </button>
                        </p>
                    </form>
                </div>

            </div><!-- .bt-bots-db-optimization -->

            <!-- إعدادات تحديد / استثناء البوتات قبل التخزين -->
            <hr style="margin: 30px 0;">

            <div class="bt-bots-filter-settings">

                <h2>
                    <?php esc_html_e( 'Bot filter (which bots are stored)', 'azayem-bots-tracker' ); ?>
                </h2>

                <p class="description">
                    <?php esc_html_e( 'Optionally limit which bots are logged into the database — for example, exclude certain tools or store only key search engine bots.', 'azayem-bots-tracker' ); ?>
                </p>

                <form method="post" style="margin-top: 15px; max-width: 100%;">

                    <?php wp_nonce_field( 'bt_bots_filter_save', 'bt_bots_filter_nonce' ); ?>

                    <div class="bt-bots-db-top-row bt-bots-filter-top-row">

                        <!-- الصندوق الأول: تفعيل الفلتر + وضع الفلترة -->
                        <div class="bt-bots-db-box">
                            <h3><?php esc_html_e( 'Filter behaviour', 'azayem-bots-tracker' ); ?></h3>

                            <table class="form-table" role="presentation">
                                <tbody>
                                <tr>
                                    <th scope="row">
                                        <label for="bt_bots_filter_enabled">
                                            <?php esc_html_e( 'Enable bot filter', 'azayem-bots-tracker' ); ?>
                                        </label>
                                    </th>
                                    <td>
                                        <label>
                                            <input
                                                type="checkbox"
                                                name="bt_bots_filter_enabled"
                                                id="bt_bots_filter_enabled"
                                                value="1"
                                                <?php checked( $bt_filter_enabled, 1 ); ?>
                                            />
                                            <?php esc_html_e( 'Only store visits based on the rules below.', 'azayem-bots-tracker' ); ?>
                                        </label>
                                        <p class="description">
                                            <?php esc_html_e( 'When disabled, all detected bot visits are stored (default behaviour).', 'azayem-bots-tracker' ); ?>
                                        </p>
                                    </td>
                                </tr>

                                <tr id="bt-bots-filter-mode-row">
                                    <th scope="row">
                                        <label for="bt_bots_filter_mode">
                                            <?php esc_html_e( 'Filter mode', 'azayem-bots-tracker' ); ?>
                                        </label>
                                    </th>
                                    <td>
                                        <select name="bt_bots_filter_mode" id="bt_bots_filter_mode">
                                            <option value="exclude" <?php selected( $bt_filter_mode, 'exclude' ); ?>>
                                                <?php esc_html_e( 'Exclude specific bots (recommended)', 'azayem-bots-tracker' ); ?>
                                            </option>
                                            <option value="include" <?php selected( $bt_filter_mode, 'include' ); ?>>
                                                <?php esc_html_e( 'Store only specific bots', 'azayem-bots-tracker' ); ?>
                                            </option>
                                        </select>
                                        <p class="description">
                                            <?php esc_html_e( 'Exclude mode: keep all bots except those you list. Include mode: store only the bots you list.', 'azayem-bots-tracker' ); ?>
                                        </p>
                                    </td>
                                </tr>
                                </tbody>
                            </table>
                        </div>

                        <!-- الصندوق الثاني: طريقة اختيار البوتات + القوائم -->
                        <div class="bt-bots-db-box">
                            <h3><?php esc_html_e( 'Which bots?', 'azayem-bots-tracker' ); ?></h3>

                            <table class="form-table" role="presentation">
                                <tbody>
                                <tr id="bt-bots-filter-source-row">
                                    <th scope="row">
                                        <label><?php esc_html_e( 'How do you select bots?', 'azayem-bots-tracker' ); ?></label>
                                    </th>
                                    <td>
                                        <fieldset>
                                            <label>
                                                <input
                                                    type="radio"
                                                    name="bt_bots_filter_source"
                                                    value="stored"
                                                    <?php checked( $bt_filter_source, 'stored' ); ?>
                                                />
                                                <?php esc_html_e( 'Choose from bots already stored in the database (recommended)', 'azayem-bots-tracker' ); ?>
                                            </label>
                                            <br />
                                            <label>
                                                <input
                                                    type="radio"
                                                    name="bt_bots_filter_source"
                                                    value="custom"
                                                    <?php checked( $bt_filter_source, 'custom' ); ?>
                                                />
                                                <?php esc_html_e( 'Type bot names manually (in beta / under improvement)', 'azayem-bots-tracker' ); ?>
                                            </label>
                                            <p class="description">
                                                <?php esc_html_e( 'Stored list is usually more reliable. Manual names are useful for bots that have not yet visited your site.', 'azayem-bots-tracker' ); ?>
                                            </p>
                                        </fieldset>
                                    </td>
                                </tr>

                                <tr class="bt-bots-filter-row-stored">
                                    <th scope="row">
                                        <label for="bt_bots_filter_selected_bots">
                                            <?php esc_html_e( 'Bots from stored list', 'azayem-bots-tracker' ); ?>
                                        </label>
                                    </th>
                                    <td>
                                        <?php
                                        $bt_available_bots = $this->get_available_bots();

                                        if ( ! empty( $bt_available_bots ) ) :
                                        ?>
                                            <select
                                                name="bt_bots_filter_selected_bots[]"
                                                id="bt_bots_filter_selected_bots"
                                                class="bt-bots-multiselect-select"
                                                multiple="multiple">
                                                <?php foreach ( $bt_available_bots as $bot_name ) : ?>
                                                    <option value="<?php echo esc_attr( $bot_name ); ?>" <?php selected( in_array( $bot_name, $bt_filter_selected_bots, true ), true ); ?>>
                                                        <?php echo esc_html( $bot_name ); ?>
                                                    </option>
                                                <?php endforeach; ?>
                                            </select>
                                            <p class="description">
                                                <?php esc_html_e( 'Click to open the list, then tick the bots you want to include/exclude.', 'azayem-bots-tracker' ); ?>
                                            </p>
                                        <?php else : ?>
                                            <p class="description">
                                                <?php esc_html_e( 'No bots found in the database yet. Once bots start visiting your site, they will appear here.', 'azayem-bots-tracker' ); ?>
                                            </p>
                                        <?php endif; ?>
                                    </td>
                                </tr>

                                <tr class="bt-bots-filter-row-custom">
                                    <th scope="row">
                                        <label for="bt_bots_filter_custom_names">
                                            <?php esc_html_e( 'Bot names (manual input)', 'azayem-bots-tracker' ); ?>
                                        </label>
                                    </th>
                                    <td>
                                        <textarea
                                            name="bt_bots_filter_custom_names"
                                            id="bt_bots_filter_custom_names"
                                            rows="4"
                                            cols="50"
                                            style="max-width: 100%;"><?php echo esc_textarea( $bt_filter_custom_names ); ?></textarea>
                                        <p class="description">
                                            <?php esc_html_e( 'Write one bot name per line, exactly as you expect it to appear (e.g. Googlebot, Bingbot, AhrefsBot).', 'azayem-bots-tracker' ); ?>
                                        </p>
                                    </td>
                                </tr>
                                </tbody>
                            </table>
                        </div>

                    </div><!-- .bt-bots-db-top-row.bt-bots-filter-top-row -->

                    <?php
                    submit_button(
                        esc_html__( 'Save bot filter settings', 'azayem-bots-tracker' ),
                        'secondary',
                        'bt_bots_filter_submit'
                    );
                    ?>
                </form>

            </div>

        </div>

        <?php
    }

    /**
     * تحميل كلاس الداتا بيز لو لم يكن محمّلًا
     */
    private function ensure_db_class() {
        if ( ! class_exists( 'BT_Bots_Tracker_Database' ) && defined( 'BT_BOTS_TRACKER_PATH' ) ) {
            require_once BT_BOTS_TRACKER_PATH . 'includes/class-bt-database.php';
        }
    }

    /**
     * إرجاع اسم جدول الزيارات
     *
     * @return string
     */
    private function get_table_name() {
        if ( class_exists( 'BT_Bots_Tracker_Database' ) && method_exists( 'BT_Bots_Tracker_Database', 'get_table_name' ) ) {
            return BT_Bots_Tracker_Database::get_table_name();
        }

        return '';
    }

    /**
     * جلب قائمة البوتات المتاحة من الداتا بيز
     *
     * @return array
     */
    private function get_available_bots() {
        $this->ensure_db_class();

        if ( class_exists( 'BT_Bots_Tracker_Database' ) && method_exists( 'BT_Bots_Tracker_Database', 'get_all_bots' ) ) {
            $bots = BT_Bots_Tracker_Database::get_all_bots();
            return is_array( $bots ) ? $bots : array();
        }

        return array();
    }
}
