<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * صفحة عرض الزيارات الرئيسية في لوحة التحكم
 */
class BT_Bots_Tracker_Page_Main {

    const PAGE_SLUG = 'bt-bots-tracker';

    /**
     * Remove control characters but keep % encoding intact.
     *
     * @param string $value Raw string.
     * @return string Cleaned string.
     */
    private static function clean_raw_search_input( $value ) {
        $value = (string) $value;
        $value = wp_unslash( $value );
        $value = trim( $value );

        // Remove ASCII control chars (keeps %D8.. intact).
        $value = preg_replace( '/[\x00-\x1F\x7F]/u', '', $value );

        return is_string( $value ) ? $value : '';
    }

    /**
     * Encode a URL path while preserving slashes.
     *
     * @param string $path Raw/decoded/mixed path.
     * @return string Encoded path with slashes preserved.
     */
    private static function encode_path_preserve_slashes( $path ) {
        $path = (string) $path;

        if ( '' === $path ) {
            return '/';
        }

        $path = '/' . ltrim( $path, '/' );

        if ( '/' === $path ) {
            return '/';
        }

        $segments = explode( '/', $path );
        foreach ( $segments as $i => $seg ) {
            if ( '' === $seg ) {
                continue;
            }
            $segments[ $i ] = rawurlencode( rawurldecode( $seg ) );
        }

        $encoded = implode( '/', $segments );
        $encoded = '/' . ltrim( $encoded, '/' );

        return $encoded;
    }

    /**
     * Build URL candidates for matching (decoded/encoded + slash/no-slash + query/no-query).
     *
     * @param string $search_raw_url Raw input (may be full URL or path, encoded or decoded).
     * @return array{display:string, like_values:string[]} Array with display string and LIKE patterns.
     */
    private static function build_url_like_candidates( $search_raw_url ) {
        $result = array(
            'display'     => '',
            'like_values' => array(),
        );

        $search_raw_url = (string) $search_raw_url;
        if ( '' === $search_raw_url ) {
            return $result;
        }

        // Display (decoded for UI)
        $result['display'] = rawurldecode( $search_raw_url );

        $parsed = wp_parse_url( $search_raw_url );

        $path = '';
        if ( is_array( $parsed ) && isset( $parsed['path'] ) && '' !== $parsed['path'] ) {
            $path = (string) $parsed['path'];
        } else {
            $path = $search_raw_url;
        }

        if ( '' === $path ) {
            $path = '/';
        }

        $query = ( is_array( $parsed ) && isset( $parsed['query'] ) ) ? (string) $parsed['query'] : '';

        // Decoded normalized
        $path_decoded = rawurldecode( $path );
        $path_decoded = '/' . ltrim( $path_decoded, '/' );

        $path_trimmed = rtrim( $path_decoded, '/' );
        if ( '' === $path_trimmed ) {
            $path_trimmed = '/';
        }

        // Encoded normalized
        $path_encoded_trimmed = self::encode_path_preserve_slashes( $path_trimmed );
        $path_encoded_trimmed = rtrim( $path_encoded_trimmed, '/' );
        if ( '' === $path_encoded_trimmed ) {
            $path_encoded_trimmed = '/';
        }

        $candidates = array();

        // With query + without query
        if ( '' !== $query ) {
            // decoded
            $candidates[] = $path_trimmed . '?' . $query;
            $candidates[] = ( '/' === $path_trimmed ) ? '/' : $path_trimmed . '/?' . $query;
            $candidates[] = $path_trimmed;
            $candidates[] = ( '/' === $path_trimmed ) ? '/' : $path_trimmed . '/';

            // encoded
            $candidates[] = $path_encoded_trimmed . '?' . $query;
            $candidates[] = ( '/' === $path_encoded_trimmed ) ? '/' : $path_encoded_trimmed . '/?' . $query;
            $candidates[] = $path_encoded_trimmed;
            $candidates[] = ( '/' === $path_encoded_trimmed ) ? '/' : $path_encoded_trimmed . '/';
        } else {
            // decoded
            $candidates[] = $path_trimmed;
            $candidates[] = ( '/' === $path_trimmed ) ? '/' : $path_trimmed . '/';

            // encoded
            $candidates[] = $path_encoded_trimmed;
            $candidates[] = ( '/' === $path_encoded_trimmed ) ? '/' : $path_encoded_trimmed . '/';
        }

        $candidates = array_values( array_unique( $candidates, SORT_STRING ) );

        foreach ( $candidates as $cand ) {
            $result['like_values'][] = $cand . '%';
        }

        return $result;
    }

    public function render() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        if ( function_exists( 'wp_enqueue_style' ) && defined( 'BT_BOTS_TRACKER_URL' ) ) {
            wp_enqueue_style(
                'bt-bots-tracker-admin',
                BT_BOTS_TRACKER_URL . 'includes/assets/css/style.css',
                array(),
                time()
            );

            wp_enqueue_script(
                'bt-bots-tracker-admin',
                BT_BOTS_TRACKER_URL . 'includes/assets/js/admin.js',
                array( 'jquery' ),
                time(),
                true
            );
        }

        global $wpdb;

        $table_name = $wpdb->prefix . 'bots_visits';

        // فلتر اسم البوت
        $bot_name = isset( $_GET['bt_bot_name'] )
            ? sanitize_text_field( wp_unslash( $_GET['bt_bot_name'] ) )
            : '';

        // نوع الفترة
        $period_type = isset( $_GET['bt_period_type'] )
            ? sanitize_text_field( wp_unslash( $_GET['bt_period_type'] ) )
            : '30';

        $custom_from = isset( $_GET['bt_date_from'] )
            ? sanitize_text_field( wp_unslash( $_GET['bt_date_from'] ) )
            : '';

        $custom_to = isset( $_GET['bt_date_to'] )
            ? sanitize_text_field( wp_unslash( $_GET['bt_date_to'] ) )
            : '';

        $date_from   = '';
        $date_to     = '';
        $range_label = '';

        $now_ts = current_time( 'timestamp' );

        if ( 'custom' === $period_type && ! empty( $custom_from ) && ! empty( $custom_to ) ) {
            $df_ts = strtotime( $custom_from . ' 00:00:00' );
            $dt_ts = strtotime( $custom_to . ' 23:59:59' );

            if ( $df_ts && $dt_ts && $dt_ts >= $df_ts ) {
                $date_from = date_i18n( 'Y-m-d', $df_ts );
                $date_to   = date_i18n( 'Y-m-d', $dt_ts );

                /* translators: 1: start date, 2: end date. */
                $range_label = sprintf(
                    __( 'From %1$s to %2$s', 'azayem-bots-tracker' ),
                    $date_from,
                    $date_to
                );
            } else {
                $period_type = '30';
            }
        }

        if ( 'custom' !== $period_type ) {
            $days = (int) $period_type;
            if ( ! in_array( $days, array( 1, 30, 60, 90, 180 ), true ) ) {
                $days = 30;
            }

            $to_ts   = $now_ts;
            $from_ts = $now_ts - ( $days * DAY_IN_SECONDS );

            $date_from = date_i18n( 'Y-m-d', $from_ts );
            $date_to   = date_i18n( 'Y-m-d', $to_ts );

            /* translators: %s: number of days. */
            $range_label = sprintf(
                _n( 'Last %s day', 'Last %s days', $days, 'azayem-bots-tracker' ),
                number_format_i18n( $days )
            );
        }

        // per page
        $per_page = isset( $_GET['bt_per_page'] ) ? (int) $_GET['bt_per_page'] : 100;
        if ( $per_page <= 0 ) {
            $per_page = 100;
        }

        // sorting
        $orderby = isset( $_GET['bt_orderby'] )
            ? sanitize_key( wp_unslash( $_GET['bt_orderby'] ) )
            : 'visit_time';

        $order = isset( $_GET['bt_order'] )
            ? strtoupper( sanitize_text_field( wp_unslash( $_GET['bt_order'] ) ) )
            : 'DESC';

        if ( ! in_array( $order, array( 'ASC', 'DESC' ), true ) ) {
            $order = 'DESC';
        }

        $allowed_orderby = array( 'bot_name', 'visit_time', 'status_code' );
        if ( ! in_array( $orderby, $allowed_orderby, true ) ) {
            $orderby = 'visit_time';
        }

        $page_no = isset( $_GET['paged'] ) ? (int) $_GET['paged'] : 1;
        if ( $page_no < 1 ) {
            $page_no = 1;
        }
        $offset = ( $page_no - 1 ) * $per_page;

        // ✅ URL filter (REAL FIX)
        $search_raw_url = isset( $_GET['bt_search_url'] )
            ? self::clean_raw_search_input( $_GET['bt_search_url'] )
            : '';

        $url_pack = self::build_url_like_candidates( $search_raw_url );
        $search_url_display = $url_pack['display'];
        $filter_like_values = $url_pack['like_values'];

        // -----------------------------
        // بناء WHERE للاستعلامين (COUNT + SELECT)
        // -----------------------------
        $date_from_full = $date_from . ' 00:00:00';
        $date_to_full   = $date_to . ' 23:59:59';

        $where_sql    = 'WHERE visit_time BETWEEN %s AND %s';
        $where_params = array( $date_from_full, $date_to_full );

        if ( '' !== $bot_name ) {
            $where_sql     .= ' AND bot_name = %s';
            $where_params[] = $bot_name;
        }

        if ( ! empty( $filter_like_values ) ) {
            $or_like = array();
            foreach ( $filter_like_values as $unused ) {
                $or_like[] = 'url_visited LIKE %s';
            }
            $where_sql = $where_sql . ' AND ( ' . implode( ' OR ', $or_like ) . ' )';
            $where_params = array_merge( $where_params, $filter_like_values );
        }

        // -----------------------------
        // ✅ Total count
        // -----------------------------
        $total_count = (int) $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM {$table_name} {$where_sql}",
                $where_params
            )
        );

        $total_pages = ( $per_page > 0 ) ? (int) ceil( $total_count / $per_page ) : 1;
        if ( $total_pages < 1 ) {
            $total_pages = 1;
        }
        if ( $page_no > $total_pages ) {
            $page_no = $total_pages;
            $offset  = ( $page_no - 1 ) * $per_page;
        }

        // -----------------------------
        // ✅ Fetch rows
        // -----------------------------
        $order_by_sql = $orderby; // safe due to whitelist
        $order_sql    = $order;

        $select_params   = $where_params;
        $select_params[] = $per_page;
        $select_params[] = $offset;

        $visits = $wpdb->get_results(
            $wpdb->prepare(
                "
                SELECT bot_name, url_visited, ip_address, visit_time, status_code
                FROM {$table_name}
                {$where_sql}
                ORDER BY {$order_by_sql} {$order_sql}
                LIMIT %d OFFSET %d
                ",
                $select_params
            )
        );

        // -----------------------------
        // 🔢 counts per bot (بنفس فلتر URL + نفس التاريخ، بدون فلتر bot_name)
        // -----------------------------
        $bots_counts = array();

        $bots_where_sql    = 'WHERE visit_time BETWEEN %s AND %s';
        $bots_where_params = array( $date_from_full, $date_to_full );

        if ( ! empty( $filter_like_values ) ) {
            $or_like = array();
            foreach ( $filter_like_values as $unused ) {
                $or_like[] = 'url_visited LIKE %s';
            }
            $bots_where_sql .= ' AND ( ' . implode( ' OR ', $or_like ) . ' )';
            $bots_where_params = array_merge( $bots_where_params, $filter_like_values );
        }

        $bots_counts_rows = $wpdb->get_results(
            $wpdb->prepare(
                "
                SELECT bot_name, COUNT(*) AS visits
                FROM {$table_name}
                {$bots_where_sql}
                GROUP BY bot_name
                ",
                $bots_where_params
            ),
            ARRAY_A
        );

        if ( ! empty( $bots_counts_rows ) ) {
            foreach ( $bots_counts_rows as $row ) {
                if ( empty( $row['bot_name'] ) ) {
                    continue;
                }
                $bots_counts[ $row['bot_name'] ] = isset( $row['visits'] ) ? (int) $row['visits'] : 0;
            }
        }

        // bots list
        $all_bots = BT_Bots_Tracker_Database::get_all_bots();

        // tabs
        $current_page_slug = isset( $_GET['page'] ) ? sanitize_key( $_GET['page'] ) : self::PAGE_SLUG;

        $tabs = array(
            'bt-bots-tracker'      => __( 'Bot visits', 'azayem-bots-tracker' ),
            'bt-bots-crawl-budget' => __( 'Crawl budget', 'azayem-bots-tracker' ),
            'bt-bots-setting'      => __( 'Settings', 'azayem-bots-tracker' ),
        );

        $current_url = '';
        if ( isset( $_SERVER['REQUEST_URI'] ) ) {
            $current_url = esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) );
        }

        $ordering_base_url = remove_query_arg(
            array( 'paged', 'bt_orderby', 'bt_order' ),
            $current_url
        );

        ?>
        <div class="wrap bt-bots-tracker-wrap">
            <h1 class="wp-heading-inline"><?php esc_html_e( 'Bots Tracker', 'azayem-bots-tracker' ); ?></h1>

            <h2 class="nav-tab-wrapper bt-bots-nav-tabs">
                <?php foreach ( $tabs as $slug => $label ) : ?>
                    <?php
                    $tab_url   = admin_url( 'admin.php?page=' . $slug );
                    $is_active = ( $slug === $current_page_slug );
                    ?>
                    <a href="<?php echo esc_url( $tab_url ); ?>"
                       class="nav-tab <?php echo $is_active ? 'nav-tab-active' : ''; ?>">
                        <?php echo esc_html( $label ); ?>
                    </a>
                <?php endforeach; ?>
            </h2>

            <p class="bt-bots-subtext">
                <?php esc_html_e(
                    'This page lists all recorded visits made by search engine bots (crawlers) to your site, with filters for bot name, date range and sortable columns.',
                    'azayem-bots-tracker'
                ); ?>
            </p>

            <form method="get" class="bt-bots-filter-form">
                <input type="hidden" name="page" value="<?php echo esc_attr( self::PAGE_SLUG ); ?>" />

                <table class="form-table">
                    <tbody>
                    <tr>
                        <th scope="row">
                            <label for="bt_bot_name"><?php esc_html_e( 'Bot name filter:', 'azayem-bots-tracker' ); ?></label>
                        </th>
                        <td>
                            <select name="bt_bot_name" id="bt_bot_name" class="bt-bots-bot-select">
                                <option value=""><?php esc_html_e( '— All bots —', 'azayem-bots-tracker' ); ?></option>
                                <?php if ( ! empty( $all_bots ) ) : ?>
                                    <?php foreach ( $all_bots as $bot ) : ?>
                                        <?php
                                        $bot_visits = isset( $bots_counts[ $bot ] ) ? (int) $bots_counts[ $bot ] : 0;
                                        $label      = $bot;
                                        if ( $bot_visits > 0 ) {
                                            $label .= ' (' . number_format_i18n( $bot_visits ) . ')';
                                        }
                                        ?>
                                        <option value="<?php echo esc_attr( $bot ); ?>" <?php selected( $bot_name, $bot ); ?>>
                                            <?php echo esc_html( $label ); ?>
                                        </option>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </select>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="bt_period_type"><?php esc_html_e( 'Date range:', 'azayem-bots-tracker' ); ?></label>
                        </th>
                        <td>
                            <fieldset>
                                <select name="bt_period_type" id="bt_period_type">
                                    <option value="1"  <?php selected( $period_type, '1' ); ?>>
                                        <?php esc_html_e( 'Last 24 hours', 'azayem-bots-tracker' ); ?>
                                    </option>
                                    <option value="30"  <?php selected( $period_type, '30' ); ?>>
                                        <?php esc_html_e( 'Last 30 days', 'azayem-bots-tracker' ); ?>
                                    </option>
                                    <option value="60"  <?php selected( $period_type, '60' ); ?>>
                                        <?php esc_html_e( 'Last 2 months (60 days)', 'azayem-bots-tracker' ); ?>
                                    </option>
                                    <option value="90"  <?php selected( $period_type, '90' ); ?>>
                                        <?php esc_html_e( 'Last 3 months (90 days)', 'azayem-bots-tracker' ); ?>
                                    </option>
                                    <option value="180" <?php selected( $period_type, '180' ); ?>>
                                        <?php esc_html_e( 'Last 6 months (180 days)', 'azayem-bots-tracker' ); ?>
                                    </option>
                                    <option value="custom" <?php selected( $period_type, 'custom' ); ?>>
                                        <?php esc_html_e( 'Custom range', 'azayem-bots-tracker' ); ?>
                                    </option>
                                </select>

                                <div id="bt-main-custom-dates" class="bt-main-custom-dates">
                                    <label>
                                        <?php esc_html_e( 'From:', 'azayem-bots-tracker' ); ?>
                                        <input type="date" name="bt_date_from" value="<?php echo esc_attr( $custom_from ); ?>" />
                                    </label>
                                    &nbsp;&nbsp;
                                    <label>
                                        <?php esc_html_e( 'To:', 'azayem-bots-tracker' ); ?>
                                        <input type="date" name="bt_date_to" value="<?php echo esc_attr( $custom_to ); ?>" />
                                    </label>
                                </div>

                                <?php if ( ! empty( $range_label ) ) : ?>
                                    <p class="description">
                                        <?php esc_html_e( 'Current range:', 'azayem-bots-tracker' ); ?>
                                        <?php echo ' ' . esc_html( $range_label ); ?>.
                                    </p>
                                <?php endif; ?>
                            </fieldset>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="bt_per_page"><?php esc_html_e( 'Items per page:', 'azayem-bots-tracker' ); ?></label>
                        </th>
                        <td>
                            <select name="bt_per_page" id="bt_per_page">
                                <?php
                                $options = array( 50, 100, 200, 300, 400, 500, 600, 700, 800, 900, 1000 );
                                foreach ( $options as $opt ) :
                                    ?>
                                    <option value="<?php echo esc_attr( $opt ); ?>" <?php selected( $per_page, $opt ); ?>>
                                        <?php
                                        /* translators: %d: number of rows displayed per page. */
                                        printf( esc_html__( '%d rows per page', 'azayem-bots-tracker' ), $opt );
                                        ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="bt_search_url"><?php esc_html_e( 'URL filter:', 'azayem-bots-tracker' ); ?></label>
                        </th>
                        <td>
                            <input type="text"
                                   id="bt_search_url"
                                   name="bt_search_url"
                                   class="bt-bots-url-input"
                                   placeholder="<?php esc_attr_e( 'https://www.example.com/...', 'azayem-bots-tracker' ); ?>"
                                   value="<?php echo esc_attr( $search_url_display ); ?>" />
                            <p class="description">
                                <?php esc_html_e( 'You can write or paste the full URL or just the path.', 'azayem-bots-tracker' ); ?>
                            </p>
                        </td>
                    </tr>
                    </tbody>
                </table>

                <p>
                    <button type="submit" class="button button-primary">
                        <?php esc_html_e( 'Apply filters', 'azayem-bots-tracker' ); ?>
                    </button>
                    <a href="<?php echo esc_url( admin_url( 'admin.php?page=' . self::PAGE_SLUG ) ); ?>" class="button">
                        <?php esc_html_e( 'Reset filters', 'azayem-bots-tracker' ); ?>
                    </a>
                </p>
            </form>

            <hr>

            <h2>
                <?php esc_html_e( 'Total visits:', 'azayem-bots-tracker' ); ?>
                <?php echo ' ' . esc_html( number_format_i18n( $total_count ) ); ?>
            </h2>

            <?php
            $build_sort_th = function( $key, $label, $width = '' ) use ( $orderby, $order, $ordering_base_url ) {
                $col_class  = 'manage-column sortable';
                $next_order = 'ASC';

                if ( $orderby === $key ) {
                    $col_class .= ' sorted ' . strtolower( $order );
                    $next_order = ( 'ASC' === $order ) ? 'DESC' : 'ASC';
                }

                $sort_url = add_query_arg(
                    array(
                        'bt_orderby' => $key,
                        'bt_order'   => $next_order,
                        'paged'      => 1,
                    ),
                    $ordering_base_url
                );

                echo '<th scope="col" class="' . esc_attr( $col_class ) . '"';
                if ( ! empty( $width ) ) {
                    echo ' style="width:' . esc_attr( $width ) . ';"';
                }
                echo '>';
                echo '<a href="' . esc_url( $sort_url ) . '">';
                echo '<span>' . esc_html( $label ) . '</span>';
                echo '<span class="sorting-indicator"></span>';
                echo '</a>';
                echo '</th>';
            };
            ?>

            <table class="widefat fixed striped bt-bots-table bt-bots-table-main">
                <thead>
                <tr>
                    <th style="width:60px;"><?php esc_html_e( '#', 'azayem-bots-tracker' ); ?></th>
                    <?php $build_sort_th( 'bot_name', __( 'Bot name', 'azayem-bots-tracker' ), '20%' ); ?>
                    <th><?php esc_html_e( 'URL', 'azayem-bots-tracker' ); ?></th>
                    <th style="width:15%;"><?php esc_html_e( 'IP address', 'azayem-bots-tracker' ); ?></th>
                    <?php
                    $build_sort_th( 'visit_time', __( 'Visit time', 'azayem-bots-tracker' ), '15%' );
                    $build_sort_th( 'status_code', __( 'Response code', 'azayem-bots-tracker' ), '10%' );
                    ?>
                </tr>
                </thead>
                <tbody>
                <?php if ( ! empty( $visits ) ) : ?>
                    <?php
                    $row_index = $offset + 1;
                    foreach ( $visits as $row ) :
                        $display_path = '';
                        $url_full     = '';

                        if ( ! empty( $row->url_visited ) ) {
                            $path         = $row->url_visited;
                            $display_path = rawurldecode( $path );
                            $url_full     = home_url( $path );
                        }

                        $formatted_time = $row->visit_time;
                        $ts             = strtotime( $row->visit_time );
                        if ( $ts ) {
                            $formatted_time = date_i18n( 'Y-m-d g:i:s A', $ts );
                        }
                        ?>
                        <tr>
                            <td><?php echo esc_html( $row_index++ ); ?></td>
                            <td><?php echo esc_html( $row->bot_name ); ?></td>
                            <td>
                                <?php if ( ! empty( $url_full ) ) : ?>
                                    <a href="<?php echo esc_url( $url_full ); ?>" target="_blank" class="bt-bots-url-link">
                                        <?php echo esc_html( $display_path ); ?>
                                    </a>
                                <?php endif; ?>
                            </td>
                            <td><?php echo ! empty( $row->ip_address ) ? esc_html( $row->ip_address ) : '—'; ?></td>
                            <td><?php echo esc_html( $formatted_time ); ?></td>
                            <td class="bt-status-code"><?php echo isset( $row->status_code ) ? esc_html( $row->status_code ) : ''; ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php else : ?>
                    <tr>
                        <td colspan="6"><?php esc_html_e( 'No visits found for the current filters.', 'azayem-bots-tracker' ); ?></td>
                    </tr>
                <?php endif; ?>
                </tbody>
            </table>

            <?php if ( $total_pages > 1 ) : ?>
                <div class="tablenav bt-bots-tablenav">
                    <div class="tablenav-pages">
                        <?php
                        $base_url = remove_query_arg( 'paged', $current_url );

                        echo '<span class="pagination-links">';

                        if ( $page_no > 1 ) {
                            $prev_url = add_query_arg( 'paged', $page_no - 1, $base_url );
                            echo '<a class="prev-page button" href="' . esc_url( $prev_url ) . '">&laquo; ' . esc_html__( 'Previous', 'azayem-bots-tracker' ) . '</a> ';
                        } else {
                            echo '<span class="tablenav-pages-navspan button disabled">&laquo; ' . esc_html__( 'Previous', 'azayem-bots-tracker' ) . '</span> ';
                        }

                        $start = max( 1, $page_no - 2 );
                        $end   = min( $total_pages, $page_no + 2 );

                        if ( $start > 1 ) {
                            $first_url = add_query_arg( 'paged', 1, $base_url );
                            echo '<a class="button" href="' . esc_url( $first_url ) . '">1</a> ';
                            if ( $start > 2 ) {
                                echo '<span class="tablenav-pages-navspan">…</span> ';
                            }
                        }

                        for ( $p = $start; $p <= $end; $p++ ) {
                            if ( $p === $page_no ) {
                                echo '<span class="button button-primary">' . esc_html( $p ) . '</span> ';
                            } else {
                                $p_url = add_query_arg( 'paged', $p, $base_url );
                                echo '<a class="button" href="' . esc_url( $p_url ) . '">' . esc_html( $p ) . '</a> ';
                            }
                        }

                        if ( $end < $total_pages ) {
                            if ( $end < $total_pages - 1 ) {
                                echo '<span class="tablenav-pages-navspan">…</span> ';
                            }
                            $last_url = add_query_arg( 'paged', $total_pages, $base_url );
                            echo '<a class="button" href="' . esc_url( $last_url ) . '">' . esc_html( $total_pages ) . '</a> ';
                        }

                        if ( $page_no < $total_pages ) {
                            $next_url = add_query_arg( 'paged', $page_no + 1, $base_url );
                            echo '<a class="button" href="' . esc_url( $next_url ) . '">' . esc_html__( 'Next', 'azayem-bots-tracker' ) . ' &raquo;</a>';
                        } else {
                            echo '<span class="tablenav-pages-navspan button disabled">' . esc_html__( 'Next', 'azayem-bots-tracker' ) . ' &raquo;</span>';
                        }

                        echo '</span>';
                        ?>
                    </div>
                </div>
            <?php endif; ?>

        </div>
        <?php
    }
}
