<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * صفحة ميزانية الزحف
 */
class BT_Bots_Tracker_Page_Crawl {

    const PAGE_SLUG = 'bt-bots-crawl-budget';

    /**
     * Remove control characters but keep % encoding intact.
     *
     * @param string $value Raw string.
     * @return string Cleaned string.
     */
    private static function clean_raw_search_input( $value ) {
        $value = (string) $value;
        $value = wp_unslash( $value );
        $value = trim( $value );

        // Remove ASCII control chars (keeps %D8.. intact).
        $value = preg_replace( '/[\x00-\x1F\x7F]/u', '', $value );

        return is_string( $value ) ? $value : '';
    }

    /**
     * Encode a URL path while preserving slashes.
     *
     * @param string $path Raw/decoded/mixed path.
     * @return string Encoded path with slashes preserved.
     */
    private static function encode_path_preserve_slashes( $path ) {
        $path = (string) $path;

        if ( '' === $path ) {
            return '/';
        }

        $path = '/' . ltrim( $path, '/' );

        if ( '/' === $path ) {
            return '/';
        }

        $segments = explode( '/', $path );
        foreach ( $segments as $i => $seg ) {
            if ( '' === $seg ) {
                continue;
            }
            $segments[ $i ] = rawurlencode( rawurldecode( $seg ) );
        }

        $encoded = implode( '/', $segments );
        $encoded = '/' . ltrim( $encoded, '/' );

        return $encoded;
    }

    /**
     * Build URL LIKE candidates (decoded/encoded + slash/no-slash + query/no-query).
     *
     * @param string $search_raw_url Raw input (may be full URL or path, encoded or decoded).
     * @return array{display:string, like_values:string[]} Array with display string and LIKE patterns.
     */
    private static function build_url_like_candidates( $search_raw_url ) {
        $result = array(
            'display'     => '',
            'like_values' => array(),
        );

        $search_raw_url = (string) $search_raw_url;
        if ( '' === $search_raw_url ) {
            return $result;
        }

        // Display (decoded for UI)
        $result['display'] = rawurldecode( $search_raw_url );

        $parsed = wp_parse_url( $search_raw_url );

        $path = '';
        if ( is_array( $parsed ) && isset( $parsed['path'] ) && '' !== $parsed['path'] ) {
            $path = (string) $parsed['path'];
        } else {
            $path = $search_raw_url;
        }

        if ( '' === $path ) {
            $path = '/';
        }

        $query = ( is_array( $parsed ) && isset( $parsed['query'] ) ) ? (string) $parsed['query'] : '';

        // Decoded normalized
        $path_decoded = rawurldecode( $path );
        $path_decoded = '/' . ltrim( $path_decoded, '/' );

        $path_trimmed = rtrim( $path_decoded, '/' );
        if ( '' === $path_trimmed ) {
            $path_trimmed = '/';
        }

        // Encoded normalized
        $path_encoded_trimmed = self::encode_path_preserve_slashes( $path_trimmed );
        $path_encoded_trimmed = rtrim( $path_encoded_trimmed, '/' );
        if ( '' === $path_encoded_trimmed ) {
            $path_encoded_trimmed = '/';
        }

        $candidates = array();

        if ( '' !== $query ) {
            // decoded
            $candidates[] = $path_trimmed . '?' . $query;
            $candidates[] = ( '/' === $path_trimmed ) ? '/' : $path_trimmed . '/?' . $query;
            $candidates[] = $path_trimmed;
            $candidates[] = ( '/' === $path_trimmed ) ? '/' : $path_trimmed . '/';

            // encoded
            $candidates[] = $path_encoded_trimmed . '?' . $query;
            $candidates[] = ( '/' === $path_encoded_trimmed ) ? '/' : $path_encoded_trimmed . '/?' . $query;
            $candidates[] = $path_encoded_trimmed;
            $candidates[] = ( '/' === $path_encoded_trimmed ) ? '/' : $path_encoded_trimmed . '/';
        } else {
            // decoded
            $candidates[] = $path_trimmed;
            $candidates[] = ( '/' === $path_trimmed ) ? '/' : $path_trimmed . '/';

            // encoded
            $candidates[] = $path_encoded_trimmed;
            $candidates[] = ( '/' === $path_encoded_trimmed ) ? '/' : $path_encoded_trimmed . '/';
        }

        $candidates = array_values( array_unique( $candidates, SORT_STRING ) );

        foreach ( $candidates as $cand ) {
            $result['like_values'][] = $cand . '%';
        }

        return $result;
    }

    public function render() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        if ( function_exists( 'wp_enqueue_style' ) && defined( 'BT_BOTS_TRACKER_URL' ) ) {

            wp_enqueue_style(
                'bt-bots-tracker-admin',
                BT_BOTS_TRACKER_URL . 'includes/assets/css/style.css',
                array(),
                time()
            );

            wp_enqueue_script(
                'bt-bots-tracker-admin-js',
                BT_BOTS_TRACKER_URL . 'includes/assets/js/admin.js',
                array( 'jquery' ),
                time(),
                true
            );
        }

        global $wpdb;

        $table_name = $wpdb->prefix . 'bots_visits';

        $all_bots = BT_Bots_Tracker_Database::get_all_bots();

        $bot_filter = isset( $_GET['bt_bot_name'] )
            ? sanitize_text_field( wp_unslash( $_GET['bt_bot_name'] ) )
            : '';

        $period_type = isset( $_GET['bt_period_type'] )
            ? sanitize_text_field( wp_unslash( $_GET['bt_period_type'] ) )
            : '30';

        $custom_from = isset( $_GET['bt_date_from'] )
            ? sanitize_text_field( wp_unslash( $_GET['bt_date_from'] ) )
            : '';
        $custom_to   = isset( $_GET['bt_date_to'] )
            ? sanitize_text_field( wp_unslash( $_GET['bt_date_to'] ) )
            : '';

        $per_page_type   = isset( $_GET['bt_per_page_type'] )
            ? sanitize_text_field( wp_unslash( $_GET['bt_per_page_type'] ) )
            : '100';
        $per_page_custom = isset( $_GET['bt_per_page_custom'] )
            ? (int) $_GET['bt_per_page_custom']
            : 0;

        if ( 'custom' === $per_page_type && $per_page_custom > 0 ) {
            $per_page = min( $per_page_custom, 5000 );
        } else {
            $allowed_per_page = array( 50, 100, 200, 300, 400, 500, 1000 );
            $per_page         = (int) $per_page_type;
            if ( ! in_array( $per_page, $allowed_per_page, true ) ) {
                $per_page      = 100;
                $per_page_type = '100';
            }
        }

        $orderby = isset( $_GET['bt_orderby'] )
            ? sanitize_key( wp_unslash( $_GET['bt_orderby'] ) )
            : 'hits';

        $order = isset( $_GET['bt_order'] )
            ? strtoupper( sanitize_text_field( wp_unslash( $_GET['bt_order'] ) ) )
            : 'DESC';

        if ( ! in_array( $order, array( 'ASC', 'DESC' ), true ) ) {
            $order = 'DESC';
        }

        $allowed_orderby = array( 'path', 'hits', 'last_visit', 'last_status' );
        if ( ! in_array( $orderby, $allowed_orderby, true ) ) {
            $orderby = 'hits';
        }

        $current_page = isset( $_GET['bt_paged'] ) ? (int) $_GET['bt_paged'] : 1;
        if ( $current_page < 1 ) {
            $current_page = 1;
        }
        $offset = ( $current_page - 1 ) * $per_page;

        $date_from   = '';
        $date_to     = '';
        $range_label = '';

        $now_ts = current_time( 'timestamp' );

        if ( 'custom' === $period_type && ! empty( $custom_from ) && ! empty( $custom_to ) ) {
            $df_ts = strtotime( $custom_from . ' 00:00:00' );
            $dt_ts = strtotime( $custom_to . ' 23:59:59' );

            if ( $df_ts && $dt_ts && $dt_ts >= $df_ts ) {
                $date_from = date_i18n( 'Y-m-d', $df_ts );
                $date_to   = date_i18n( 'Y-m-d', $dt_ts );

                /* translators: 1: start date, 2: end date. */
                $range_label = sprintf(
                    __( 'From %1$s to %2$s', 'azayem-bots-tracker' ),
                    $date_from,
                    $date_to
                );
            } else {
                $period_type = '30';
            }
        }

        if ( 'custom' !== $period_type ) {
            $days = (int) $period_type;
            if ( ! in_array( $days, array( 1, 30, 60, 90, 180 ), true ) ) {
                $days = 30;
            }

            $date_to_ts   = $now_ts;
            $date_from_ts = $now_ts - ( $days * DAY_IN_SECONDS );

            $date_from = date_i18n( 'Y-m-d', $date_from_ts );
            $date_to   = date_i18n( 'Y-m-d', $date_to_ts );

            /* translators: %s: number of days. */
            $range_label = sprintf(
                _n( 'Last %s day', 'Last %s days', $days, 'azayem-bots-tracker' ),
                number_format_i18n( $days )
            );
        }

        $date_from_full = $date_from . ' 00:00:00';
        $date_to_full   = $date_to . ' 23:59:59';

        // ✅ WHERE الأساسي: فترة
        $where_sql    = 'WHERE visit_time BETWEEN %s AND %s';
        $where_params = array( $date_from_full, $date_to_full );

        // ✅ URL Search (REAL FIX: OR LIKE on decoded+encoded variants)
        $search_raw = isset( $_GET['bt_search_url'] )
            ? self::clean_raw_search_input( $_GET['bt_search_url'] )
            : '';

        $url_pack       = self::build_url_like_candidates( $search_raw );
        $search_display = $url_pack['display'];
        $like_values    = $url_pack['like_values'];

        if ( ! empty( $like_values ) ) {
            $or_parts = array();
            foreach ( $like_values as $unused ) {
                $or_parts[] = 'url_visited LIKE %s';
            }
            $where_sql .= ' AND ( ' . implode( ' OR ', $or_parts ) . ' )';

            foreach ( $like_values as $lv ) {
                $where_params[] = $lv;
            }
        }

        /**
         * ✅ حساب عدد زيارات كل بوت لنفس (الفترة + URL filter) بدون فلتر bot
         */
        $bots_counts = array();

        $bots_counts_rows = $wpdb->get_results(
            $wpdb->prepare(
                "
                SELECT bot_name, COUNT(*) AS visits
                FROM {$table_name}
                {$where_sql}
                GROUP BY bot_name
                ",
                $where_params
            ),
            ARRAY_A
        );

        if ( ! empty( $bots_counts_rows ) ) {
            foreach ( $bots_counts_rows as $row ) {
                if ( ! isset( $row['bot_name'] ) ) {
                    continue;
                }
                $name   = $row['bot_name'];
                $visits = isset( $row['visits'] ) ? (int) $row['visits'] : 0;
                $bots_counts[ $name ] = $visits;
            }
        }

        // ⬅️ تطبيق فلتر البوت للعرض فقط
        $where_sql_for_report    = $where_sql;
        $where_params_for_report = $where_params;

        if ( ! empty( $bot_filter ) ) {
            $where_sql_for_report     .= ' AND bot_name = %s';
            $where_params_for_report[] = $bot_filter;
        }

        $total_urls = (int) $wpdb->get_var(
            $wpdb->prepare(
                "
                SELECT COUNT(DISTINCT url_visited)
                FROM {$table_name}
                {$where_sql_for_report}
                ",
                $where_params_for_report
            )
        );

        $total_pages = ( $total_urls > 0 && $per_page > 0 ) ? (int) ceil( $total_urls / $per_page ) : 1;
        if ( $total_pages < 1 ) {
            $total_pages = 1;
        }
        if ( $current_page > $total_pages ) {
            $current_page = $total_pages;
            $offset       = ( $current_page - 1 ) * $per_page;
        }

        switch ( $orderby ) {
            case 'path':
                $order_by_sql = 't.url_visited';
                break;
            case 'last_visit':
                $order_by_sql = 't.last_visit';
                break;
            case 'last_status':
                $order_by_sql = 'last_status';
                break;
            case 'hits':
            default:
                $order_by_sql = 't.visits';
                break;
        }

        $data_params   = $where_params_for_report;
        $data_params[] = $per_page;
        $data_params[] = $offset;

        $rows = $wpdb->get_results(
            $wpdb->prepare(
                "
                SELECT
                    t.url_visited,
                    t.visits,
                    t.last_visit,
                    v.status_code AS last_status
                FROM (
                    SELECT
                        url_visited,
                        COUNT(*)        AS visits,
                        MAX(visit_time) AS last_visit
                    FROM {$table_name}
                    {$where_sql_for_report}
                    GROUP BY url_visited
                ) AS t
                JOIN {$table_name} AS v
                  ON v.url_visited = t.url_visited
                 AND v.visit_time  = t.last_visit
                ORDER BY {$order_by_sql} {$order}, t.url_visited ASC
                LIMIT %d OFFSET %d
                ",
                $data_params
            ),
            ARRAY_A
        );

        $total_visits = 0;
        if ( $total_urls > 0 ) {
            $total_visits = (int) $wpdb->get_var(
                $wpdb->prepare(
                    "
                    SELECT COUNT(*)
                    FROM {$table_name}
                    {$where_sql_for_report}
                    ",
                    $where_params_for_report
                )
            );
        }

        $filter_args = array(
            'page'               => self::PAGE_SLUG,
            'bt_bot_name'        => $bot_filter,
            'bt_period_type'     => $period_type,
            'bt_date_from'       => $custom_from,
            'bt_date_to'         => $custom_to,
            'bt_per_page_type'   => $per_page_type,
            'bt_per_page_custom' => ( 'custom' === $per_page_type && $per_page_custom > 0 ) ? $per_page_custom : '',
            'bt_search_url'      => $search_raw, // ✅ نحفظ الـ raw عشان ما نفقدش الترميز في الرابط
        );

        foreach ( $filter_args as $key => $value ) {
            if ( '' === $value || null === $value ) {
                unset( $filter_args[ $key ] );
            }
        }

        $ordering_base_url = add_query_arg( $filter_args, admin_url( 'admin.php' ) );

        $pagination_args               = $filter_args;
        $pagination_args['bt_orderby'] = $orderby;
        $pagination_args['bt_order']   = $order;

        $pagination_base_url = add_query_arg( $pagination_args, admin_url( 'admin.php' ) );

        $custom_dates_class    = ( 'custom' === $period_type ) ? '' : ' bt-is-hidden';
        $per_page_custom_class = ( 'custom' === $per_page_type ) ? '' : ' bt-is-hidden';

        ?>
        <div class="wrap bt-bots-tracker-wrap">

            <h1 class="wp-heading-inline">
                <?php esc_html_e( 'Crawl budget', 'azayem-bots-tracker' ); ?>
            </h1>

            <?php
            if ( function_exists( 'bt_bots_render_tabs' ) ) {
                bt_bots_render_tabs( self::PAGE_SLUG );
            }
            ?>

            <p class="bt-bots-subtext">
                <?php esc_html_e(
                    'This report shows all URLs that have been crawled by bots, including how many times each URL was visited in a given period. You can filter by bot name, date range, and control how many rows are shown per page.',
                    'azayem-bots-tracker'
                ); ?>
            </p>

            <div class="bt-bots-crawl-forms-row">

                <form method="get" class="bt-bots-filters-form">
                    <input type="hidden" name="page" value="<?php echo esc_attr( self::PAGE_SLUG ); ?>" />

                    <table class="form-table">
                        <tbody>

                        <tr>
                            <th scope="row">
                                <label for="bt_bot_name">
                                    <?php esc_html_e( 'Bot name filter:', 'azayem-bots-tracker' ); ?>
                                </label>
                            </th>
                            <td>
                                <select name="bt_bot_name" id="bt_bot_name" class="bt-bots-bot-select">
                                    <option value=""><?php esc_html_e( '— All bots —', 'azayem-bots-tracker' ); ?></option>
                                    <?php if ( ! empty( $all_bots ) ) : ?>
                                        <?php foreach ( $all_bots as $bot ) : ?>
                                            <?php
                                            $bot_visits = isset( $bots_counts[ $bot ] ) ? (int) $bots_counts[ $bot ] : 0;
                                            $label      = $bot;
                                            if ( $bot_visits > 0 ) {
                                                $label .= ' (' . number_format_i18n( $bot_visits ) . ')';
                                            }
                                            ?>
                                            <option value="<?php echo esc_attr( $bot ); ?>" <?php selected( $bot_filter, $bot ); ?>>
                                                <?php echo esc_html( $label ); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </select>
                                <p class="description">
                                    <?php esc_html_e(
                                        'You can select a specific bot (for example Googlebot or GPTBot), or leave this empty to include all bots.',
                                        'azayem-bots-tracker'
                                    ); ?>
                                </p>
                            </td>
                        </tr>

                        <tr>
                            <th scope="row">
                                <?php esc_html_e( 'Date range:', 'azayem-bots-tracker' ); ?>
                            </th>
                            <td>
                                <fieldset>
                                    <select name="bt_period_type" id="bt_period_type">
                                        <option value="1"  <?php selected( $period_type, '1' ); ?>>
                                            <?php esc_html_e( 'Last 24 hours', 'azayem-bots-tracker' ); ?>
                                        </option>
                                        <option value="30"  <?php selected( $period_type, '30' ); ?>>
                                            <?php esc_html_e( 'Last 30 days', 'azayem-bots-tracker' ); ?>
                                        </option>
                                        <option value="60"  <?php selected( $period_type, '60' ); ?>>
                                            <?php esc_html_e( 'Last 2 months (60 days)', 'azayem-bots-tracker' ); ?>
                                        </option>
                                        <option value="90"  <?php selected( $period_type, '90' ); ?>>
                                            <?php esc_html_e( 'Last 3 months (90 days)', 'azayem-bots-tracker' ); ?>
                                        </option>
                                        <option value="180" <?php selected( $period_type, '180' ); ?>>
                                            <?php esc_html_e( 'Last 6 months (180 days)', 'azayem-bots-tracker' ); ?>
                                        </option>
                                        <option value="custom" <?php selected( $period_type, 'custom' ); ?>>
                                            <?php esc_html_e( 'Custom range', 'azayem-bots-tracker' ); ?>
                                        </option>
                                    </select>

                                    <div id="bt-custom-dates" class="bt-custom-dates<?php echo esc_attr( $custom_dates_class ); ?>">
                                        <label>
                                            <?php esc_html_e( 'To:', 'azayem-bots-tracker' ); ?>
                                            <input type="date" name="bt_date_to" value="<?php echo esc_attr( $custom_to ); ?>" />
                                        </label>
                                        &nbsp;&nbsp;
                                        <label>
                                            <?php esc_html_e( 'From:', 'azayem-bots-tracker' ); ?>
                                            <input type="date" name="bt_date_from" value="<?php echo esc_attr( $custom_from ); ?>" />
                                        </label>
                                    </div>

                                    <p class="description">
                                        <?php esc_html_e(
                                            'By default, the last 30 days are used. When you choose "Custom range", you can select a start and end date manually.',
                                            'azayem-bots-tracker'
                                        ); ?>
                                    </p>
                                </fieldset>
                            </td>
                        </tr>

                        <tr>
                            <th scope="row">
                                <?php esc_html_e( 'Rows per page:', 'azayem-bots-tracker' ); ?>
                            </th>
                            <td>
                                <fieldset>
                                    <select name="bt_per_page_type" id="bt_per_page_type">
                                        <option value="50"   <?php selected( $per_page_type, '50' ); ?>><?php esc_html_e( '50 rows per page', 'azayem-bots-tracker' ); ?></option>
                                        <option value="100"  <?php selected( $per_page_type, '100' ); ?>><?php esc_html_e( '100 rows per page', 'azayem-bots-tracker' ); ?></option>
                                        <option value="200"  <?php selected( $per_page_type, '200' ); ?>><?php esc_html_e( '200 rows per page', 'azayem-bots-tracker' ); ?></option>
                                        <option value="300"  <?php selected( $per_page_type, '300' ); ?>><?php esc_html_e( '300 rows per page', 'azayem-bots-tracker' ); ?></option>
                                        <option value="400"  <?php selected( $per_page_type, '400' ); ?>><?php esc_html_e( '400 rows per page', 'azayem-bots-tracker' ); ?></option>
                                        <option value="500"  <?php selected( $per_page_type, '500' ); ?>><?php esc_html_e( '500 rows per page', 'azayem-bots-tracker' ); ?></option>
                                        <option value="1000" <?php selected( $per_page_type, '1000' ); ?>><?php esc_html_e( '1000 rows per page', 'azayem-bots-tracker' ); ?></option>
                                        <option value="custom" <?php selected( $per_page_type, 'custom' ); ?>><?php esc_html_e( 'Custom…', 'azayem-bots-tracker' ); ?></option>
                                    </select>

                                    <div id="bt-per-page-custom-wrap" class="bt-per-page-custom-wrap<?php echo esc_attr( $per_page_custom_class ); ?>">
                                        <label>
                                            <?php esc_html_e( 'Custom rows count:', 'azayem-bots-tracker' ); ?>
                                            <input type="number"
                                                   name="bt_per_page_custom"
                                                   min="1"
                                                   max="5000"
                                                   value="<?php echo ( 'custom' === $per_page_type && $per_page_custom > 0 ) ? esc_attr( $per_page_custom ) : ''; ?>"
                                                   class="bt-per-page-custom-input" />
                                        </label>
                                    </div>

                                    <p class="description">
                                        <?php esc_html_e(
                                            'Choose how many rows to display on each page, or use a custom value (up to 5000 rows).',
                                            'azayem-bots-tracker'
                                        ); ?>
                                    </p>
                                </fieldset>
                            </td>
                        </tr>

                        <tr>
                            <th scope="row">
                                <?php esc_html_e( 'Search by URL:', 'azayem-bots-tracker' ); ?>
                            </th>
                            <td>
                                <input type="text"
                                       name="bt_search_url"
                                       value="<?php echo esc_attr( $search_display ); ?>"
                                       placeholder="<?php echo esc_attr__( 'https://www.example.com/...', 'azayem-bots-tracker' ); ?>"
                                       class="bt-bots-url-input" />

                                <p class="description">
                                    <?php esc_html_e( 'You can write or paste the full URL or just the path.', 'azayem-bots-tracker' ); ?>
                                </p>
                            </td>
                        </tr>

                        </tbody>
                    </table>

                    <p class="bt-bots-actions">
                        <button type="submit" class="button button-primary">
                            <?php esc_html_e( 'Apply filters', 'azayem-bots-tracker' ); ?>
                        </button>
                        <a href="<?php echo esc_url( admin_url( 'admin.php?page=' . self::PAGE_SLUG ) ); ?>" class="button">
                            <?php esc_html_e( 'Reset filters', 'azayem-bots-tracker' ); ?>
                        </a>
                    </p>
                </form>

                <!-- فورم التصدير CSV (بدون تغيير) -->
                <?php $export_form_action = admin_url( 'admin-post.php' ); ?>
                <form method="get"
                      action="<?php echo esc_url( $export_form_action ); ?>"
                      class="bt-bots-export-form">

                    <?php wp_nonce_field( 'bt_bots_export_crawl_csv', 'bt_bots_export_crawl_nonce' ); ?>

                    <input type="hidden" name="action" value="bt_bots_export_crawl_csv" />

                    <fieldset class="bt-export-fieldset">
                        <legend class="bt-export-legend">
                            <?php esc_html_e( 'CSV date range:', 'azayem-bots-tracker' ); ?>
                        </legend>

                        <?php
                        $export_period_type = isset( $_GET['bt_export_period_type'] )
                            ? sanitize_text_field( wp_unslash( $_GET['bt_export_period_type'] ) )
                            : '30';

                        $export_custom_from = isset( $_GET['bt_export_date_from'] )
                            ? sanitize_text_field( wp_unslash( $_GET['bt_export_date_from'] ) )
                            : '';

                        $export_custom_to = isset( $_GET['bt_export_date_to'] )
                            ? sanitize_text_field( wp_unslash( $_GET['bt_export_date_to'] ) )
                            : '';

                        $export_dates_class = ( 'custom' === $export_period_type ) ? '' : ' bt-is-hidden';
                        ?>

                        <select name="bt_export_period_type" id="bt_export_period_type">
                            <option value="1"  <?php selected( $export_period_type, '1' ); ?>><?php esc_html_e( 'Last 24 hours', 'azayem-bots-tracker' ); ?></option>
                            <option value="30"  <?php selected( $export_period_type, '30' ); ?>><?php esc_html_e( 'Last 30 days', 'azayem-bots-tracker' ); ?></option>
                            <option value="60"  <?php selected( $export_period_type, '60' ); ?>><?php esc_html_e( 'Last 2 months (60 days)', 'azayem-bots-tracker' ); ?></option>
                            <option value="90"  <?php selected( $export_period_type, '90' ); ?>><?php esc_html_e( 'Last 3 months (90 days)', 'azayem-bots-tracker' ); ?></option>
                            <option value="180" <?php selected( $export_period_type, '180' ); ?>><?php esc_html_e( 'Last 6 months (180 days)', 'azayem-bots-tracker' ); ?></option>
                            <option value="custom" <?php selected( $export_period_type, 'custom' ); ?>><?php esc_html_e( 'Custom range', 'azayem-bots-tracker' ); ?></option>
                        </select>

                        <div id="bt-export-custom-dates" class="bt-export-custom-dates<?php echo esc_attr( $export_dates_class ); ?>">
                            <label>
                                <?php esc_html_e( 'To:', 'azayem-bots-tracker' ); ?>
                                <input type="date" name="bt_export_date_to" value="<?php echo esc_attr( $export_custom_to ); ?>" />
                            </label>
                            &nbsp;&nbsp;
                            <label>
                                <?php esc_html_e( 'From:', 'azayem-bots-tracker' ); ?>
                                <input type="date" name="bt_export_date_from" value="<?php echo esc_attr( $export_custom_from ); ?>" />
                            </label>
                        </div>

                        <p class="description">
                            <?php esc_html_e(
                                'This date range applies only to the exported CSV file. It does not affect the on-screen report filters.',
                                'azayem-bots-tracker'
                            ); ?>
                        </p>
                    </fieldset>

                    <fieldset class="bt-export-fieldset">
                        <legend class="bt-export-legend">
                            <?php esc_html_e( 'Export as CSV', 'azayem-bots-tracker' ); ?>
                        </legend>

                        <label class="bt-export-radio">
                            <input type="radio" name="bt_export_scope" value="all" checked="checked" />
                            <?php esc_html_e( 'All bots', 'azayem-bots-tracker' ); ?>
                        </label>

                        <label class="bt-export-radio">
                            <input type="radio" name="bt_export_scope" value="bot" />
                            <?php esc_html_e( 'Specific bot:', 'azayem-bots-tracker' ); ?>

                            <select name="bt_export_bot_name"
                                    id="bt_export_bot_name"
                                    class="bt-export-bot-select"
                                    disabled>
                                <option value=""><?php esc_html_e( 'Select a bot…', 'azayem-bots-tracker' ); ?></option>
                                <?php if ( ! empty( $all_bots ) ) : ?>
                                    <?php foreach ( $all_bots as $bot ) : ?>
                                        <option value="<?php echo esc_attr( $bot ); ?>">
                                            <?php echo esc_html( $bot ); ?>
                                        </option>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </select>
                        </label>

                        <p class="description">
                            <?php esc_html_e( '• "All bots" will export all crawl data in the selected date range.', 'azayem-bots-tracker' ); ?><br />
                            <?php esc_html_e( '• "Specific bot" will export only URLs crawled by the selected bot in that date range.', 'azayem-bots-tracker' ); ?>
                        </p>
                    </fieldset>

                    <input type="hidden" name="bt_orderby" value="<?php echo esc_attr( $orderby ); ?>" />
                    <input type="hidden" name="bt_order" value="<?php echo esc_attr( $order ); ?>" />

                    <p class="bt-export-actions">
                        <button type="submit" class="button button-secondary">
                            <?php esc_html_e( 'Download CSV', 'azayem-bots-tracker' ); ?>
                        </button>
                    </p>
                </form>

            </div><!-- .bt-bots-crawl-forms-row -->

            <div class="bt-bots-summary-grid">
                <div class="bt-bots-summary-card">
                    <div class="bt-bots-summary-label"><?php esc_html_e( 'Current range', 'azayem-bots-tracker' ); ?></div>
                    <div class="bt-bots-summary-value"><?php echo esc_html( $range_label ); ?></div>
                </div>

                <div class="bt-bots-summary-card">
                    <div class="bt-bots-summary-label"><?php esc_html_e( 'To date', 'azayem-bots-tracker' ); ?></div>
                    <div class="bt-bots-summary-value"><?php echo esc_html( $date_to ); ?></div>
                </div>

                <div class="bt-bots-summary-card">
                    <div class="bt-bots-summary-label"><?php esc_html_e( 'From date', 'azayem-bots-tracker' ); ?></div>
                    <div class="bt-bots-summary-value"><?php echo esc_html( $date_from ); ?></div>
                </div>

                <div class="bt-bots-summary-card">
                    <div class="bt-bots-summary-label"><?php esc_html_e( 'Total URLs', 'azayem-bots-tracker' ); ?></div>
                    <div class="bt-bots-summary-value"><?php echo esc_html( $total_urls ); ?></div>
                </div>

                <div class="bt-bots-summary-card">
                    <div class="bt-bots-summary-label"><?php esc_html_e( 'Total visits in range', 'azayem-bots-tracker' ); ?></div>
                    <div class="bt-bots-summary-value"><?php echo esc_html( $total_visits ); ?></div>
                </div>

                <div class="bt-bots-summary-card">
                    <div class="bt-bots-summary-label"><?php esc_html_e( 'Rows per page', 'azayem-bots-tracker' ); ?></div>
                    <div class="bt-bots-summary-value"><?php echo esc_html( $per_page ); ?></div>
                </div>
            </div>

            <?php if ( empty( $rows ) ) : ?>
                <div class="notice notice-warning bt-bots-notice-space">
                    <p><?php esc_html_e( 'No visits were found for the current filters and date range.', 'azayem-bots-tracker' ); ?></p>
                </div>
            <?php else : ?>

                <?php
                $build_sort_th = function( $key, $label ) use ( $orderby, $order, $ordering_base_url ) {
                    $col_class  = 'manage-column sortable';
                    $next_order = 'ASC';

                    if ( $orderby === $key ) {
                        $col_class .= ' sorted ' . strtolower( $order );
                        $next_order = ( 'ASC' === $order ) ? 'DESC' : 'ASC';
                    }

                    $sort_url = add_query_arg(
                        array(
                            'bt_orderby' => $key,
                            'bt_order'   => $next_order,
                            'bt_paged'   => 1,
                        ),
                        $ordering_base_url
                    );

                    echo '<th scope="col" class="' . esc_attr( $col_class ) . '">';
                    echo '<a href="' . esc_url( $sort_url ) . '">';
                    echo '<span>' . esc_html( $label ) . '</span>';
                    echo '<span class="sorting-indicator"></span>';
                    echo '</a>';
                    echo '</th>';
                };
                ?>

                <table class="widefat fixed striped bt-bots-table bt-bots-table-crawl">
                    <thead>
                    <tr>
                        <th style="width:60px;"><?php esc_html_e( '#', 'azayem-bots-tracker' ); ?></th>
                        <th style="width:30%;"><?php esc_html_e( 'Path', 'azayem-bots-tracker' ); ?></th>
                        <th style="width:90px;"><?php esc_html_e( 'File type', 'azayem-bots-tracker' ); ?></th>
                        <?php $build_sort_th( 'hits', __( 'Bot hits', 'azayem-bots-tracker' ) ); ?>
                        <th style="width:140px;"><?php esc_html_e( 'Share of hits %', 'azayem-bots-tracker' ); ?></th>
                        <th style="width:160px;"><?php esc_html_e( 'Crawl frequency', 'azayem-bots-tracker' ); ?></th>
                        <?php
                        $build_sort_th( 'last_visit', __( 'Last crawl', 'azayem-bots-tracker' ) );
                        $build_sort_th( 'last_status', __( 'Last status', 'azayem-bots-tracker' ) );
                        ?>
                    </tr>
                    </thead>
                    <tbody>
                    <?php
                    $i = $offset + 1;

                    foreach ( $rows as $row ) :
                        $url         = isset( $row['url_visited'] ) ? $row['url_visited'] : '';
                        $hits        = isset( $row['visits'] ) ? (int) $row['visits'] : 0;
                        $last_visit  = isset( $row['last_visit'] ) ? $row['last_visit'] : '';
                        $last_status = isset( $row['last_status'] ) ? $row['last_status'] : '';

                        if ( ! empty( $last_visit ) ) {
                            $ts = strtotime( $last_visit );
                            if ( $ts ) {
                                $last_visit = date_i18n( 'Y-m-d g:i:s A', $ts );
                            }
                        }

                        $path         = '';
                        $display_path = '';
                        $link_url     = '';

                        if ( ! empty( $url ) ) {
                            if ( 0 === strpos( $url, '/' ) ) {
                                $path         = $url;
                                $display_path = rawurldecode( $url );
                                $link_url     = home_url( $url );
                            } else {
                                $parts = wp_parse_url( $url );
                                $path_part  = isset( $parts['path'] ) ? $parts['path'] : '';
                                $query_part = ( isset( $parts['query'] ) && '' !== $parts['query'] ) ? '?' . $parts['query'] : '';

                                $path_with_query = $path_part . $query_part;
                                if ( '' === $path_with_query ) {
                                    $path_with_query = '/';
                                }

                                $path         = $path_part;
                                $display_path = rawurldecode( $path_with_query );
                                $link_url     = $url;
                            }
                        }

                        $ext = pathinfo( $path, PATHINFO_EXTENSION );
                        $file_type = ( '' === $ext ) ? 'html' : strtolower( $ext );

                        $share = '0%';
                        if ( $hits > 0 && $total_visits > 0 ) {
                            $share = round( ( $hits / $total_visits ) * 100, 2 ) . '%';
                        }

                        $freq_label = self::calculate_crawl_frequency_label( $hits, $date_from, $date_to );
                        ?>
                        <tr>
                            <td><?php echo esc_html( $i++ ); ?></td>
                            <td>
                                <?php if ( ! empty( $link_url ) ) : ?>
                                    <a href="<?php echo esc_url( $link_url ); ?>" target="_blank" class="bt-bots-url-link">
                                        <?php echo esc_html( $display_path ); ?>
                                    </a>
                                <?php else : ?>
                                    <span class="bt-bots-url-link bt-bots-url-text"><?php echo esc_html( $display_path ); ?></span>
                                <?php endif; ?>
                            </td>
                            <td><?php echo esc_html( $file_type ); ?></td>
                            <td><?php echo esc_html( $hits ); ?></td>
                            <td><?php echo esc_html( $share ); ?></td>
                            <td><?php echo esc_html( $freq_label ); ?></td>
                            <td><?php echo esc_html( $last_visit ); ?></td>
                            <td><?php echo esc_html( $last_status ); ?></td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>

                <?php if ( $total_pages > 1 ) : ?>
                    <div class="tablenav bt-bots-tablenav">
                        <div class="tablenav-pages">
                            <?php
                            $base_url = $pagination_base_url;

                            echo '<span class="pagination-links">';

                            if ( $current_page > 1 ) {
                                $prev_url = add_query_arg( 'bt_paged', $current_page - 1, $base_url );
                                echo '<a class="prev-page button" href="' . esc_url( $prev_url ) . '">&laquo; ' . esc_html__( 'Previous', 'azayem-bots-tracker' ) . '</a> ';
                            } else {
                                echo '<span class="tablenav-pages-navspan button disabled">&laquo; ' . esc_html__( 'Previous', 'azayem-bots-tracker' ) . '</span> ';
                            }

                            $start = max( 1, $current_page - 2 );
                            $end   = min( $total_pages, $current_page + 2 );

                            if ( $start > 1 ) {
                                $first_url = add_query_arg( 'bt_paged', 1, $base_url );
                                echo '<a class="button" href="' . esc_url( $first_url ) . '">1</a> ';
                                if ( $start > 2 ) {
                                    echo '<span class="tablenav-pages-navspan">…</span> ';
                                }
                            }

                            for ( $p = $start; $p <= $end; $p++ ) {
                                if ( $p === $current_page ) {
                                    echo '<span class="button button-primary">' . esc_html( $p ) . '</span> ';
                                } else {
                                    $p_url = add_query_arg( 'bt_paged', $p, $base_url );
                                    echo '<a class="button" href="' . esc_url( $p_url ) . '">' . esc_html( $p ) . '</a> ';
                                }
                            }

                            if ( $end < $total_pages ) {
                                if ( $end < $total_pages - 1 ) {
                                    echo '<span class="tablenav-pages-navspan">…</span> ';
                                }
                                $last_url = add_query_arg( 'bt_paged', $total_pages, $base_url );
                                echo '<a class="button" href="' . esc_url( $last_url ) . '">' . esc_html( $total_pages ) . '</a> ';
                            }

                            if ( $current_page < $total_pages ) {
                                $next_url = add_query_arg( 'bt_paged', $current_page + 1, $base_url );
                                echo '<a class="next-page button" href="' . esc_url( $next_url ) . '">' . esc_html__( 'Next', 'azayem-bots-tracker' ) . ' &raquo;</a>';
                            } else {
                                echo '<span class="tablenav-pages-navspan button disabled">' . esc_html__( 'Next', 'azayem-bots-tracker' ) . ' &raquo;</span>';
                            }

                            echo '</span>';
                            ?>
                        </div>
                    </div>
                <?php endif; ?>

            <?php endif; ?>

        </div>
        <?php
    }

    private static function calculate_crawl_frequency_label( $hits, $date_from, $date_to ) {
        if ( $hits <= 1 ) {
            return __( 'N/A', 'azayem-bots-tracker' );
        }

        if ( empty( $date_from ) || empty( $date_to ) ) {
            return __( 'N/A', 'azayem-bots-tracker' );
        }

        $from_ts = strtotime( $date_from . ' 00:00:00' );
        $to_ts   = strtotime( $date_to . ' 23:59:59' );

        if ( ! $from_ts || ! $to_ts || $to_ts <= $from_ts ) {
            return __( 'N/A', 'azayem-bots-tracker' );
        }

        $days_range = ( $to_ts - $from_ts ) / DAY_IN_SECONDS;
        if ( $days_range <= 0 ) {
            return __( 'N/A', 'azayem-bots-tracker' );
        }

        $avg_per_day = $hits / $days_range;
        if ( $avg_per_day <= 0 ) {
            return __( 'N/A', 'azayem-bots-tracker' );
        }

        $hours_per_visit = 24 / $avg_per_day;

        if ( $hours_per_visit <= 1.5 ) {
            return __( 'Every 1 hour', 'azayem-bots-tracker' );
        }

        if ( $hours_per_visit < 24 ) {
            $h = (int) round( $hours_per_visit );
            /* translators: %d = number of hours */
            return sprintf( __( 'Every %d hours', 'azayem-bots-tracker' ), $h );
        }

        $days_per_visit = $hours_per_visit / 24;

        if ( $days_per_visit <= 1.5 ) {
            return __( 'Every 1 day', 'azayem-bots-tracker' );
        }

        $d = (int) round( $days_per_visit );
        /* translators: %d = number of days */
        return sprintf( __( 'Every %d days', 'azayem-bots-tracker' ), $d );
    }

    public static function export_csv() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You are not allowed to export this data.', 'azayem-bots-tracker' ) );
        }

        check_admin_referer( 'bt_bots_export_crawl_csv', 'bt_bots_export_crawl_nonce' );

        global $wpdb;

        $table_name = $wpdb->prefix . 'bots_visits';

        $export_scope = isset( $_GET['bt_export_scope'] )
            ? sanitize_text_field( wp_unslash( $_GET['bt_export_scope'] ) )
            : 'all';

        $export_bot_name = isset( $_GET['bt_export_bot_name'] )
            ? sanitize_text_field( wp_unslash( $_GET['bt_export_bot_name'] ) )
            : '';

        $export_period_type = isset( $_GET['bt_export_period_type'] )
            ? sanitize_text_field( wp_unslash( $_GET['bt_export_period_type'] ) )
            : '30';

        $export_custom_from = isset( $_GET['bt_export_date_from'] )
            ? sanitize_text_field( wp_unslash( $_GET['bt_export_date_from'] ) )
            : '';
        $export_custom_to   = isset( $_GET['bt_export_date_to'] )
            ? sanitize_text_field( wp_unslash( $_GET['bt_export_date_to'] ) )
            : '';

        $orderby = isset( $_GET['bt_orderby'] )
            ? sanitize_key( wp_unslash( $_GET['bt_orderby'] ) )
            : 'hits';

        $order = isset( $_GET['bt_order'] )
            ? strtoupper( sanitize_text_field( wp_unslash( $_GET['bt_order'] ) ) )
            : 'DESC';

        if ( ! in_array( $order, array( 'ASC', 'DESC' ), true ) ) {
            $order = 'DESC';
        }

        $allowed_orderby = array( 'path', 'hits', 'last_visit', 'last_status' );
        if ( ! in_array( $orderby, $allowed_orderby, true ) ) {
            $orderby = 'hits';
        }

        $date_from = '';
        $date_to   = '';

        $now_ts = current_time( 'timestamp' );

        if ( 'custom' === $export_period_type && ! empty( $export_custom_from ) && ! empty( $export_custom_to ) ) {
            $df_ts = strtotime( $export_custom_from . ' 00:00:00' );
            $dt_ts = strtotime( $export_custom_to . ' 23:59:59' );

            if ( $df_ts && $dt_ts && $dt_ts >= $df_ts ) {
                $date_from = date_i18n( 'Y-m-d', $df_ts );
                $date_to   = date_i18n( 'Y-m-d', $dt_ts );
            } else {
                $export_period_type = '30';
            }
        }

        if ( 'custom' !== $export_period_type ) {
            $days = (int) $export_period_type;
            if ( ! in_array( $days, array( 1, 30, 60, 90, 180 ), true ) ) {
                $days = 30;
            }

            $date_to_ts   = $now_ts;
            $date_from_ts = $now_ts - ( $days * DAY_IN_SECONDS );

            $date_from = date_i18n( 'Y-m-d', $date_from_ts );
            $date_to   = date_i18n( 'Y-m-d', $date_to_ts );
        }

        $date_from_full = $date_from . ' 00:00:00';
        $date_to_full   = $date_to . ' 23:59:59';

        $where_sql    = 'WHERE visit_time BETWEEN %s AND %s';
        $where_params = array( $date_from_full, $date_to_full );

        $bot_filter = '';
        if ( 'bot' === $export_scope && ! empty( $export_bot_name ) ) {
            $bot_filter     = $export_bot_name;
            $where_sql     .= ' AND bot_name = %s';
            $where_params[] = $bot_filter;
        }

        $total_visits = (int) $wpdb->get_var(
            $wpdb->prepare(
                "
                SELECT COUNT(*)
                FROM {$table_name}
                {$where_sql}
                ",
                $where_params
            )
        );

        switch ( $orderby ) {
            case 'path':
                $order_by_sql = 't.url_visited';
                break;
            case 'last_visit':
                $order_by_sql = 't.last_visit';
                break;
            case 'last_status':
                $order_by_sql = 'last_status';
                break;
            case 'hits':
            default:
                $order_by_sql = 't.visits';
                break;
        }

        $data_prepared = $wpdb->prepare(
            "
            SELECT
                t.url_visited,
                t.visits,
                t.last_visit,
                v.status_code AS last_status
            FROM (
                SELECT
                    url_visited,
                    COUNT(*)        AS visits,
                    MAX(visit_time) AS last_visit
                FROM {$table_name}
                {$where_sql}
                GROUP BY url_visited
            ) AS t
            JOIN {$table_name} AS v
              ON v.url_visited = t.url_visited
             AND v.visit_time  = t.last_visit
            ORDER BY {$order_by_sql} {$order}, t.url_visited ASC
            ",
            $where_params
        );

        $rows = $wpdb->get_results( $data_prepared, ARRAY_A );

        nocache_headers();
        header( 'Content-Type: text/csv; charset=utf-8' );

        $suffix = 'all-bots';
        if ( ! empty( $bot_filter ) ) {
            $clean_bot = preg_replace( '/[^a-zA-Z0-9_\-]+/', '-', $bot_filter );
            $suffix    = 'bot-' . strtolower( $clean_bot );
        }
        $filename = 'bots-crawl-budget-' . $suffix . '-' . gmdate( 'Y-m-d' ) . '.csv';
        header( 'Content-Disposition: attachment; filename=' . $filename );

        echo "\xEF\xBB\xBF";

        $output = fopen( 'php://output', 'w' );

        fputcsv(
            $output,
            array(
                '#',
                __( 'Path', 'azayem-bots-tracker' ),
                __( 'File type', 'azayem-bots-tracker' ),
                __( 'Bot hits', 'azayem-bots-tracker' ),
                __( 'Share of hits %', 'azayem-bots-tracker' ),
                __( 'Crawl frequency', 'azayem-bots-tracker' ),
                __( 'Last crawl', 'azayem-bots-tracker' ),
                __( 'Last status', 'azayem-bots-tracker' ),
            )
        );

        $index = 1;

        if ( ! empty( $rows ) ) {
            foreach ( $rows as $row ) {
                $url         = isset( $row['url_visited'] ) ? $row['url_visited'] : '';
                $hits        = isset( $row['visits'] ) ? (int) $row['visits'] : 0;
                $last_visit  = isset( $row['last_visit'] ) ? $row['last_visit'] : '';
                $last_status = isset( $row['last_status'] ) ? $row['last_status'] : '';

                $path         = '';
                $display_path = '';
                $link_url     = '';

                if ( ! empty( $url ) ) {
                    if ( 0 === strpos( $url, '/' ) ) {
                        $path         = $url;
                        $display_path = rawurldecode( $url );
                        $link_url     = home_url( $url );
                    } else {
                        $parts = wp_parse_url( $url );
                        $path_part  = isset( $parts['path'] ) ? $parts['path'] : '';
                        $query_part = ( isset( $parts['query'] ) && '' !== $parts['query'] ) ? '?' . $parts['query'] : '';

                        $path_with_query = $path_part . $query_part;
                        if ( '' === $path_with_query ) {
                            $path_with_query = '/';
                        }

                        $path         = $path_part;
                        $display_path = rawurldecode( $path_with_query );
                        $link_url     = $url;
                    }
                }

                $ext = pathinfo( $path, PATHINFO_EXTENSION );
                $file_type = ( '' === $ext ) ? 'html' : strtolower( $ext );

                $share = '0%';
                if ( $hits > 0 && $total_visits > 0 ) {
                    $share = round( ( $hits / $total_visits ) * 100, 2 ) . '%';
                }

                $freq_label = self::calculate_crawl_frequency_label( $hits, $date_from, $date_to );

                $csv_path_value = $display_path;

                if ( ! empty( $link_url ) && ! empty( $display_path ) ) {
                    $safe_url  = str_replace( '"', '""', $link_url );
                    $safe_text = str_replace( '"', '""', $display_path );

                    $csv_path_value = '=HYPERLINK("' . $safe_url . '","' . $safe_text . '")';
                }

                fputcsv(
                    $output,
                    array(
                        $index++,
                        $csv_path_value,
                        $file_type,
                        $hits,
                        $share,
                        $freq_label,
                        $last_visit,
                        $last_status,
                    )
                );
            }
        }

        exit;
    }
}

add_action(
    'admin_post_bt_bots_export_crawl_csv',
    array( 'BT_Bots_Tracker_Page_Crawl', 'export_csv' )
);
