<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class AYR_SEO_Bridge {

    public static function init() {
        add_action( 'rest_api_init', array( __CLASS__, 'register_routes' ) );
        add_action( 'rest_api_init', array( __CLASS__, 'register_legacy_routes' ) );
    }

    /**
     * Ruta principal: /wp-json/ayrseo/v1/update
     */
    public static function register_routes() {
        register_rest_route( 'ayrseo/v1', '/update', array(
            'methods'  => 'POST',
            'callback' => array( __CLASS__, 'update_meta' ),
            'permission_callback' => array( __CLASS__, 'check_permissions' ),
            'args' => array(
                'post_id' => array(
                    'required' => true,
                    'validate_callback' => function( $param, $request, $key ) {
                        return is_numeric( $param );
                    }
                ),
                'focus_keyword' => array(
                    'sanitize_callback' => 'sanitize_text_field',
                ),
                'meta_description' => array(
                    'sanitize_callback' => 'sanitize_textarea_field',
                )
            )
        ));
    }

    /**
     * Ruta de compatibilidad: /wp-json/custom/v1/update-yoast-meta
     */
    public static function register_legacy_routes() {
        register_rest_route( 'custom/v1', '/update-yoast-meta', array(
            'methods'  => 'POST',
            'callback' => array( __CLASS__, 'update_meta' ),
            'permission_callback' => array( __CLASS__, 'check_permissions' ),
            'args' => array(
                'post_id' => array(
                    'required' => true,
                    'validate_callback' => function( $param, $request, $key ) {
                        return is_numeric( $param );
                    }
                ),
                'focus_keyword' => array(
                    'sanitize_callback' => 'sanitize_text_field',
                ),
                'meta_description' => array(
                    'sanitize_callback' => 'sanitize_textarea_field',
                )
            )
        ));
    }

    /**
     * Verificar permisos y rate limiting
     */
    public static function check_permissions( $request ) {
        // Verificar autenticación
        if ( ! is_user_logged_in() ) {
            return new WP_Error( 
                'unauthorized', 
                __( 'Authentication required', 'ayr-seo-bridge' ), 
                array( 'status' => 401 ) 
            );
        }

        // Rate limiting
        $user_id = get_current_user_id();
        $transient_key = 'ayr_seo_rate_limit_' . $user_id;
        
        if ( get_transient( $transient_key ) ) {
            return new WP_Error( 
                'rate_limit', 
                __( 'Too many requests. Please wait before trying again.', 'ayr-seo-bridge' ), 
                array( 'status' => 429 ) 
            );
        }

        // Establecer rate limit (1 petición por minuto)
        set_transient( $transient_key, true, 60 );

        // Verificar capacidades básicas
        if ( ! current_user_can( 'edit_posts' ) ) {
            return new WP_Error( 
                'forbidden', 
                __( 'Insufficient permissions', 'ayr-seo-bridge' ), 
                array( 'status' => 403 ) 
            );
        }

        return true;
    }

    /**
     * Función que actualiza metadatos SEO según el plugin activo
     */
    public static function update_meta( $request ) {
        global $wpdb;

        // Obtener y validar parámetros
        $post_id          = intval( $request->get_param('post_id') );
        $focus_keyword    = $request->get_param('focus_keyword');
        $meta_description = $request->get_param('meta_description');

        // Validación del post_id
        if ( empty( $post_id ) ) {
            return new WP_Error( 
                'missing_post_id', 
                __( 'post_id is required', 'ayr-seo-bridge' ), 
                array( 'status' => 400 ) 
            );
        }

        // Verificar que el post existe
        $post = get_post( $post_id );
        if ( ! $post ) {
            return new WP_Error( 
                'invalid_post', 
                __( 'The specified post does not exist', 'ayr-seo-bridge' ), 
                array( 'status' => 404 ) 
            );
        }

        // Verificar permisos específicos para este post
        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            return new WP_Error( 
                'forbidden_post', 
                __( 'You do not have permission to edit this post', 'ayr-seo-bridge' ), 
                array( 'status' => 403 ) 
            );
        }

        // Validar que al menos uno de los campos tenga contenido
        if ( empty( $focus_keyword ) && empty( $meta_description ) ) {
            return new WP_Error( 
                'empty_data', 
                __( 'At least one field (focus_keyword or meta_description) is required', 'ayr-seo-bridge' ), 
                array( 'status' => 400 ) 
            );
        }

        $plugin  = null;
        $updated = false;
        $errors  = array();

        try {
            // --- Yoast SEO ---
            if ( is_plugin_active( 'wordpress-seo/wp-seo.php' ) ) {
                if ( ! empty( $focus_keyword ) ) {
                    update_post_meta( $post_id, '_yoast_wpseo_focuskw', $focus_keyword );
                }
                if ( ! empty( $meta_description ) ) {
                    update_post_meta( $post_id, '_yoast_wpseo_metadesc', $meta_description );
                }
                $plugin  = 'yoast';
                $updated = true;
            }

            // --- Rank Math ---
            if ( is_plugin_active( 'seo-by-rank-math/rank-math.php' ) ) {
                if ( ! empty( $focus_keyword ) ) {
                    update_post_meta( $post_id, 'rank_math_focus_keyword', $focus_keyword );
                }
                if ( ! empty( $meta_description ) ) {
                    update_post_meta( $post_id, 'rank_math_description', $meta_description );
                }
                $plugin  = 'rankmath';
                $updated = true;
            }

            // --- AIOSEO ---
            if ( is_plugin_active( 'all-in-one-seo-pack/all_in_one_seo_pack.php' ) ) {
                $table_name = $wpdb->prefix . 'aioseo_posts';
                
                // Verificar que la tabla existe
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table_name ) ) === $table_name ) {
                    $update_data = array();
                    
                    if ( ! empty( $focus_keyword ) ) {
                        $update_data['keyphrases'] = wp_json_encode( array(
                            'focus' => array(
                                'keyphrase' => $focus_keyword,
                                'score'     => 0
                            )
                        ));
                    }
                    
                    if ( ! empty( $meta_description ) ) {
                        $update_data['description'] = $meta_description;
                    }

                    if ( ! empty( $update_data ) ) {
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                        $result = $wpdb->update(
                            $table_name,
                            $update_data,
                            array( 'post_id' => $post_id ),
                            array( '%s', '%s' ),
                            array( '%d' )
                        );
                        
                        if ( false === $result ) {
                            $errors[] = __( 'Failed to update AIOSEO data', 'ayr-seo-bridge' );
                        }
                    }
                }
                $plugin  = 'aioseo';
                $updated = true;
            }

            // --- SEOPress ---
            if ( is_plugin_active( 'wp-seopress/seopress.php' ) ) {
                if ( ! empty( $focus_keyword ) ) {
                    update_post_meta( $post_id, '_seopress_analysis_target_kw', $focus_keyword );
                }
                if ( ! empty( $meta_description ) ) {
                    update_post_meta( $post_id, '_seopress_titles_desc', $meta_description );
                }
                $plugin  = 'seopress';
                $updated = true;
            }

            // Si no se detectó ningún plugin SEO
            if ( ! $plugin ) {
                return new WP_Error( 
                    'no_seo_plugin', 
                    __( 'No compatible SEO plugin detected. Supported plugins: Yoast SEO, Rank Math, AIOSEO, SEOPress', 'ayr-seo-bridge' ), 
                    array( 'status' => 424 ) 
                );
            }

        } catch ( Exception $e ) {
            return new WP_Error( 
                'update_failed', 
                __( 'An error occurred while updating SEO metadata', 'ayr-seo-bridge' ), 
                array( 'status' => 500 ) 
            );
        }

        // Respuesta exitosa
        $response = array(
            'success'          => $updated,
            'plugin_detected'  => $plugin,
            'post_id'          => $post_id,
            'post_title'       => $post->post_title,
            'focus_keyword'    => $focus_keyword,
            'meta_description' => $meta_description,
            'timestamp'        => current_time( 'mysql' )
        );

        if ( ! empty( $errors ) ) {
            $response['warnings'] = $errors;
        }

        return rest_ensure_response( $response );
    }
}