<?php
/**
 * Plugin Name: AYR SEO Bridge
 * Description: Connect WordPress with automation platforms to automatically update SEO metadata in Yoast SEO, Rank Math, AIOSEO, and SEOPress.
 * Version:     1.0.1
 * Author:      AYR Creations
 * Author URI:  https://ayrcreations.com/
 * License:     GPL2
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: ayr-seo-bridge
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.8
 * Requires PHP: 8.0
 *
 * @package AYR_SEO_Bridge
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
    die;
}

// Plugin version
define( 'AYR_SEO_BRIDGE_VERSION', '1.0.1' );

// Plugin path
define( 'AYR_SEO_BRIDGE_PATH', plugin_dir_path( __FILE__ ) );

// Plugin URL
define( 'AYR_SEO_BRIDGE_URL', plugin_dir_url( __FILE__ ) );

// Plugin basename
define( 'AYR_SEO_BRIDGE_BASENAME', plugin_basename( __FILE__ ) );

/**
 * Check if WordPress version is compatible
 */
function ayr_seo_bridge_check_wp_version() {
    global $wp_version;
    $minimum_wp_version = '5.0';
    
    if ( version_compare( $wp_version, $minimum_wp_version, '<' ) ) {
        deactivate_plugins( AYR_SEO_BRIDGE_BASENAME );
        wp_die( 
            sprintf( 
                /* translators: %1$s: minimum WordPress version, %2$s: current WordPress version */
                esc_html__( 'AYR SEO Bridge requires WordPress %1$s or higher. You are running version %2$s. Please upgrade WordPress.', 'ayr-seo-bridge' ), 
                esc_html( $minimum_wp_version ), 
                esc_html( $wp_version )
            ) 
        );
    }
}

/**
 * Check if PHP version is compatible
 */
function ayr_seo_bridge_check_php_version() {
    $minimum_php_version = '8.0';
    
    if ( version_compare( PHP_VERSION, $minimum_php_version, '<' ) ) {
        deactivate_plugins( AYR_SEO_BRIDGE_BASENAME );
        wp_die( 
            sprintf( 
                /* translators: %1$s: minimum PHP version, %2$s: current PHP version */
                esc_html__( 'AYR SEO Bridge requires PHP %1$s or higher. You are running version %2$s. Please upgrade PHP.', 'ayr-seo-bridge' ), 
                esc_html( $minimum_php_version ), 
                esc_html( PHP_VERSION )
            ) 
        );
    }
}

/**
 * Plugin activation hook
 */
function ayr_seo_bridge_activate() {
    // Check system requirements
    ayr_seo_bridge_check_wp_version();
    ayr_seo_bridge_check_php_version();
    
    // Set plugin version option
    add_option( 'ayr_seo_bridge_version', AYR_SEO_BRIDGE_VERSION );
}
register_activation_hook( __FILE__, 'ayr_seo_bridge_activate' );

/**
 * Plugin deactivation hook
 */
function ayr_seo_bridge_deactivate() {
    // Clean up transients
    global $wpdb;
    
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $wpdb->query( 
        $wpdb->prepare(
            "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
            '_transient_ayr_seo_rate_limit_%'
        )
    );
    
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $wpdb->query( 
        $wpdb->prepare(
            "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
            '_transient_timeout_ayr_seo_rate_limit_%'
        )
    );
}
register_deactivation_hook( __FILE__, 'ayr_seo_bridge_deactivate' );

/**
 * Load the main plugin class
 */
require_once AYR_SEO_BRIDGE_PATH . 'includes/class-ayr-seo-bridge.php';

/**
 * Initialize the plugin
 */
function ayr_seo_bridge_init() {
    // Check system requirements on every load
    ayr_seo_bridge_check_wp_version();
    ayr_seo_bridge_check_php_version();
    
    // Initialize the main class
    AYR_SEO_Bridge::init();
}
add_action( 'plugins_loaded', 'ayr_seo_bridge_init' );

/**
 * Add plugin action links
 */
function ayr_seo_bridge_plugin_action_links( $links ) {
    $settings_link = sprintf(
        '<a href="%s" target="_blank">%s</a>',
        'https://ayrcreations.com/presencia-en-redes-sociales',
        __( 'Documentation', 'ayr-seo-bridge' )
    );
    
    $contact_link = sprintf(
        '<a href="%s" target="_blank">%s</a>',
        'https://ayrcreations.com/#contacto',
        __( 'Contact', 'ayr-seo-bridge' )
    );
    
    array_unshift( $links, $settings_link, $contact_link );
    
    return $links;
}
add_filter( 'plugin_action_links_' . AYR_SEO_BRIDGE_BASENAME, 'ayr_seo_bridge_plugin_action_links' );

/**
 * Add plugin meta links
 */
function ayr_seo_bridge_plugin_row_meta( $links, $file ) {
    if ( AYR_SEO_BRIDGE_BASENAME === $file ) {
        $row_meta = array(
            'website' => sprintf(
                '<a href="%s" target="_blank">%s</a>',
                'https://ayrcreations.com/',
                __( 'AYR Creations', 'ayr-seo-bridge' )
            ),
            'contact' => sprintf(
                '<a href="%s" target="_blank">%s</a>',
                'https://ayrcreations.com/#contacto',
                __( 'Contact Us', 'ayr-seo-bridge' )
            )
        );
        
        return array_merge( $links, $row_meta );
    }
    
    return $links;
}
add_filter( 'plugin_row_meta', 'ayr_seo_bridge_plugin_row_meta', 10, 2 );