<?php
if(!defined('ABSPATH')) exit;

/**
 * 1. ADD FIELDS TO CHECKOUT
 */
add_filter('woocommerce_checkout_fields', function($fields) {
    // UPDATED: Using the correct option key from your settings page
    $settings = get_option('awdts_settings', []);
    
    $date_label = !empty($settings['checkout_date_label']) ? $settings['checkout_date_label'] : 'Delivery Date';
    $time_label = !empty($settings['checkout_time_label']) ? $settings['checkout_time_label'] : 'Delivery Time';

    // Extract just the dates from the holiday list for the JS
    $holiday_dates = [];
    if (!empty($settings['holiday_list'])) {
        foreach ($settings['holiday_list'] as $h) {
            if (!empty($h['date'])) $holiday_dates[] = $h['date'];
        }
    }

    $custom_attrs = [
        'data-min-days'      => intval($settings['min_delivery_days'] ?? 0),
        'data-max-days'      => intval($settings['max_delivery_days'] ?? 30),
        'data-blocked-days'  => implode(',', $settings['blocked_days'] ?? []),
        'data-holidays'      => implode(',', $holiday_dates), // New Key
        'readonly'           => 'readonly',
        'placeholder'        => __('Select a date...', 'awesome-delivery-scheduler')
    ];

    $fields['billing']['awdts_delivery_date'] = [
        'label'       => esc_html($date_label),
        'required'    => true,
        'class'       => ['form-row-first'],
        'type'        => 'text',
        'custom_attributes' => $custom_attrs
    ];

    $fields['billing']['awdts_delivery_slot'] = [
        'label'       => esc_html($time_label),
        'required'    => true,
        'class'       => ['form-row-last'],
        'type'        => 'select',
        'options'     => ['' => __('Choose date first', 'awesome-delivery-scheduler')],
        'custom_attributes' => [
            'data-slots' => htmlspecialchars(json_encode($settings['slots'] ?? []), ENT_QUOTES, 'UTF-8')
        ]
    ];

    return $fields;
});

/**
 * 2. VALIDATION
 */
add_action('woocommerce_checkout_process', function() {
    if (empty($_POST['awdts_delivery_date'])) {
        wc_add_notice(__('Please select a delivery date.', 'awesome-delivery-scheduler'), 'error');
    }
    if (empty($_POST['awdts_delivery_slot'])) {
        wc_add_notice(__('Please select a delivery time slot.', 'awesome-delivery-scheduler'), 'error');
    }
});

/**
 * 3. SAVE TO ORDER
 */
add_action('woocommerce_checkout_create_order', function($order, $data) {
    if(!empty($_POST['awdts_delivery_date'])) {
        $order->update_meta_data('_awdts_delivery_date', sanitize_text_field($_POST['awdts_delivery_date']));
    }

    if(!empty($_POST['awdts_delivery_slot'])) {
        $settings = get_option('awdts_settings', []);
        $slot_index = intval($_POST['awdts_delivery_slot']);
        $slots = $settings['slots'] ?? [];
        
        if(isset($slots[$slot_index])) {
            $slot = $slots[$slot_index];
            $display = $slot['start_time'] . ' - ' . $slot['end_time'];
            $order->update_meta_data('_awdts_delivery_slot', $display);
        }
    }
}, 10, 2);

/**
 * 4. FRONTEND JAVASCRIPT
 */
add_action('wp_footer', function() {
    if(!is_checkout()) return;
    wp_enqueue_script('jquery-ui-datepicker');
    ?>
    <link rel="stylesheet" href="https://code.jquery.com/ui/1.13.2/themes/base/jquery-ui.css">
    <style>
        #ui-datepicker-div { z-index: 9999 !important; border: none !important; box-shadow: 0 10px 15px -3px rgba(0,0,0,0.1); border-radius: 8px; }
        .ui-datepicker-header { background: #6366f1 !important; color: white !important; border: none !important; border-radius: 8px 8px 0 0; }
    </style>

    <script>
    jQuery(function($) {
        const $dateInput = $('#awdts_delivery_date');
        const $slotInput = $('#awdts_delivery_slot');
        if(!$dateInput.length) return;

        const slotsData    = $slotInput.data('slots') || [];
        const blockedDays  = ($dateInput.data('blocked-days') || '').split(',');
        const holidays     = ($dateInput.data('holidays') || '').split(',');
        const minDays      = parseInt($dateInput.data('min-days') || 0);
        const maxDays      = parseInt($dateInput.data('max-days') || 30);
        const weekdays     = ['Sunday','Monday','Tuesday','Wednesday','Thursday','Friday','Saturday'];

        $dateInput.datepicker({
            dateFormat: 'yy-mm-dd',
            minDate: minDays,
            maxDate: maxDays,
            beforeShowDay: function(date) {
                const dayName = weekdays[date.getDay()];
                const formatted = $.datepicker.formatDate('yy-mm-dd', date);
                // Check if day is blocked or if date is a holiday
                const isBlocked = blockedDays.includes(dayName) || holidays.includes(formatted);
                return [!isBlocked, ""];
            },
            onSelect: function(dateText) {
                updateSlots(dateText);
            }
        });

        function updateSlots(selectedDate) {
            const dateObj = new Date(selectedDate.replace(/-/g, "/")); // Fix cross-browser date parsing
            const dayName = weekdays[dateObj.getDay()];
            
            $slotInput.empty().append('<option value="">Select Time</option>');

            let found = false;
            slotsData.forEach(function(slot, index) {
                if (slot.day === dayName) {
                    $slotInput.append(`<option value="${index}">${slot.start_time} - ${slot.end_time}</option>`);
                    found = true;
                }
            });

            if (!found) {
                $slotInput.html('<option value="">No slots for this day</option>');
            }
        }

        // Auto-select first available date
        let checkDate = new Date();
        checkDate.setDate(checkDate.getDate() + minDays);
        for (let i = 0; i < 60; i++) {
            let dayName = weekdays[checkDate.getDay()];
            let formatted = $.datepicker.formatDate('yy-mm-dd', checkDate);
            if (!blockedDays.includes(dayName) && !holidays.includes(formatted)) {
                $dateInput.datepicker('setDate', checkDate);
                updateSlots(formatted);
                break;
            }
            checkDate.setDate(checkDate.getDate() + 1);
        }
    });
    </script>
    <?php
});