<?php
if (! defined('ABSPATH')) {
    exit;
}

/**
 * Class for registering and handling custom REST endpoints.
 */
class AutopilotSEO_REST
{

    const NAMESPACE = 'autopilotseo/v1';

    public function __construct()
    {
        add_action('rest_api_init', [$this, 'register_routes']);
    }

    /**
     * Register REST routes.
     */
    public function register_routes()
    {
        // Existing push-post route.
        register_rest_route(self::NAMESPACE, '/push-post', [
            'methods'             => 'POST',
            'callback'            => [$this, 'handle_push_post'],
            'permission_callback' => [$this, 'permissions_check'],
        ]);

        // New test-connection route.
        register_rest_route(self::NAMESPACE, '/test-connection', [
            'methods'             => WP_REST_Server::CREATABLE, // POST method
            'callback'            => [$this, 'handle_test_connection'],
            'permission_callback' => [$this, 'permissions_check'],
        ]);

        // New sync-connection route
        register_rest_route(self::NAMESPACE, '/sync-connection', [
            'methods'             => WP_REST_Server::CREATABLE, // POST method
            'callback'            => [$this, 'handle_sync_connection'],
            'permission_callback' => [$this, 'permissions_check'],
        ]);
    }

    /**
     * Permissions check for the push-post route.
     *
     * This callback requires that the request include:
     *   - An API key provided via the Authorization header (Bearer scheme) or as an "api_key" parameter.
     *   - A "site_slug" parameter that matches the stored Website ID.
     *
     * @param WP_REST_Request $request
     * @return bool|WP_Error
     */
    public function permissions_check($request)
    {
        // Get API key.
        $provided_key = null;
        $auth_header  = $request->get_header('authorization');
        if ($auth_header && preg_match('/Bearer\s(\S+)/', $auth_header, $matches)) {
            $provided_key = $matches[1];
        }
        if (! $provided_key) {
            $provided_key = $request->get_param('api_key');
        }

        // Get website slug.
        $provided_slug = $request->get_param('site_slug');

        // Retrieve stored values.
        $stored_key  = get_option(AutopilotSEO_Admin::OPTION_NAME, '');
        $stored_slug = get_option(AutopilotSEO_Admin::WEBSITE_OPTION_NAME, '');

        if ($stored_key && $provided_key === $stored_key && $provided_slug === $stored_slug) {
            return true;
        }

        return new WP_Error(
            'rest_forbidden',
            __('Invalid or missing API key or site slug.', 'autopilotseo-connector'),
            ['status' => 403]
        );
    }

    /**
     * Handles the push-post request from your remote service.
     *
     * Expected request parameters (in JSON):
     *   - site_slug: (string) The website slug (must match the stored Website ID)
     *   - title: (string) The post title.
     *   - content: (string) The post content (HTML).
     *   - status: (string, optional) Post status (default: "draft").
     *   - author_id: (int, optional) The user ID of the post author.
     *   - categories: (array, optional) An array of category IDs or slugs.
     *   - seo_meta: (object, optional) An associative array of SEO-related metadata.
     *
     * @param WP_REST_Request $request
     * @return array|WP_Error
     */
    public function handle_push_post(WP_REST_Request $request)
    {
        // Ensure the WordPress media functions are available
        if (! function_exists('media_sideload_image')) {
            require_once ABSPATH . 'wp-admin/includes/image.php';
            require_once ABSPATH . 'wp-admin/includes/file.php';
            require_once ABSPATH . 'wp-admin/includes/media.php';
        }

        $title   = $request->get_param('title');
        $content = $request->get_param('content');
        $status  = $request->get_param('status') ?: 'draft';
        $seo_meta = $request->get_param('seo_meta') ?: [];
        $featured_image_url = $request->get_param('featured_image');

        $description = $seo_meta['description'] ?? '';

        if (empty($title) || empty($content)) {
            return new WP_Error('rest_missing_params', __('Title and content are required.', 'autopilotseo-connector'), ['status' => 400]);
        }

        $author_id = $request->get_param('author_id') ?: 1;
        if (! get_user_by('id', $author_id)) {
            return new WP_Error('invalid_author', __('Invalid author_id provided.', 'autopilotseo-connector'), ['status' => 400]);
        }

        $converter = new AutopilotSEO_HTML_To_Gutenberg();
        $gutenberg_markup = $converter->convert($content);

        $new_post = [
            'post_title'   => wp_strip_all_tags($title),
            'post_content' => $gutenberg_markup,
            'post_status'  => $status,
            'post_type'    => 'post',
            'post_author'  => $author_id,
            'post_excerpt' => $description,
        ];

        $post_id = wp_insert_post($new_post);
        if (is_wp_error($post_id)) {
            return $post_id;
        }

        // Set featured image (if provided)
        if (! empty($featured_image_url)) {
            $downloaded = $converter->download_external_image_by_url($featured_image_url, $post_id, $title);
            if (!is_wp_error($downloaded)) {
                set_post_thumbnail($post_id, $downloaded['attachment_id']);
            }
        }

        $this->set_seo_metadata($post_id, $title, $description, $seo_meta);

        return [
            'success' => true,
            'post_id' => $post_id,
            'message' => __('Post created successfully!', 'autopilotseo-connector'),
        ];
    }

    /**
     * Parses the HTML content for <img> tags, sideloads each image, 
     * and replaces the external src with the new local URL.
     *
     * @param string $content
     * @return string Modified content with updated image src attributes.
     */
    function handle_inline_images($content, $post_id)
    {
        $converter = new AutopilotSEO_HTML_To_Gutenberg();
        // Regex for <img src="">
        $pattern = '/<img[^>]+src=[\'"]([^\'"]+)[\'"][^>]*>/i';
        if (preg_match_all($pattern, $content, $matches)) {
            $image_urls = $matches[1];
            foreach ($image_urls as $image_url) {
                // Skip local URLs
                if ($this->is_local_url($image_url)) {
                    continue;
                }
                $downloaded = $converter->download_external_image_by_url($image_url, $post_id);
                if (! is_wp_error($downloaded)) {
                    // Replace in content
                    $content = str_replace($image_url, $downloaded['url'], $content);
                }
            }
        }
        return $content;
    }

    private function is_local_url($url)
    {
        $home_url = home_url();
        return (strpos($url, $home_url) !== false);
    }

    private function set_seo_metadata($post_id, $title, $description, $seo_meta)
    {
        $has_yoast    = class_exists('WPSEO_Meta');
        $has_rankmath = class_exists('RankMath');
        $canonical_url = get_permalink($post_id);

        // Format focus keywords
        $focus_keywords = $seo_meta['focusKeywords'] ?? '';
        // If focus_keyword is an array, convert to appropriate format
        if (is_array($focus_keywords)) {
            $rank_math_keywords = implode(',', $focus_keywords);
            $yoast_keyword = reset($focus_keywords) ?: ''; // Get first keyword for Yoast
        } else {
            $rank_math_keywords = $focus_keywords;
            $yoast_keyword = $focus_keywords;
        }

        // Set SEO Meta Fields if Yoast is installed
        if ($has_yoast) {
            update_post_meta($post_id, '_yoast_wpseo_title', $title);
            update_post_meta($post_id, '_yoast_wpseo_metadesc', $description);
            update_post_meta($post_id, '_yoast_wpseo_focuskw', $yoast_keyword); // Single keyword
            update_post_meta($post_id, '_yoast_wpseo_canonical', $canonical_url);
        }

        // Set SEO Meta Fields if Rank Math is installed
        if ($has_rankmath) {
            update_post_meta($post_id, 'rank_math_title', $title);
            update_post_meta($post_id, 'rank_math_description', $description);
            update_post_meta($post_id, 'rank_math_focus_keyword', $rank_math_keywords); // Comma-separated keywords
            update_post_meta($post_id, 'rank_math_canonical_url', $canonical_url);
        }

        // Set Open Graph & Twitter Metadata for Social Media
        $thumbnail_url = get_the_post_thumbnail_url($post_id);
        update_post_meta($post_id, '_yoast_wpseo_opengraph-title', $title);
        update_post_meta($post_id, '_yoast_wpseo_opengraph-description', $description);
        update_post_meta($post_id, '_yoast_wpseo_opengraph-image', $thumbnail_url);

        update_post_meta($post_id, '_yoast_wpseo_twitter-title', $title);
        update_post_meta($post_id, '_yoast_wpseo_twitter-description', $description);
        update_post_meta($post_id, '_yoast_wpseo_twitter-image', $thumbnail_url);
    }

    /**
     * Handles the autopilotseo/test-connection request.
     *
     * This function verifies that the required options are set and accessible,
     * and attempts to connect with the remote service.
     *
     * @param WP_REST_Request $request The current request.
     * @return WP_REST_Response|WP_Error The connection result.
     */
    public function handle_test_connection(WP_REST_Request $request)
    {
        $api_key    = get_option(AutopilotSEO_Admin::OPTION_NAME, '');
        $website_id = get_option(AutopilotSEO_Admin::WEBSITE_OPTION_NAME, '');

        if (empty($api_key) || empty($website_id)) {
            return new WP_Error(
                'missing_credentials',
                __('API key and Website ID are required.', 'autopilotseo-connector'),
                ['status' => 400]
            );
        }

        // Attempt to connect with the remote service
        $connection = AutopilotSEO_Admin::connect_with_service($website_id, $api_key);

        if (is_wp_error($connection)) {
            return $connection; // Return the WP_Error from the connection attempt
        }

        // Update the last verified timestamp on successful connection
        update_option(AutopilotSEO_Admin::LAST_VERIFIED_OPTION_NAME, current_time('mysql'));

        return [
            'success' => true,
            'message' => __('Connection verified successfully!', 'autopilotseo-connector'),
        ];
    }

    /**
     * Handles the sync-connection request.
     * 
     * This endpoint returns the current connection status and site information.
     *
     * @param WP_REST_Request $request The current request.
     * @return WP_REST_Response|WP_Error The sync result.
     */
    public function handle_sync_connection(WP_REST_Request $request)
    {
        $api_key    = get_option(AutopilotSEO_Admin::OPTION_NAME, '');
        $website_id = get_option(AutopilotSEO_Admin::WEBSITE_OPTION_NAME, '');
        $last_verified = get_option(AutopilotSEO_Admin::LAST_VERIFIED_OPTION_NAME, '');

        if (empty($api_key) || empty($website_id)) {
            return new WP_Error(
                'missing_credentials',
                __('API key and Website ID are required.', 'autopilotseo-connector'),
                ['status' => 400]
            );
        }

        // Fetch authors (users who can publish posts)
        $authors = get_users([
            'role__in' => ['administrator', 'editor', 'author'],
            'fields'   => ['ID', 'display_name', 'user_email']
        ]);

        // Format author data
        $author_list = array_map(function ($author) {
            return [
                'id'    => $author->ID,
                'name'  => $author->display_name,
                'email' => $author->user_email
            ];
        }, $authors);

        return new WP_REST_Response([
            'success' => true,
            'message' => __('Site information retrieved successfully!', 'autopilotseo-connector'),
            'last_verified' => $last_verified,
            'site_info' => [
                'website_id' => $website_id,
                'site_url' => get_site_url(),
                'site_name' => get_bloginfo('name'),
                'site_description' => get_bloginfo('description'),
                'site_language' => get_bloginfo('language'),
                'wp_version' => get_bloginfo('version'),
                'site_admin_email' => get_bloginfo('admin_email'),
                'home_url' => get_home_url(),
                'is_multisite' => is_multisite(),
                'authors' => $author_list,
            ]
        ], 200);
    }
}
