<?php
if (! defined('ABSPATH')) {
    exit;
}

/**
 * Download a remote image and create an attachment in the Media Library.
 * Returns an array with 'attachment_id' and 'url' on success, or WP_Error on failure.
 *
 * @param string $url      The remote image URL.
 * @param int    $post_id  Optional. Post to attach the media to (0 = unattached).
 * @param string $desc     Optional. Description (post title) for the attachment.
 */
function download_external_image($url, $post_id = 0, $desc = '')
{

    // 1) Fetch the remote image headers/body
    $response = wp_remote_get($url, ['timeout' => 20]);
    if (is_wp_error($response)) {
        return new WP_Error('http_error', 'Failed to fetch remote URL.', $response->get_error_message());
    }

    $code = wp_remote_retrieve_response_code($response);
    if ($code !== 200) {
        return new WP_Error('http_error', "Image request returned status code $code", $response);
    }

    $body       = wp_remote_retrieve_body($response);
    $content_type = wp_remote_retrieve_header($response, 'content-type');

    if (empty($body)) {
        return new WP_Error('empty_body', 'Remote image body is empty.');
    }


    // 2) Determine extension from the content-type, fallback to .jpg
    // e.g. "image/jpeg" -> ".jpg", "image/png" -> ".png", etc.
    $extension = '.jpg'; // default
    if ($content_type) {
        $mime_to_ext = [
            'image/jpeg' => '.jpg',
            'image/jpg'  => '.jpg',
            'image/png'  => '.png',
            'image/gif'  => '.gif',
            'image/webp' => '.webp',
            // Add more as needed
        ];
        if (isset($mime_to_ext[$content_type])) {
            $extension = $mime_to_ext[$content_type];
        }
    }

    // 3) Create a temporary file with correct extension
    $temp_file = wp_tempnam($url);
    if (! $temp_file) {
        return new WP_Error('temp_file_failed', 'Could not create a temp file.');
    }

    // Initialize WordPress Filesystem
    require_once ABSPATH . 'wp-admin/includes/file.php';
    WP_Filesystem();
    global $wp_filesystem;

    if (! $wp_filesystem) {
        return new WP_Error('filesystem_error', 'Could not initialize WordPress filesystem.');
    }

    // Rename it with our extension so WordPress recognizes the file type
    $temp_with_ext = $temp_file . $extension;
    if (! $wp_filesystem->move($temp_file, $temp_with_ext, true)) {
        return new WP_Error('rename_failed', 'Could not rename temporary file.');
    }
    $wp_filesystem->put_contents($temp_with_ext, $body);

    // 4) Use wp_handle_sideload() to move it into wp-content/uploads
    $file_array = [
        'name'     => basename($temp_with_ext), // final file name
        'tmp_name' => $temp_with_ext,
    ];
    // ensure media functions are loaded
    if (! function_exists('wp_handle_sideload')) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/media.php';
    }
    $sideload = wp_handle_sideload($file_array, ['test_form' => false]);

    if (isset($sideload['error'])) {
        return new WP_Error('sideload_error', $sideload['error']);
    }

    // 5) Build attachment post data and insert it
    $attachment = [
        'post_mime_type' => $sideload['type'],
        'post_title'     => ($desc ? $desc : basename($sideload['file'])),
        'post_content'   => '',
        'post_status'    => 'inherit',
    ];
    $attach_id = wp_insert_attachment($attachment, $sideload['file'], $post_id);
    if (is_wp_error($attach_id)) {
        return $attach_id; // pass up the error
    }

    // Generate image metadata
    require_once ABSPATH . 'wp-admin/includes/image.php';
    $attach_data = wp_generate_attachment_metadata($attach_id, $sideload['file']);
    wp_update_attachment_metadata($attach_id, $attach_data);

    return [
        'attachment_id' => $attach_id,
        'url'           => wp_get_attachment_url($attach_id),
    ];
}

/**
 * Checks if an attachment with the given remote URL already exists.
 *
 * @param string $url The remote image URL.
 * @return int The attachment ID if found, or 0 if not found.
 */
function find_attachment_by_remote_url($url)
{
    $args = [
        'post_type'      => 'attachment',
        'post_status'    => 'any',
        'meta_query'     => [
            [
                'key'   => '_remote_url',
                'value' => $url,
            ],
        ],
        'fields'         => 'ids',
        'posts_per_page' => 1,
    ];
    $query = new WP_Query($args);

    if (! empty($query->posts)) {
        return (int) $query->posts[0];
    }
    return 0;
}
