<?php
if (! defined('ABSPATH')) {
    exit;
}

/**
 * Handles the Admin Menu for AutopilotSEO.
 */
class AutopilotSEO_Admin_Menu
{

    public function __construct()
    {
        add_action('admin_menu', [$this, 'add_settings_page']);
    }

    /**
     * Add the settings page under the "Settings" menu.
     */
    public function add_settings_page()
    {
        add_options_page(
            __('AutopilotSEO Connector', 'autopilotseo-connector'),            // Page title.
            __('AutopilotSEO Connector', 'autopilotseo-connector'),            // Menu title.
            'manage_options',                                                  // Capability.
            'autopilotseo_settings',                                           // Menu slug.
            [$this, 'render_settings_page']                                    // Callback.
        );
    }

    /**
     * Render the settings page.
     */
    public function render_settings_page()
    {
        // Check user capabilities.
        if (! current_user_can('manage_options')) {
            return;
        }

        // Process form submission if it's a POST request.
        if (isset($_SERVER['REQUEST_METHOD']) && 'POST' === $_SERVER['REQUEST_METHOD']) {
            check_admin_referer('autopilotseo_settings_nonce', 'autopilotseo_settings_nonce_field');

            // Determine the action from the hidden field.
            $action = isset($_POST['autopilotseo_action']) ? sanitize_text_field(wp_unslash($_POST['autopilotseo_action'])) : 'connect';

            // For "reverify", use the saved values.
            if ('reverify' === $action) {
                $api_key    = get_option(AutopilotSEO_Admin::OPTION_NAME, '');
                $website_id = get_option(AutopilotSEO_Admin::WEBSITE_OPTION_NAME, '');
            } else {
                // For "connect" (initial connection) or "save" (update) use the submitted values.
                $api_key    = isset($_POST['autopilotseo_api_key']) ? sanitize_text_field(wp_unslash($_POST['autopilotseo_api_key'])) : '';
                $website_id = isset($_POST['autopilotseo_website_id']) ? sanitize_text_field(wp_unslash($_POST['autopilotseo_website_id'])) : '';
            }

            // Validate required fields.
            if (empty($api_key) || empty($website_id)) {
                add_settings_error(
                    'autopilotseo_messages',
                    'autopilotseo_required_error',
                    __('Both API Key and Website ID are required.', 'autopilotseo-connector'),
                    'error'
                );
            } else {
                // Attempt to connect with the remote service.
                $connection = AutopilotSEO_Admin::connect_with_service($website_id, $api_key);
                $existing_last_verified = get_option(AutopilotSEO_Admin::LAST_VERIFIED_OPTION_NAME, '');

                if (! is_wp_error($connection)) {
                    // On successful connection, store the new values and current time.
                    update_option(AutopilotSEO_Admin::OPTION_NAME, $api_key);
                    update_option(AutopilotSEO_Admin::WEBSITE_OPTION_NAME, $website_id);
                    update_option(AutopilotSEO_Admin::LAST_VERIFIED_OPTION_NAME, current_time('mysql'));
                    add_settings_error(
                        'autopilotseo_messages',
                        'autopilotseo_key_saved',
                        __('API Key verified and site connected successfully!', 'autopilotseo-connector'),
                        'updated'
                    );
                } else {
                    // If verification fails, clear the "last verified" timestamp so the UI shows a warning.
                    // For an update ("save") where there is an existing valid connection, keep the saved credentials,
                    // but clear the last verified flag.
                    if ('save' === $action && ! empty($existing_last_verified)) {
                        update_option(AutopilotSEO_Admin::LAST_VERIFIED_OPTION_NAME, '');
                        add_settings_error(
                            'autopilotseo_messages',
                            'autopilotseo_key_error',
                            __('Connection failed: ', 'autopilotseo-connector') . $connection->get_error_message(),
                            'error'
                        );
                    } else {
                        // For initial connection attempts or if no valid saved values exist, clear all options.
                        update_option(AutopilotSEO_Admin::OPTION_NAME, '');
                        update_option(AutopilotSEO_Admin::WEBSITE_OPTION_NAME, '');
                        update_option(AutopilotSEO_Admin::LAST_VERIFIED_OPTION_NAME, '');
                        add_settings_error(
                            'autopilotseo_messages',
                            'autopilotseo_key_error',
                            __('Connection failed: ', 'autopilotseo-connector') . $connection->get_error_message(),
                            'error'
                        );
                    }
                }
            }
        }

        settings_errors('autopilotseo_messages');
        // Retrieve saved values (the last valid credentials).
        $saved_api_key    = get_option(AutopilotSEO_Admin::OPTION_NAME, '');
        $saved_website_id = get_option(AutopilotSEO_Admin::WEBSITE_OPTION_NAME, '');
        $last_verified    = get_option(AutopilotSEO_Admin::LAST_VERIFIED_OPTION_NAME, '');

        // Determine which notice to show:
        // If credentials exist but no valid last_verified, then verification failed.
        if (! empty($saved_api_key) && ! empty($saved_website_id) && empty($last_verified)) {
            $connection_notice = '<div class="notice notice-error autopilotseo-connection-notice"><p>'
                . __('Connection verification failed. Please reverify your credentials.', 'autopilotseo-connector')
                . '</p></div>';
        } elseif (! empty($last_verified)) {
            $connection_notice = '<div class="notice notice-success autopilotseo-connection-notice"><p>'
                . sprintf(
                    /* translators: %s: date and time of last successful connection verification */
                    __('Site connected. Last verified at: %s', 'autopilotseo-connector'),
                    esc_html($last_verified)
                )
                . '</p></div>';
        } else {
            $connection_notice = '';
        }
?>
        <div class="wrap autopilotseo-wrap">
            <h1>AutopilotSEO Settings</h1>

            <?php echo wp_kses_post($connection_notice); ?>

            <form method="post" action="">
                <?php wp_nonce_field('autopilotseo_settings_nonce', 'autopilotseo_settings_nonce_field'); ?>
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="autopilotseo_website_id"><?php esc_html_e('Website ID', 'autopilotseo-connector'); ?></label>
                        </th>
                        <td>
                            <input type="text" id="autopilotseo_website_id" name="autopilotseo_website_id"
                                value="<?php echo esc_attr($saved_website_id); ?>"
                                class="regular-text" <?php echo (! empty($last_verified) ? 'disabled' : ''); ?> />
                            <p class="description"><?php esc_html_e('Enter your unique Website ID provided in your AutopilotSEO dashboard.', 'autopilotseo-connector'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="autopilotseo_api_key"><?php esc_html_e('API Key', 'autopilotseo-connector'); ?></label>
                        </th>
                        <td class="autopilotseo-api-key-wrapper">
                            <input type="<?php echo (! empty($saved_api_key) && empty($_GET['update']) ? 'password' : 'text'); ?>"
                                id="autopilotseo_api_key" name="autopilotseo_api_key"
                                value="<?php echo esc_attr($saved_api_key); ?>"
                                class="regular-text" <?php echo (! empty($last_verified) ? 'disabled' : ''); ?> />
                            <?php if (! empty($saved_api_key)) : ?>
                                <button type="button" id="toggle_api_key" class="button-secondary autopilotseo-toggle-key">Show</button>
                            <?php endif; ?>
                            <p class="description"><?php esc_html_e('Enter your API Key provided in your AutopilotSEO dashboard.', 'autopilotseo-connector'); ?></p>
                        </td>
                    </tr>
                </table>

                <!-- Hidden field to track the action: "connect" (default), "reverify", or "save" (when updating) -->
                <input type="hidden" name="autopilotseo_action" value="connect" id="autopilotseo_action_field" />

                <?php if (! empty($saved_api_key) && ! empty($saved_website_id)) : ?>
                    <!-- Button container for normal (non-editing) mode -->
                    <p id="normal_buttons">
                        <input type="submit" class="button button-primary" value="<?php esc_attr_e('Reverify', 'autopilotseo-connector'); ?>" onclick="document.getElementById('autopilotseo_action_field').value='reverify';" />
                        <button type="button" class="button" id="update_button"><?php esc_html_e('Update', 'autopilotseo-connector'); ?></button>
                    </p>
                    <!-- Button container for editing mode (hidden by default) -->
                    <p id="editing_buttons" style="display: none;">
                        <input type="submit" class="button button-primary" value="<?php esc_attr_e('Save', 'autopilotseo-connector'); ?>" onclick="document.getElementById('autopilotseo_action_field').value='save';" />
                        <button type="button" class="button" id="cancel_button"><?php esc_html_e('Cancel', 'autopilotseo-connector'); ?></button>
                    </p>
                <?php else : ?>
                    <?php submit_button(__('Connect', 'autopilotseo-connector')); ?>
                <?php endif; ?>

                <!-- Hidden fields to store original values for cancellation -->
                <input type="hidden" id="autopilotseo_original_api_key" value="<?php echo esc_attr($saved_api_key); ?>" />
                <input type="hidden" id="autopilotseo_original_website_id" value="<?php echo esc_attr($saved_website_id); ?>" />
            </form>

            <!-- Modal with proper classes -->
            <div id="update_modal" style="display: none;">
                <div class="autopilotseo-modal-overlay"></div>
                <div class="autopilotseo-modal-content">
                    <h2><?php esc_html_e('Warning', 'autopilotseo-connector'); ?></h2>
                    <p><?php esc_html_e('Changing these values will break the current connection. You must verify again to reconnect.', 'autopilotseo-connector'); ?></p>
                    <p class="autopilotseo-modal-actions">
                        <button type="button" class="button button-primary" id="modal_continue"><?php esc_html_e('Continue', 'autopilotseo-connector'); ?></button>
                        <button type="button" class="button" id="modal_cancel"><?php esc_html_e('Cancel', 'autopilotseo-connector'); ?></button>
                    </p>
                </div>
            </div>
        </div>
<?php
    }
}
