<?php
if (! defined('ABSPATH')) {
    exit;
}

/**
 * Handles core admin functionality for the AutopilotSEO plugin.
 */
class AutopilotSEO_Admin
{

    // Option for the API key.
    const OPTION_NAME = 'autopilotseo_api_key';
    // New option for the Website ID (slug) provided by AutopilotSEO.
    const WEBSITE_OPTION_NAME = 'autopilotseo_website_id';
    // Option for storing the last verified timestamp.
    const LAST_VERIFIED_OPTION_NAME = 'autopilotseo_last_verified';
    // (Optional) The original verification URL.
    const VERIFY_URL  = 'https://autopilotseo.ai/api/verify-key';

    public function __construct()
    {
        // Register settings when the admin initializes.
        add_action('admin_init', [$this, 'register_settings']);
    }

    public function sanitize_api_key($value)
    {
        return sanitize_text_field($value);
    }

    public function sanitize_website_id($value)
    {
        return sanitize_text_field($value);
    }

    public function register_settings()
    {
        register_setting(
            'autopilotseo_settings_group',
            self::OPTION_NAME,
            ['type' => 'string', 'sanitize_callback' => [$this, 'sanitize_api_key'], 'default' => '']
        );

        register_setting(
            'autopilotseo_settings_group',
            self::WEBSITE_OPTION_NAME,
            ['type' => 'string', 'sanitize_callback' => [$this, 'sanitize_website_id'], 'default' => '']
        );
    }


    /**
     * Connects with the AutopilotSEO service using the provided Website ID and API key.
     *
     * This function makes an API call to the endpoint:
     * http://localhost:3000/api/connect-wordpress
     *
     * In the request body, it sends the API key, Website ID, and some information
     * about the current site (its URL).
     *
     * @param string $website_id The unique identifier for the site.
     * @param string $api_key    The API key provided by the user.
     * @return array|WP_Error    Returns the response data if successful, or a WP_Error on failure.
     */
    public static function connect_with_service($website_id, $api_key)
    {
        $site_url = get_site_url();
        $site_name = get_bloginfo('name');
        $site_description = get_bloginfo('description');
        $site_language = get_bloginfo('language');
        $wp_version = get_bloginfo('version');
        $site_admin_email = get_bloginfo('admin_email');
        $home_url = get_home_url();
        $is_multisite = is_multisite();

        // Fetch authors (users who can publish posts).
        $authors = get_users([
            'role__in' => ['administrator', 'editor', 'author'], // Roles that can publish posts
            'fields'   => ['ID', 'display_name', 'user_email']
        ]);

        // Format author data for the API request.
        $author_list = array_map(function ($author) {
            return [
                'id'    => $author->ID,
                'name'  => $author->display_name,
                'email' => $author->user_email
            ];
        }, $authors);

        $api_url = 'https://autopilotseo.ai/api/connect-wordpress';
        // $api_url = 'http://localhost:3000/api/connect-wordpress';

        $response = wp_remote_post($api_url, [
            'timeout' => 30,
            'headers' => [
                'Content-Type' => 'application/json',
            ],
            'body' => wp_json_encode([
                'api_key'    => $api_key,
                'website_id' => $website_id,
                'site_url'   => $site_url,
                'site_name'  => $site_name,
                'site_description' => $site_description,
                'site_language' => $site_language,
                'wp_version' => $wp_version,
                'site_admin_email' => $site_admin_email,
                'home_url' => $home_url,
                'is_multisite' => $is_multisite,
                'authors' => $author_list,
            ]),
        ]);

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $json = json_decode($body, true);

        // We expect a JSON response containing a 'success' key.
        if (200 === $code && isset($json['success']) && $json['success']) {
            return $json;
        } else {
            $error_message = isset($json['message']) ? $json['message'] : __('Unknown error connecting to the AutopilotSEO service.', 'autopilotseo-connector');
            return new WP_Error('autopilotseo_connection_error', $error_message);
        }
    }
}
