<?php

if ( ! defined( 'ABSPATH' )) exit;

class CPFM_Feedback_Notice {
    
    private static $registered_notices = [];
    
    public function __construct() {
        
        add_action('admin_init', [ $this, 'cpfm_listen_for_external_notice_registration' ]);
        add_action('admin_enqueue_scripts', [ $this, 'cpfm_enqueue_assets' ]);
        add_action('wp_ajax_cpfm_handle_opt_in', [ $this, 'cpfm_handle_opt_in_choice' ]);
        add_action('admin_footer', [ $this, 'cpfm_render_notice_panel' ]);
        
    }
    
    public static function cpfm_register_notice($key, $args) {
        
        if (!current_user_can('manage_options')) {
            
            return;
        }
        
        $safe_key = sanitize_key($key);
        
        $notice = wp_parse_args($args, [
            'title'          => '',
            'message'        => '',
            'pages'          => [],
            'always_show_on' => [],
            'plugin_name'    => '',
        ]);

        // Sanitize and coerce expected types
        $notice['title']          = is_string($notice['title']) ? wp_strip_all_tags($notice['title']) : '';
        $notice['message']        = is_string($notice['message']) ? wp_strip_all_tags($notice['message']) : '';
        $notice['pages']          = array_values(array_filter(array_map('sanitize_key', (array) $notice['pages'])));
        $notice['always_show_on'] = array_values(array_filter(array_map('sanitize_key', (array) $notice['always_show_on'])));
        $notice['plugin_name']    = is_string($notice['plugin_name']) ? sanitize_key($notice['plugin_name']) : '';

        self::$registered_notices[$safe_key] = $notice;
    }
    
    public function cpfm_listen_for_external_notice_registration() {
        

        if (!current_user_can('manage_options')) {

            return;
        }

        /**
         * Allow other plugins to register notices dynamically.
         * Example usage in other plugins:
         * do_action('cpf_cpfm_register_notice', 'crypto', [
         *     'title' => 'Crypto Plugin Notice',
         *     'message' => 'This is a crypto dashboard setup notice.',
         *     'pages' => ['dashboard', 'cpfm_'],
         * ]);
         */
        // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Hook has cpfm_ prefix which is the feedback module prefix
        do_action('cpfm_register_notice');
    }

    public function cpfm_enqueue_assets() {

        if (!current_user_can('manage_options')) {

            return;

        }

 
        $screen         = get_current_screen();
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Just checking current admin page for display purposes, no action taken
        $current_page   = isset($_GET['page'])? sanitize_key($_GET['page']):'';
    
        // Gather all unique pages from registered notices
        $allowed_pages = [];
        
        foreach (self::$registered_notices as $notice) {
            if (!empty($notice['pages']) && is_array($notice['pages'])) {
                $allowed_pages = array_merge($allowed_pages, $notice['pages']);
            }
        }
    
        // Early return if not needed
        if (!in_array($current_page, array_unique($allowed_pages))) {
            return;
        }
        wp_enqueue_style('cpfm-common-review-style', ATLT_URL . 'admin/feedback/css/cpfm-admin-feedback.css', [], ATLT_VERSION);
        wp_enqueue_script(
            'cpfm-common-review-script', 
            ATLT_URL . 'admin/feedback/js/cpfm-admin-feedback.js', 
            ['jquery'], 
            ATLT_VERSION, 
            true

        );
        wp_localize_script('cpfm-common-review-script', 'adminNotice', [
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce'   => wp_create_nonce('dismiss_admin_notice'),
            'autoShowPages' => array_unique(
                array_merge(
                    [],
                    ...array_filter(
                        array_column(self::$registered_notices, 'always_show_on'),
                        function($pages) { return !empty($pages); }
                    )
                )
                    ),
        ]);
    }
  
    public function cpfm_handle_opt_in_choice() {

        if (!current_user_can('manage_options')) {

            wp_send_json_error('Unauthorized access.');
        }

        check_ajax_referer('dismiss_admin_notice', 'nonce');

        $category           = isset($_POST['category']) ? sanitize_key( wp_unslash( $_POST['category'] ) ): '';
        $opt_in_raw         = isset($_POST['opt_in']) ? sanitize_text_field( wp_unslash( $_POST['opt_in'] ) ) : '';
        $opt_in             = ($opt_in_raw === 'yes') ? 'yes' : 'no';

        if (!$category || !isset(self::$registered_notices[$category])) {

            wp_send_json_error('Invalid notice category.');
        }

        update_option("cpfm_opt_in_choice_{$category}", $opt_in);

        $review_option = get_option("cpfm_opt_in_choice_{$category}");
        
       
        if ($review_option === 'yes') {
            $notice = self::$registered_notices[$category];
            $plugin_name = isset($notice['plugin_name']) ? sanitize_key($notice['plugin_name']) : '';
            if ($plugin_name) {
                // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Dynamic hook name with proper cpfm_ prefix
                do_action('cpfm_after_opt_in_' . $plugin_name, $category);
            }
        }

        wp_send_json_success();
    }

    public function cpfm_render_notice_panel() {
        
        if (!current_user_can('manage_options') || !function_exists('get_current_screen')) { 
            return;
        }

        $screen         = get_current_screen();
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Just checking current admin page for display purposes, no action taken
        $current_page   = isset($_GET['page']) ? sanitize_key($_GET['page']) : '';

       
        $unread_count   = 0;
        $auto_show      = false;
    
        foreach (self::$registered_notices as $notice) {

            if (!empty($notice['always_show_on']) && in_array($current_page, (array) $notice['always_show_on'])) {
                $auto_show = true;
                break;
            }
        }
    
        $output = '';
        $output .= '<div id="cpfNoticePanel" class="notice-panel"' . ($auto_show ? ' data-auto-show="true"' : '') . '>';
        $output .= '<div class="notice-panel-header">' . esc_html__('Help Improve Plugins', 'automatic-translator-addon-for-loco-translate') . ' <span class="dashicons dashicons-no" id="cpfm_remove_notice"></span></div>';
        $output .= '<div class="notice-panel-content">';
    
        foreach (self::$registered_notices as $key => $notice) {

            $choice = get_option("cpfm_opt_in_choice_{$key}");

            if ($choice !== false) continue;
    
            $should_show = false;
            foreach ((array) $notice['pages'] as $match) {
                
                if ($current_page === $match || strpos($current_page, $match) === 0) {
                
                    $should_show = true;
                    break;
                }
            }
    
            if (!$should_show) continue;
            $unread_count++;
    
            $output .= '<div class="notice-item unread" data-notice-id="' . esc_attr($key) . '">';
            $output .= '<strong>' . esc_html($notice['title']) . '</strong>';
            $output .= '<div class="notice-message-with-toggle">';
            $output .= '<p>' . esc_html($notice['message']) . '<a href="#" class="cpf-toggle-extra">' . esc_html__(' More info', 'automatic-translator-addon-for-loco-translate') . '</a></p>';
            $output .= '</div>';
            $output .= '<div class="cpf-extra-info">';
            $output .= '<p>' . esc_html__('Opt in to receive email updates about security improvements, new features, helpful tutorials, and occasional special offers. We\'ll collect:', 'automatic-translator-addon-for-loco-translate') . '</p>';
            $output .= '<ul>';
            $output .= '<li>' . esc_html__('Your website home URL and WordPress admin email.', 'automatic-translator-addon-for-loco-translate') . '</li>';
            
            $output .= '<li>' .
    sprintf(
        wp_kses_post(
            /* translators: %s: API provider name like OpenAI or Gemini */ __('To check plugin compatibility, we will collect the following: list of active plugins and themes, server type, MySQL version, WordPress version, memory limit, site language and database prefix. <a href="%s" target="_blank" rel="noopener noreferrer">Click here</a>.', 'automatic-translator-addon-for-loco-translate')
        ),
        esc_url('https://my.coolplugins.net/terms/usage-tracking/')
    ) .
'</li>';

            $output .= '</ul>';
            
            $output .= '</div>';
            
            $output .= '<div class="notice-actions">';
            $output .= '<button class="button button-primary opt-in-yes" data-category="' . esc_attr($key) . '" id="yes-share-data" value="yes">' . esc_html__("Yes, I Agree", 'automatic-translator-addon-for-loco-translate') . '</button>';
            $output .= '<button class="button opt-in-no" data-category="' . esc_attr($key) . '" id="no-share-data" value="no">' . esc_html__('No, Thanks', 'automatic-translator-addon-for-loco-translate') . '</button>';
            $output .= '</div>';
            
            $output .= '</div>';
        }
    
        $output .= '</div>';
        $output .= '</div>'; 
     
        if ($unread_count > 0) {
            echo wp_kses_post($output);
        }
    }
}
new CPFM_Feedback_Notice();
