<?php
class ZIAI_Handler
{
    private $access_token;
    private $post_author;
    private $post_type;
    private $taxonomy;
    private $ziai_added = 0;
    private $ziai_imported = array();

    public function __construct($zl_external_article)
    {
        $this->access_token = $zl_external_article['access_token'];
        $this->post_type    = $zl_external_article['import_post_type'];
        $this->post_author  = $zl_external_article['import_author'];
        $this->taxonomy     = $zl_external_article['import_taxonomy'];
    }

    public function ziai_import_article()
    {
        // Validate required fields
        if (empty($this->access_token)) {
            return array(
                'status'   => 'errors',
                'message'  => __('Access token is required.', 'ziai-articles'),
            );
        }

        if (empty($this->post_type)) {
            return array(
                'status'   => 'errors',
                'message'  => __('Post type is required.', 'ziai-articles'),
            );
        }

        if (empty($this->taxonomy)) {
            return array(
                'status'   => 'errors',
                'message'  => __('Taxonomy is required.', 'ziai-articles'),
            );
        }

        // Make API request to get articles
        $response = wp_remote_get( 'https://api.intercom.io/articles/?per_page=-1',
            array(
                'method' => 'GET',
                'timeout' => 30,
                'headers' => array(
                    'Authorization' => 'Bearer ' . $this->access_token,
                    'Accept' => 'application/json'
                )
            )
        );

        // Check for wp_remote_get errors
        if (is_wp_error($response)) {
            return array(
                'status'   => 'errors',
                'message'  => sprintf(__('API request failed: %s', 'ziai-articles'), $response->get_error_message()),
            );
        }

        // Check HTTP response code
        $response_code = wp_remote_retrieve_response_code($response);
        if ($response_code !== 200) {
            $response_message = wp_remote_retrieve_response_message($response);
            return array(
                'status'   => 'errors',
                'message'  => sprintf(__('API request failed with status code %d: %s', 'ziai-articles'), $response_code, $response_message),
            );
        }

        // Get response body
        $response_body = wp_remote_retrieve_body($response);
        if (empty($response_body)) {
            return array(
                'status'   => 'errors',
                'message'  => __('Empty response from API.', 'ziai-articles'),
            );
        }

        // Decode JSON response
        $response_data = json_decode($response_body, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            return array(
                'status'   => 'errors',
                'message'  => sprintf(__('Invalid JSON response: %s', 'ziai-articles'), json_last_error_msg()),
            );
        }

        // Check for API errors
        if (isset($response_data['errors']) && !empty($response_data['errors'])) {
            $error_message = is_array($response_data['errors']) && isset($response_data['errors'][0]['message']) 
                ? $response_data['errors'][0]['message'] 
                : (is_object($response_data['errors'][0]) && isset($response_data['errors'][0]->message) 
                    ? $response_data['errors'][0]->message 
                    : __('Unknown API error', 'ziai-articles'));
            return array(
                'status'   => 'errors',
                'message'  => esc_html($error_message),
            );
        }

        // Check if data exists
        if (!isset($response_data['data']) || !is_array($response_data['data'])) {
            return array(
                'status'   => 'errors',
                'message'  => __('No articles data found in API response.', 'ziai-articles'),
            );
        }

        // Process each article
        foreach($response_data['data'] as $data){
            // Validate required data fields
            if (!isset($data['id']) || !isset($data['title']) || !isset($data['body'])) {
                continue; // Skip invalid articles
            }

            // Get collection information if parent_id exists
            $collection_name = '';
            if (isset($data['parent_id']) && !empty($data['parent_id'])) {
                $collection_response = wp_remote_get( 'https://api.intercom.io/help_center/collections/' . absint($data['parent_id']),
                    array(
                        'method' => 'GET',
                        'timeout' => 15,
                        'headers' => array(
                            'Authorization' => 'Bearer ' . $this->access_token,
                            'Accept' => 'application/json'
                        )
                    )
                );

                if (!is_wp_error($collection_response)) {
                    $collection_code = wp_remote_retrieve_response_code($collection_response);
                    if ($collection_code === 200) {
                        $collection_body = wp_remote_retrieve_body($collection_response);
                        $collection_data = json_decode($collection_body, true);
                        if (isset($collection_data['name'])) {
                            $collection_name = sanitize_text_field($collection_data['name']);
                        }
                    }
                }
                // Add small delay to avoid rate limiting
                usleep(100000); // 0.1 second delay
            }

            $article = array(
                'id' => absint($data['id']),
                'title' => sanitize_text_field($data['title']),
                'content' => wp_kses_post($data['body']),
                'status'  => 'published', // Always set to published for imported articles
                'collection' => $collection_name
            );
            $this->ziai_create_article($article);
        }

        return array(
            'status'   => 'success',
            'message'  => __('Articles imported successfully!!', 'ziai-articles'),
            'count'    => $this->ziai_added,
            'episodes' => $this->ziai_imported,
        );
    }

    protected function ziai_create_article($article)
    {
        $post_data = $this->ziai_get_post_data($article);
        
        // Validate post data
        if (empty($post_data) || !is_array($post_data)) {
            return;
        }

        // Insert or update post
        $post_id = wp_insert_post($post_data, true);
        
        /**
         * If an error occurred adding a post, continue the loop
         */
        if (is_wp_error($post_id)) {
            return;
        }

        // Force status to published (in case wp_insert_post didn't update it for existing posts)
        wp_update_post(array(
            'ID' => $post_id,
            'post_status' => 'publish'
        ));

        // Set post terms if category exists
        if (!empty($post_data['post_category']) && is_array($post_data['post_category']) && isset($post_data['post_category'][0])) {
            $term_result = wp_set_post_terms($post_id, array($post_data['post_category'][0]), $this->taxonomy);
            if (is_wp_error($term_result)) {
                // Log error but continue
                error_log('ZIAI: Failed to set post terms - ' . $term_result->get_error_message());
            }
        }
        
        $this->ziai_added++;
        if (isset($post_data['post_title'])) {
            $this->ziai_imported[] = $post_data['post_title'];
        }
    }

    public function ziai_get_post_data($article)
    {
        // Validate article data
        if (empty($article['id']) || empty($article['title'])) {
            return false;
        }

        // Handle collection/term creation
        $term_id = 0;
        if (!empty($article['collection'])) {
            $term_a = term_exists($article['collection'], $this->taxonomy);
            
            if ($term_a && isset($term_a['term_id'])) {
                $term_id = $term_a['term_id'];
            } else {
                // Create new term if it doesn't exist
                $term_result = wp_insert_term(
                    sanitize_text_field($article['collection']),
                    $this->taxonomy,
                    array(
                        'slug' => sanitize_title($article['collection']),
                    )
                );

                if (!is_wp_error($term_result) && isset($term_result['term_id'])) {
                    $term_id = $term_result['term_id'];
                } else {
                    // If term creation failed, try to get it again
                    $term_a = term_exists($article['collection'], $this->taxonomy);
                    if ($term_a && isset($term_a['term_id'])) {
                        $term_id = $term_a['term_id'];
                    }
                }
            }
        }

        // Always set status to published for imported articles
        $status = 'publish';

        // Prepare post data
        $post_data = array();
        $post_data['post_content']  = isset($article['content']) ? wp_kses_post($article['content']) : '';
        $post_data['post_title']    = sanitize_text_field($article['title']);
        $post_data['post_status']   = $status;
        $post_data['post_author']   = !empty($this->post_author) ? absint($this->post_author) : get_current_user_id();
        $post_data['post_type']     = $this->post_type;
        $post_data['meta_input']    = array(
            'zl_ziai_id' => absint($article['id']),
        );

        // Add category if term exists
        if ($term_id > 0) {
            $post_data['post_category'] = array($term_id);
        }

        // Check if post already exists
        $args = array(
            'post_type' => $this->post_type,
            'post_status' => array('publish', 'draft', 'pending', 'private'),
            'posts_per_page' => 1,
            'meta_query' => array(
                array(
                    'key' => 'zl_ziai_id',
                    'value' => absint($article['id']),
                    'compare' => '=',
                )
            )
        );

        $already_added = new WP_Query($args);
        if ($already_added->have_posts()) {
            $already_added->the_post();
            $post_id = get_the_ID();
            $post_data['ID'] = $post_id;
            wp_reset_postdata();
        }

        return $post_data;
    }
}