<?php

namespace AutoCraftPlayer\Framework\View;

use AutoCraftPlayer\Framework\Foundation\Application;
use Exception;

class View
{
    protected Application $app;

    public function __construct(Application $app)
    {
        $this->app = $app;
    }

    /**
     * Render the view.
     *
     * @param string $path
     * @param array $data
     * @return void
     */
    public function render(string $path, array $data = []): void
    {
        echo $this->make($path, $data); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
    }

    /**
     * Generate the view content.
     *
     * @param string $path
     * @param array $data
     * @return string
     * @throws Exception
     */
    public function make(string $path, array $data = []): string
    {
        $fullPath = $this->resolvePath($path);

        // Extract data safely
        extract($data, EXTR_SKIP);

        ob_start();

        try {
            require $fullPath;
        } catch (\Throwable $e) {
            ob_end_clean();
            throw new Exception("Error rendering view '{$path}': " . $e->getMessage(), 0, $e);
        }

        return ob_get_clean();
    }

    /**
     * Resolve the full path of the view file.
     *
     * @param string $path
     * @return string
     * @throws Exception
     */
    protected function resolvePath(string $path): string
    {
        $fullPath = $this->normalizePath($this->app['path.views'] . DIRECTORY_SEPARATOR . $path . '.php');
        if (file_exists($fullPath)) {
            return $fullPath;
        }

        throw new Exception("View file not found: {$fullPath}");
    }

    /**
     * Normalize the file path.
     *
     * @param string $path
     * @return string
     */
    protected function normalizePath(string $path): string
    {
        return str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $path);
    }
}
