<?php

namespace AutoCraftPlayer\Framework\Validation;

class Validator
{
    protected array $data;
    protected array $rules;
    protected array $messages;
    protected array $errors = [];

    public function __construct(array $data, array $rules, array $messages = [])
    {
        $this->data     = $data;
        $this->rules    = $rules;
        $this->messages = $messages;
    }

    public function validate(): bool
    {
        foreach ($this->rules as $field => $rules) {
            $value = $this->data[$field] ?? null;

            foreach (explode('|', $rules) as $rule) {
                $this->validateRule($field, $value, $rule);
            }
        }

        return empty($this->errors);
    }

    protected function validateRule(string $field, $value, string $rule): void
    {
        [$ruleName, $parameter] = array_pad(explode(':', $rule), 2, null);

        $method = "validate" . ucfirst($ruleName);

        if (!method_exists($this, $method)) {
            throw new \Exception("Validation rule {$ruleName} is not supported.");
        }

        if (!$this->$method($value, $parameter)) {
            $this->addError($field, $ruleName, $parameter);
        }
    }

    protected function addError(string $field, string $ruleName, $parameter): void
    {
        $messageKey             = "{$field}.{$ruleName}";
        $message                = $this->messages[$messageKey] ?? $this->getDefaultMessage(
                $field,
                $ruleName,
                $parameter
            );
        $this->errors[$field][] = $message;
    }

    protected function getDefaultMessage(string $field, string $ruleName, $parameter): string
    {
        $messages = [
            'required'       => 'The :attribute field is required.',
            'email'          => 'The :attribute must be a valid email address.',
            'min'            => 'The :attribute must be at least :min characters.',
            'max'            => 'The :attribute may not be greater than :max characters.',
            'between'        => 'The :attribute must be between :min and :max.',
            'integer'        => 'The :attribute must be an integer.',
            'string'         => 'The :attribute must be a string.',
            'boolean'        => 'The :attribute field must be true or false.',
            'date'           => 'The :attribute is not a valid date.',
            'url'            => 'The :attribute format is invalid.',
            'in'             => 'The selected :attribute is invalid.',
            'not_in'         => 'The selected :attribute is invalid.',
            'unique'         => 'The :attribute has already been taken.',
            'digits'         => 'The :attribute must be :digits digits.',
            'digits_between' => 'The :attribute must be between :min and :max digits.',
            'alpha'          => 'The :attribute may only contain letters.',
            'alpha_dash'     => 'The :attribute may only contain letters, numbers, dashes, and underscores.',
            'alpha_num'      => 'The :attribute may only contain letters and numbers.',
            'array'          => 'The :attribute must be an array.',
            'confirmed'      => 'The :attribute confirmation does not match.',
            'file'           => 'The :attribute must be a file.',
            'mimetypes'      => 'The :attribute must be a file of type: :values.',
            'exists'         => 'The selected :attribute is invalid.',
        ];

        $message = $messages[$ruleName] ?? 'The :attribute field has an error.';

        return str_replace([':attribute', ':min', ':max', ':digits', ':values'],
            [$field, $parameter, $parameter, $parameter, $parameter],
            $message);
    }

    public function errors(): array
    {
        return $this->errors;
    }

    protected function validateRequired($value, $parameter): bool
    {
        return !is_null($value) && $value !== '';
    }

    protected function validateEmail($value, $parameter): bool
    {
        return !is_null($value) && filter_var($value, FILTER_VALIDATE_EMAIL) !== false;
    }

    protected function validateMin($value, $parameter): bool
    {
        return !is_null($value) && (is_string($value) ? strlen($value) >= $parameter : $value >= $parameter);
    }

    protected function validateMax($value, $parameter): bool
    {
        return !is_null($value) && (is_string($value) ? strlen($value) <= $parameter : $value <= $parameter);
    }

    protected function validateBetween($value, $parameter): bool
    {
        if (is_null($value)) {
            return false;
        }

        [$min, $max] = explode(',', $parameter);
        if (is_string($value)) {
            $length = strlen($value);

            return $length >= $min && $length <= $max;
        } elseif (is_numeric($value)) {
            return $value >= $min && $value <= $max;
        }

        return false;
    }

    protected function validateInteger($value, $parameter): bool
    {
        return !is_null($value) && filter_var($value, FILTER_VALIDATE_INT) !== false;
    }

    protected function validateString($value, $parameter): bool
    {
        return !is_null($value) && is_string($value);
    }

    protected function validateBoolean($value, $parameter): bool
    {
        return !is_null($value) && (is_bool($value) || in_array($value, ['0', '1', 0, 1], true));
    }

    protected function validateDate($value, $parameter): bool
    {
        return !is_null($value) && strtotime($value) !== false;
    }

    protected function validateUrl($value, $parameter): bool
    {
        return !is_null($value) && filter_var($value, FILTER_VALIDATE_URL) !== false;
    }

    protected function validateIn($value, $parameter): bool
    {
        if (is_null($value)) {
            return false;
        }

        $list = explode(',', $parameter);

        return in_array($value, $list);
    }

    protected function validateNotIn($value, $parameter): bool
    {
        if (is_null($value)) {
            return false;
        }

        $list = explode(',', $parameter);

        return !in_array($value, $list);
    }

    protected function validateUnique($value, $parameter): bool
    {
        if (is_null($value)) {
            return false;
        }

        global $wpdb;
        [$table, $field] = explode(',', $parameter);

        $table = $wpdb->prefix . $table;

        $query = $wpdb->prepare("SELECT COUNT(*) FROM {$table} WHERE {$field} = %s", $value);
        $count = $wpdb->get_var($query);

        return $count == 0;
    }

    protected function validateDigits($value, $parameter): bool
    {
        return !is_null($value) && is_numeric($value) && strlen((string)$value) == $parameter;
    }

    protected function validateDigitsBetween($value, $parameter): bool
    {
        if (is_null($value)) {
            return false;
        }

        [$min, $max] = explode(',', $parameter);
        $length = strlen((string)$value);

        return is_numeric($value) && $length >= $min && $length <= $max;
    }

    protected function validateAlpha($value, $parameter): bool
    {
        return !is_null($value) && ctype_alpha($value);
    }

    protected function validateAlphaDash($value, $parameter): bool
    {
        return !is_null($value) && preg_match('/^[a-zA-Z0-9_-]+$/', $value);
    }

    protected function validateAlphaNum($value, $parameter): bool
    {
        return !is_null($value) && ctype_alnum($value);
    }

    protected function validateArray($value, $parameter): bool
    {
        return !is_null($value) && is_array($value);
    }

    protected function validateConfirmed($value, $parameter): bool
    {
        $confirmationField = "{$parameter}_confirmation";

        return isset($this->data[$confirmationField]) && $value === $this->data[$confirmationField];
    }

    protected function validateFile($value, $parameter): bool
    {
        return isset($value['tmp_name']) && is_uploaded_file($value['tmp_name']);
    }

    protected function validateMimetypes($value, $parameter): bool
    {
        if (!$this->validateFile($value, $parameter)) {
            return false;
        }

        $allowedMimes = explode(',', $parameter);
        $fileMime     = mime_content_type($value['tmp_name']);

        return in_array($fileMime, $allowedMimes);
    }

    protected function validateExists($value, $parameter): bool
    {
        global $wpdb;
        [$table, $field] = explode(',', $parameter);
        $table = $wpdb->prefix . $table;

        if (is_array($value)) {
            foreach ($value as $v) {
                $query = $wpdb->prepare("SELECT COUNT(*) FROM {$table} WHERE {$field} = %s", $v);
                if ($wpdb->get_var($query) == 0) {
                    return false;
                }
            }

            return true;
        }

        $query = $wpdb->prepare("SELECT COUNT(*) FROM {$table} WHERE {$field} = %s", $value);

        return $wpdb->get_var($query) > 0;
    }

    // Add more validation rules as needed...
}
