<?php

namespace AutoCraftPlayer\Framework\Support;

use AutoCraftPlayer\Framework\Support\Facades\Config;

class Session
{
    /**
     * The prefix for session keys.
     *
     * @var string
     */
    protected static $prefix = '';

    /**
     * Set the session key prefix.
     *
     * @param string $prefix
     *
     * @return void
     */
    public static function setPrefix(string $prefix)
    {
        static::$prefix = $prefix ? $prefix . '__' :  Config::get('app.plugin_prefix');
    }

    /**
     * Get the current session key prefix.
     *
     * @return string
     */
    public static function getPrefix(): string
    {
        return static::$prefix;
    }

    /**
     * Get the prefixed version of a key.
     *
     * @param string $key
     *
     * @return string
     */
    protected static function getPrefixedKey(string $key): string
    {
        return static::$prefix . $key;
    }

    /**
     * Start the session if it hasn't been started yet.
     *
     * @return void
     */
    protected static function start()
    {
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
    }

    /**
     * Get a value from the session.
     *
     * @param string $key
     * @param mixed $default
     *
     * @return mixed
     */
    public static function get($key, $default = null)
    {
        self::start();

        return $_SESSION[self::getPrefixedKey($key)] ?? $default;
    }

    /**
     * Put a value into the session.
     *
     * @param string $key
     * @param mixed $value
     *
     * @return void
     */
    public static function put($key, $value)
    {
        self::start();
        $_SESSION[self::getPrefixedKey($key)] = $value;
    }

    /**
     * Determine if a key exists in the session.
     *
     * @param string $key
     *
     * @return bool
     */
    public static function has($key)
    {
        self::start();

        return isset($_SESSION[self::getPrefixedKey($key)]);
    }

    /**
     * Remove an item from the session.
     *
     * @param string $key
     *
     * @return void
     */
    public static function forget($key)
    {
        self::start();
        unset($_SESSION[self::getPrefixedKey($key)]);
    }

    /**
     * Flush all session data.
     *
     * @param bool $onlyPrefixed Whether to only flush prefixed items
     *
     * @return void
     */
    public static function flush($onlyPrefixed = false)
    {
        self::start();
        if ($onlyPrefixed && static::$prefix) {
            foreach ($_SESSION as $key => $value) {
                if (strpos($key, static::$prefix) === 0) {
                    unset($_SESSION[$key]);
                }
            }
        } else {
            $_SESSION = [];
        }
    }

    /**
     * Regenerate the session ID.
     *
     * @return void
     */
    public static function regenerate()
    {
        self::start();
        session_regenerate_id(true);
    }

    /**
     * Flash a value to the session for the next request.
     *
     * @param string $key
     * @param mixed $value
     *
     * @return void
     */
    public static function flash($key, $value)
    {
        self::put('_flash_' . $key, $value);
    }

    /**
     * Get a flashed item and remove it.
     *
     * @param string $key
     * @param mixed $default
     *
     * @return mixed
     */
    public static function getFlash($key, $default = null)
    {
        $value = self::get('_flash_' . $key, $default);
        self::forget('_flash_' . $key);

        return $value;
    }
}
