<?php

namespace AutoCraftPlayer\Framework\Support;

class Pluralizer
{
    /**
     * Pluralize an English word.
     *
     * @param string $word
     * @return string
     */
    public static function plural($word)
    {
        $plural = [
            '/(quiz)$/i'                => '$1zes',
            '/^(ox)$/i'                 => '$1en',
            '/([m|l])ouse$/i'           => '$1ice',
            '/(matr|vert|ind)(?:ix|ex)$/i' => '$1ices',
            '/(x|ch|ss|sh)$/i'          => '$1es',
            '/([^aeiouy]|qu)y$/i'       => '$1ies',
            '/(hive)$/i'                => '$1s',
            '/(?:([^f])fe|([lr])f)$/i'  => '$1$2ves',
            '/sis$/i'                   => 'ses',
            '/([ti])um$/i'              => '$1a',
            '/(buffal|tomat)o$/i'       => '$1oes',
            '/(bu)s$/i'                 => '$1ses',
            '/(alias|status)$/i'        => '$1es',
            '/(octop|vir)us$/i'         => '$1i',
            '/(ax|test)is$/i'           => '$1es',
            '/s$/i'                     => 's',
            '/$/'                       => 's',
        ];

        $irregular = [
            'move'   => 'moves',
            'foot'   => 'feet',
            'goose'  => 'geese',
            'sex'    => 'sexes',
            'child'  => 'children',
            'man'    => 'men',
            'tooth'  => 'teeth',
            'person' => 'people',
        ];

        $uncountable = [
            'sheep',
            'fish',
            'deer',
            'series',
            'species',
            'money',
            'rice',
            'information',
            'equipment',
        ];

        // Save some time in the case that singular and plural are the same
        if (in_array(strtolower($word), $uncountable)) {
            return $word;
        }

        // Check for irregular singular forms
        foreach ($irregular as $pattern => $result) {
            if (preg_match('/' . $pattern . '$/i', $word)) {
                return preg_replace('/' . $pattern . '$/i', $result, $word);
            }
        }

        // Check for matches using regular expressions
        foreach ($plural as $pattern => $result) {
            if (preg_match($pattern, $word)) {
                return preg_replace($pattern, $result, $word);
            }
        }

        return $word;
    }

    /**
     * Singularize an English word.
     *
     * @param string $word
     * @return string
     */
    public static function singular($word)
    {
        $singular = [
            '/(quiz)zes$/i'             => '$1',
            '/(matr)ices$/i'            => '$1ix',
            '/(vert|ind)ices$/i'        => '$1ex',
            '/^(ox)en$/i'               => '$1',
            '/(alias|status)es$/i'      => '$1',
            '/(octop|vir)i$/i'          => '$1us',
            '/(cris|ax|test)es$/i'      => '$1is',
            '/(shoe)s$/i'               => '$1',
            '/(o)es$/i'                 => '$1',
            '/(bus)es$/i'               => '$1',
            '/([m|l])ice$/i'            => '$1ouse',
            '/(x|ch|ss|sh)es$/i'        => '$1',
            '/(m)ovies$/i'              => '$1ovie',
            '/(s)eries$/i'              => '$1eries',
            '/([^aeiouy]|qu)ies$/i'     => '$1y',
            '/([lr])ves$/i'             => '$1f',
            '/(tive)s$/i'               => '$1',
            '/(hive)s$/i'               => '$1',
            '/([^f])ves$/i'             => '$1fe',
            '/(^analy)ses$/i'           => '$1sis',
            '/([ti])a$/i'               => '$1um',
            '/(n)ews$/i'                => '$1ews',
            '/s$/i'                     => '',
        ];

        $irregular = [
            'moves'   => 'move',
            'feet'    => 'foot',
            'geese'   => 'goose',
            'sexes'   => 'sex',
            'children'=> 'child',
            'men'     => 'man',
            'teeth'   => 'tooth',
            'people'  => 'person',
        ];

        $uncountable = [
            'sheep',
            'fish',
            'deer',
            'series',
            'species',
            'money',
            'rice',
            'information',
            'equipment',
        ];

        // Save some time in the case that singular and plural are the same
        if (in_array(strtolower($word), $uncountable)) {
            return $word;
        }

        // Check for irregular plural forms
        foreach ($irregular as $result => $pattern) {
            if (preg_match('/' . $result . '$/i', $word)) {
                return preg_replace('/' . $result . '$/i', $pattern, $word);
            }
        }

        // Check for matches using regular expressions
        foreach ($singular as $pattern => $result) {
            if (preg_match($pattern, $word)) {
                return preg_replace($pattern, $result, $word);
            }
        }

        return $word;
    }
}
