<?php


namespace AutoCraftPlayer\Framework\Http;

use AutoCraftPlayer\Framework\Foundation\Application;
use AutoCraftPlayer\Framework\Foundation\Http\FormRequest;
use WP_REST_Request;
use WP_Error;
use Closure;
use ReflectionMethod;
use ReflectionFunction;

class Router
{
    protected Application $app;
    protected array $middlewareStack = [];
    protected array $groupAttributesStack = [];
    protected array $currentMiddlewareStack = [];
    protected array $routeMiddlewareStack = [];
    protected array $groupMiddlewareStack = [];

    public function __construct(Application $app)
    {
        $this->app = $app;
    }

    protected function restNameSpaceWithVersion(): string
    {
        return $this->app->config->get('app.rest_namespace') . '/' . $this->app->config->get('app.rest_version');
    }

    public function get(string $route, $callback): void
    {
        $this->registerRoute('GET', $route, $callback);
    }

    public function post(string $route, $callback): void
    {
        $this->registerRoute('POST', $route, $callback);
    }

    public function put(string $route, $callback): void
    {
        $this->registerRoute('PUT', $route, $callback);
    }

    public function patch(string $route, $callback): void
    {
        $this->registerRoute('PATCH', $route, $callback);
    }

    public function delete(string $route, $callback): void
    {
        $this->registerRoute('DELETE', $route, $callback);
    }

    public function middleware(array $middleware): self
    {
        $this->currentMiddlewareStack = $middleware;

        return $this;
    }

    public function group(...$params): void
    {
        $attributes = [];
        $routes     = null;

        foreach ($params as $param) {
            if (is_array($param)) {
                $attributes = array_merge($attributes, $param);
            } elseif (is_callable($param)) {
                $routes = $param;
            }
        }

        $parentAttributes = $this->groupAttributesStack;
        $parentMiddleware = $this->middlewareStack;

        // Merge attributes and handle prefix stacking
        if (isset($attributes['prefix'])) {
            $attributes['prefix'] = isset($parentAttributes['prefix'])
                ? trim($parentAttributes['prefix'], '/') . '/' . trim($attributes['prefix'], '/')
                : trim($attributes['prefix'], '/');
        }

        $this->groupAttributesStack = array_merge($parentAttributes, $attributes);
        $this->middlewareStack      = $this->groupMiddlewareStack;

        // Merge middleware if set by chaining
        if (!empty($this->currentMiddlewareStack)) {
            $this->groupMiddlewareStack = array_merge($this->groupMiddlewareStack, $this->currentMiddlewareStack);
        }

        if ($routes) {
            $routes($this);
        }

        // Restore parent state
        $this->groupAttributesStack   = $parentAttributes;
        $this->middlewareStack        = $parentMiddleware;
        $this->currentMiddlewareStack = [];
    }

    public function prefix(string $prefix): self
    {
        $this->groupAttributesStack['prefix'] = isset($this->groupAttributesStack['prefix'])
            ? trim($this->groupAttributesStack['prefix'], '/') . '/' . trim($prefix, '/')
            : trim($prefix, '/');

        return $this;
    }

    public function policy(string $policy): self
    {
        $this->groupAttributesStack['policy'] = $policy;

        return $this;
    }

    protected function registerRoute(string $method, string $route, $callback): void
    {
        $prefix    = $this->groupAttributesStack['prefix'] ?? '';
        $fullRoute = trim($prefix, '/') . '/' . trim($route, '/');

        // Merge current middleware stack with route-specific middleware stack
        $currentMiddlewareStack = array_merge($this->groupMiddlewareStack, $this->currentMiddlewareStack);

        add_action('rest_api_init', function () use ($method, $fullRoute, $callback, $currentMiddlewareStack) {
            $pattern = preg_replace_callback('/\{([^\/]+)\}/', function ($matches) {
                return '(?P<' . $matches[1] . '>[^\/]+)';
            }, $fullRoute);

            $args = [
                'methods'             => $method,
                'callback'            => function (WP_REST_Request $request) use ($callback, $currentMiddlewareStack) {
                    return $this->handleCallback($request, $callback, $currentMiddlewareStack);
                },
                'args'                => [],
                'permission_callback' => '__return_true'
            ];

            register_rest_route($this->restNameSpaceWithVersion(), $pattern, $args);
        });

        // Reset route-specific middleware stack after registering the route
        $this->currentMiddlewareStack = [];
    }

    protected function handleCallback(WP_REST_Request $request, $callback, array $middleware)
    {
        $requestObj = $this->resolveRequest($request, $callback);

        if (is_wp_error($requestObj)) {
            return $requestObj;
        }

        $policy = $this->groupAttributesStack['policy'] ?? '';
        if (!empty($policy) && !$this->checkPolicy($policy)) {
            return new WP_Error(
                'rest_forbidden',
                ('You are not authorized to access this route'),
                ['status' => 403]
            );
        }

        return $this->handleMiddleware($requestObj, function () use ($callback, $requestObj, $request) {
            return $this->executeCallback($callback, $requestObj, $request);
        }, $middleware);
    }

    protected function executeCallback($callback, $requestObj, WP_REST_Request $request)
    {
        if (is_array($callback) && count($callback) === 2 && class_exists($callback[0]) && method_exists(
                $callback[0],
                $callback[1]
            )) {
            $controller = $this->app->make($callback[0]);
            $method     = new ReflectionMethod($callback[0], $callback[1]);
            $args       = $this->resolveMethodDependencies($method, $requestObj, $request);

            if ($args instanceof WP_Error) {
                return $args;
            }

            return $method->invokeArgs($controller, $args);
        } elseif (is_callable($callback)) {
            $refFunction = new ReflectionFunction($callback);
            $args        = $this->resolveFunctionDependencies($refFunction, $requestObj, $request);

            if ($args instanceof WP_Error) {
                return $args;
            }

            return call_user_func_array($callback, $args);
        } else {
            return new WP_Error('rest_invalid_callback', ('Invalid callback'), ['status' => 500]);
        }
    }

    protected function handleMiddleware(Request $request, Closure $next, array $middleware)
    {
        if (empty($middleware)) {
            return $next($request);
        }

        $currentMiddleware = array_shift($middleware);

        return $this->app->make($currentMiddleware)->handle($request, function ($newRequest) use ($next, $middleware) {
            return $this->handleMiddleware($newRequest, $next, $middleware);
        });
    }

    protected function resolveRequest(WP_REST_Request $request, $callback)
    {
        $requestClass = Request::class;

        if (is_array($callback) && count($callback) === 2 && class_exists($callback[0]) && method_exists(
                $callback[0],
                $callback[1]
            )) {
            $reflectionMethod = new ReflectionMethod($callback[0], $callback[1]);
            $parameters       = $reflectionMethod->getParameters();
            foreach ($parameters as $parameter) {
                $type = $parameter->getType();
                if ($type && !$type->isBuiltin() && is_subclass_of($type->getName(), Request::class)) {
                    $requestClass = $type->getName();
                    break;
                }
            }
        }

        $requestObj = $this->app->make($requestClass, [
            'query'      => $request->get_query_params(),
            'request'    => $request->get_body_params(),
            'attributes' => [],
            'cookies'    => $_COOKIE,
            'files'      => $request->get_file_params(),
            'server'     => $_SERVER,
            'content'    => $request->get_body()
        ]);

        if ($requestObj instanceof FormRequest) {
            $requestObj->validateResolved();
        }

        return $requestObj;
    }

    protected function resolveMethodDependencies(ReflectionMethod $method, $requestObj, WP_REST_Request $request)
    {
        return $this->resolveDependencies($method->getParameters(), $requestObj, $request);
    }

    protected function resolveFunctionDependencies(ReflectionFunction $function, $requestObj, WP_REST_Request $request)
    {
        return $this->resolveDependencies($function->getParameters(), $requestObj, $request);
    }

    protected function resolveDependencies(array $parameters, Request $requestObj, WP_REST_Request $request): array
    {
        $args = [];
        foreach ($parameters as $parameter) {
            $type = $parameter->getType();
            if ($type && !$type->isBuiltin() && $type->getName() === get_class($requestObj)) {
                $args[] = $requestObj;
            } elseif (isset($request[$parameter->getName()])) {
                $args[] = $request[$parameter->getName()];
            } elseif ($parameter->isDefaultValueAvailable()) {
                $args[] = $parameter->getDefaultValue();
            } elseif ($parameter->allowsNull()) {
                $args[] = null;
            } else {
                return new WP_Error(
                    'rest_invalid_callback',
                    ("Missing parameter: {$parameter->getName()}"),
                    ['status' => 400]
                );
            }
        }

        return $args;
    }

    protected function checkPolicy(string $policy): bool
    {
        // Logic to check policy, return true or false based on policy validation
        return true; // Placeholder logic, replace with actual policy check
    }
}
