<?php

namespace AutoCraftPlayer\Framework\Http;

use ArrayAccess;
use AutoCraftPlayer\Framework\Support\Arr;
use AutoCraftPlayer\Framework\Support\Str;
use AutoCraftPlayer\Framework\Support\Traits\Macroable;
use AutoCraftPlayer\Framework\Validation\Validator;

class Request implements ArrayAccess
{
    use Macroable;

    protected $json;
    protected $attributes = [];
    protected $query = [];
    protected $request = [];
    protected $server = [];
    protected $headers = [];
    protected $cookies = [];
    protected $files = [];
    protected $content;

    public function __construct(
        array $query = [],
        array $request = [],
        array $attributes = [],
        array $cookies = [],
        array $files = [],
        array $server = [],
        $content = null
    ) {
        $this->initialize($query, $request, $attributes, $cookies, $files, $server, $content);
    }

    public static function capture()
    {
        return static::createFromGlobals();
    }

    public static function createFromGlobals()
    {
        return new static($_GET, $_POST, [], $_COOKIE, $_FILES, $_SERVER, file_get_contents('php://input'));
    }

    public function initialize(
        array $query = [],
        array $request = [],
        array $attributes = [],
        array $cookies = [],
        array $files = [],
        array $server = [],
        $content = null
    ) {
        $this->query      = $query;
        $this->request    = $request;
        $this->attributes = $attributes;
        $this->cookies    = $cookies;
        $this->files      = $files;
        $this->server     = $server;
        $this->headers    = $this->getHeadersFromServer($server);
        $this->content    = $content;

        if (strpos($this->headers['CONTENT_TYPE'] ?? '', 'application/json') === 0) {
            $this->json = json_decode($this->content, true);
        }
    }

    protected function getHeadersFromServer(array $server)
    {
        $headers = [];
        foreach ($server as $key => $value) {
            if (strpos($key, 'HTTP_') === 0) {
                $headers[str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', substr($key, 5)))))] = $value;
            }
        }

        return $headers;
    }

    public function method()
    {
        return $this->server['REQUEST_METHOD'] ?? 'GET';
    }

    public function url()
    {
        $scheme = $this->isSecure() ? 'https://' : 'http://';
        $host   = $this->server['HTTP_HOST'] ?? $this->server['SERVER_NAME'] ?? '';
        $uri    = $this->server['REQUEST_URI'] ?? '';

        return $scheme . $host . strtok($uri, '?');
    }

    public function fullUrl()
    {
        $queryString = $this->server['QUERY_STRING'] ?? '';

        return $queryString ? $this->url() . '?' . $queryString : $this->url();
    }

    public function fullUrlWithQuery(array $query)
    {
        return $this->fullUrl() . '&' . http_build_query($query);
    }

    public function fullUrlWithoutQuery($keys)
    {
        $query = $this->query();
        foreach ((array)$keys as $key) {
            unset($query[$key]);
        }

        return $this->url() . (count($query) ? '?' . http_build_query($query) : '');
    }

    public function path()
    {
        return trim($this->server['REQUEST_URI'] ?? '/', '/');
    }

    public function decodedPath()
    {
        return rawurldecode($this->path());
    }

    public function segment($index, $default = null)
    {
        $segments = $this->segments();

        return $segments[$index - 1] ?? $default;
    }

    public function segments()
    {
        $segments = explode('/', $this->decodedPath());

        return array_filter($segments);
    }

    public function is(...$patterns)
    {
        $path = $this->decodedPath();
        foreach ($patterns as $pattern) {
            if (Str::is($pattern, $path)) {
                return true;
            }
        }

        return false;
    }

    public function json($key = null, $default = null)
    {
        if (is_null($this->json)) {
            $this->json = json_decode($this->content, true);
        }

        if (is_null($key)) {
            return $this->json;
        }

        return Arr::get($this->json, $key, $default);
    }

    public function query()
    {
        return $this->query;
    }

    public function all()
    {
        return array_merge($this->query, $this->request, $this->files);
    }

    public function get(string $key, $default = null)
    {
        return Arr::get($this->all(), $key, $default);
    }

    public function toArray(): array
    {
        return $this->all();
    }

    #[\ReturnTypeWillChange]
    public function offsetExists($offset)
    {
        return Arr::has($this->all(), $offset);
    }

    #[\ReturnTypeWillChange]
    public function offsetGet($offset)
    {
        return Arr::get($this->all(), $offset);
    }

    #[\ReturnTypeWillChange]
    public function offsetSet($offset, $value)
    {
        $this->request[$offset] = $value;
    }

    #[\ReturnTypeWillChange]
    public function offsetUnset($offset)
    {
        unset($this->request[$offset]);
    }

    public function merge(array $input)
    {
        $this->request = array_merge($this->request, $input);

        return $this;
    }

    public function mergeIfMissing(array $input)
    {
        $this->request = array_merge($input, $this->request);

        return $this;
    }

    public function replace(array $input)
    {
        $this->request = $input;

        return $this;
    }

    protected function isSecure()
    {
        return ( ! empty($this->server['HTTPS']) && $this->server['HTTPS'] !== 'off')
               || $this->server['SERVER_PORT'] == 443;
    }

    public function ip()
    {
        return $this->server['REMOTE_ADDR'] ?? null;
    }

    public function ips()
    {
        return array_map(
            'trim',
            explode(',', $this->server['HTTP_X_FORWARDED_FOR'] ?? $this->server['REMOTE_ADDR'] ?? '')
        );
    }

    public function userAgent()
    {
        return $this->headers['User-Agent'] ?? null;
    }

    public function getContent()
    {
        return $this->content;
    }

    public function validate(array $rules, array $messages = [])
    {
        $validator = new Validator($this->all(), $rules, $messages);

        if ( ! $validator->validate()) {
            $errors       = $validator->errors();
            $errorMessage = $this->formatValidationErrors($errors);
            $response     = [
                'message' => $errorMessage,
                'errors'  => $errors
            ];

            wp_send_json($response, 422);
        }

        return true;
    }

    protected function formatValidationErrors(array $errors): string
    {
        $messages = [];
        foreach ($errors as $field => $fieldErrors) {
            foreach ($fieldErrors as $error) {
                $messages[] = $error;
            }
        }

        $firstMessage = $messages[0] ?? 'Validation error occurred.';
        $additionalCount = count($messages) - 1;

        if ($additionalCount > 0) {
            $firstMessage .= " (and $additionalCount more error" . ($additionalCount > 1 ? 's' : '') . ")";
        }

        return $firstMessage;
    }
}
