<?php

namespace AutoCraftPlayer\Framework\Http;

class Ajax
{
    protected $app;
    protected $currentScope = ['public', 'private'];
    protected $callbacks = [];

    public function __construct($app)
    {
        $this->app = $app;
        $this->app->addAction('init', function () {
            $this->registerCallbacks();
        });
    }

    protected function registerCallbacks()
    {
        foreach ($this->callbacks as $callback) {
            list($requestMethod, $slug, $methodCallback) = $callback;
            $this->method($requestMethod, $slug, $methodCallback);
        }
        $this->callbacks = [];
    }

    protected function method($requestMethod, $slug, $callback)
    {
        $slug = $this->app->config->get('app.slug') . '_' . $slug;

        if ($this->getRequestMethod() !== $requestMethod) {
            return;
        }

        $actionCallback = function () use ($callback) {
            if (!$this->verifyNonce()) {
                wp_send_json([
                    'message' => 'Nonce verification failed'
                ], 403);
            }
            call_user_func($this->wrapCallback($callback));
        };

        if (in_array('public', $this->currentScope)) {
            $this->app->addAction('wp_ajax_nopriv_' . $slug, $actionCallback);
        }

        if (in_array('private', $this->currentScope)) {
            $this->app->addAction('wp_ajax_' . $slug, $actionCallback);
        }
    }

    public function get($slug, $callback)
    {
        $this->callbacks[] = ['GET', $slug, $callback];

        return $this;
    }

    public function post($slug, $callback)
    {
        $this->callbacks[] = ['POST', $slug, $callback];

        return $this;
    }

    public function put($slug, $callback)
    {
        $this->callbacks[] = ['PUT', $slug, $callback];

        return $this;
    }

    public function patch($slug, $callback)
    {
        $this->callbacks[] = ['PATCH', $slug, $callback];

        return $this;
    }

    public function delete($slug, $callback)
    {
        $this->callbacks[] = ['DELETE', $slug, $callback];

        return $this;
    }

    public function scope($scopes)
    {
        if (is_string($scopes)) {
            $scopes = explode(',', $scopes);
        }

        if ( ! empty($scopes)) {
            $this->currentScope = $scopes;
        } else {
            $this->currentScope = ['public', 'private'];
        }

        return $this;
    }

    protected function getRequestMethod()
    {
        return sanitize_text_field($_SERVER['REQUEST_METHOD']);
    }

    protected function verifyNonce()
    {
        if (!isset($_REQUEST['nonce'])) {
            return false;
        }

        $nonce = sanitize_text_field($_REQUEST['nonce']);

        return wp_verify_nonce($nonce, $this->app->config->get('app.slug') . '_wp_ajax');
    }

    private function wrapCallback($callback)
    {
        if (is_array($callback) && is_string($callback[0])) {
            return function (...$args) use ($callback) {
                $instance = $this->app->make($callback[0]);

                return $instance->{$callback[1]}(...$args);
            };
        }

        return $callback;
    }
}
