<?php

namespace AutoCraftPlayer\Framework\Foundation;

/**
 * Trait FoundationTrait
 *
 * Provides methods to interact with WordPress hooks.
 */
trait FoundationTrait
{
    /**
     * Add an action hook.
     *
     * @param string $tag
     * @param callable|array $callback
     * @param int $priority
     * @param int $accepted_args
     *
     * @return void
     */
    public function addAction(string $tag, $callback, int $priority = 10, int $accepted_args = 1): void
    {
        add_action($tag, $this->wrapCallback($callback), $priority, $accepted_args);
    }

    /**
     * Add a filter hook.
     *
     * @param string $tag
     * @param callable|array $callback
     * @param int $priority
     * @param int $accepted_args
     *
     * @return void
     */
    public function addFilter(string $tag, $callback, int $priority = 10, int $accepted_args = 1): void
    {
        add_filter($tag, $this->wrapCallback($callback), $priority, $accepted_args);
    }

    /**
     * Remove an action hook.
     *
     * @param string $tag
     * @param callable|array $callback
     * @param int $priority
     *
     * @return void
     */
    public function removeAction(string $tag, $callback, int $priority = 10): void
    {
        remove_action($tag, $this->wrapCallback($callback), $priority);
    }

    /**
     * Remove a filter hook.
     *
     * @param string $tag
     * @param callable|array $callback
     * @param int $priority
     *
     * @return void
     */
    public function removeFilter(string $tag, $callback, int $priority = 10): void
    {
        remove_filter($tag, $this->wrapCallback($callback), $priority);
    }

    /**
     * Execute functions hooked on a specific action hook.
     *
     * @param string $tag
     * @param mixed ...$args
     *
     * @return void
     */
    public function doAction(string $tag, ...$args): void
    {
        do_action($tag, ...$args);
    }

    /**
     * Apply filters hooked on a specific filter hook.
     *
     * @param string $tag
     * @param mixed $value
     * @param mixed ...$args
     *
     * @return mixed
     */
    public function applyFilters(string $tag, $value, ...$args)
    {
        return apply_filters($tag, $value, ...$args);
    }

    /**
     * Check if any action has been registered for a hook.
     *
     * @param string $tag
     * @param callable|false $callback
     *
     * @return bool|int
     */
    public function hasAction(string $tag, $callback = false)
    {
        return has_action($tag, $callback);
    }

    /**
     * Check if any filter has been registered for a hook.
     *
     * @param string $tag
     * @param callable|false $callback
     *
     * @return bool|int
     */
    public function hasFilter(string $tag, $callback = false)
    {
        return has_filter($tag, $callback);
    }

    /**
     * Wrap the callback to resolve class instances using the container.
     *
     * @param callable|array $callback
     *
     * @return callable
     */
    protected function wrapCallback($callback)
    {
        if (is_array($callback) && is_string($callback[0])) {
            return function (...$args) use ($callback) {
                $instance = $this->app->make($callback[0]);

                return $instance->{$callback[1]}(...$args);
            };
        }

        return $callback;
    }

    private function getPluginFilePath(): string
    {
        return dirname(__DIR__, 5) . '/' . basename(dirname(__DIR__, 5)) . '.php';
    }

    protected function getRestInfo(): array
    {
        $nameSpace = $this->config->get('app.rest_namespace');
        $version   = $this->config->get('app.rest_version');
        $url       = esc_url(rest_url($nameSpace . '/' . $version));

        return [
            'nonce'     => wp_create_nonce('wp_rest'),
            'url'       => $url,
            'version'   => $version,
            'namespace' => $nameSpace,
        ];
    }

    protected function getAjaxInfo(): array
    {
        return [
            'nonce' => wp_create_nonce($this->config->get('app.slug') . '_wp_ajax'),
            'url'   => admin_url('admin-ajax.php'),
        ];
    }

    protected function getUserInfo(): array
    {
        $user = wp_get_current_user();

        return [
            'id'    => $user->ID,
            'email' => $user->user_email,
            'role'  => $user->roles[0] ?? 'subscriber',
            'name'  => $user->display_name,
            'login' => $user->user_login,
            'avatar' => get_avatar_url($user->ID),
        ];
    }

    protected function dynamicAssetsUrl()
    {
        $env = $this->config->get('app.env');

        if (in_array($env, ['production', 'prod'])) {
            return $this['url.assets'];
        }

        return $this->config->get('app.vite.resource_url') ?? 'http://127.0.0.1:9999/resources/';
    }
}
