<?php

namespace AutoCraftPlayer\Framework\Foundation;

use AutoCraftPlayer\Framework\Config\Repository;
use AutoCraftPlayer\Framework\Container\Container;
use AutoCraftPlayer\Framework\Database\Schema\Builder;
use AutoCraftPlayer\Framework\Database\WPDBConnector;
use AutoCraftPlayer\Framework\Http\Ajax;
use AutoCraftPlayer\Framework\Http\Router;
use AutoCraftPlayer\Framework\Http\Request;
use AutoCraftPlayer\Framework\View\View;

class Application extends Container
{
    use FoundationTrait;

    protected $file = null;
    protected $baseUrl = null;
    protected $basePath = null;

    public function __construct($file = null)
    {
        $this->init($file);
        $this->bootstrapApplication();
    }

    protected function init($file)
    {
        if (is_null($file)) {
            $file = $this->getPluginFilePath();
        }

        $this['file']   = $this->file = $file;
        $this->baseUrl  = plugin_dir_url($file);
        $this->basePath = plugin_dir_path($file);
    }

    protected function bootstrapApplication(): void
    {
        $this->bindApp();
        $this->bindPathsAndUrls();
        $this->bindCoreClasses();
        $this->registerTextDomain();
        $this->localizeScripts();
        $this->requireGlobalsFiles();
        $this->requireCommonFiles($this);
        $this->loadRoutes($this);
        $this->loadWPConnection();
    }

    protected function bindApp(): void
    {
        App::setInstance($this);
        $this->instance('app', $this);
    }

    protected function bindPathsAndUrls(): void
    {
        $this['url']              = $this->baseUrl;
        $this['url.assets']       = $this->baseUrl . 'assets/';
        $this['path']             = $this->basePath;
        $this['path.app']         = $this->basePath . 'app/';
        $this['path.hooks']       = $this->basePath . 'app/Hooks/';
        $this['path.http']        = $this->basePath . 'app/Http/';
        $this['path.controllers'] = $this->basePath . 'app/Http/Controllers/';
        $this['path.config']      = $this->basePath . 'config/';
        $this['path.assets']      = $this->basePath . 'assets/';
        $this['path.resources']   = $this->basePath . 'resources/';
        $this['path.routes']      = $this->basePath . 'routes/';
        $this['path.views']       = $this->basePath . 'app/Views/';
        $this['path.database']    = $this->basePath . 'database/';
    }

    protected function bindCoreClasses(): void
    {
        $this->bind('config', function ($app) {
            return new Repository($app);
        });

        $this->bind('view', function ($app) {
            return new View($app);
        });

        $this->singleton('router', function ($app) {
            return new Router($app);
        });

        $this->singleton(Request::class, function ($app, $params) {
            return new Request(
                $params['query'] ?? [],
                $params['request'] ?? [],
                [],
                $params['cookies'] ?? [],
                $params['files'] ?? [],
                $params['server'] ?? [],
                $params['content'] ?? null
            );
        });

        $this->singleton('db', function () {
            return WPDBConnector::getCapsuleInstance();
        });

        $this->singleton('schema', function ($app) {
            return new Builder($app);
        });

        $this->singleton('ajax', function ($app) {
            return new Ajax($app);
        });
    }

    protected function registerTextDomain(): void
    {
        add_action('plugins_loaded', function () {
            $textDomain = $this->config->get('app.text_domain');
            $domainPath = dirname(plugin_basename($this->file)) . $this->config->get('app.domain_path');
            load_plugin_textdomain($textDomain, false, $domainPath);
        });
    }

    protected function localizeScripts(): void
    {
        add_action('wp_loaded', function () {
            wp_localize_script('jquery', 'autoCraftPlayer', $this->localizedScripts());
        });
    }

    public function localizedScripts()
    {
        return [
            'rest'       => $this->getRestInfo(),
            'ajax'       => $this->getAjaxInfo(),
            'user'       => $this->getUserInfo(),
            'slug'       => $this->config->get('app.slug'),
            'assets_url' => $this->dynamicAssetsUrl(),
        ];
    }

    protected function requireGlobalsFiles()
    {
        if (file_exists($dev = $this->basePath . 'dev/globals-dev.php')) {
            require_once $dev;
        }
        require_once $this->basePath . 'bootstrap/globals.php';
    }

    protected function requireCommonFiles($app): void
    {
        require_once $this->basePath . 'app/Hooks/actions.php';
        require_once $this->basePath . 'app/Hooks/filters.php';
    }

    protected function loadRoutes($app): void
    {
        $this->restRoutes($app);
        $this->ajaxRoutes($app);
    }

    protected function loadWPConnection(): void
    {
        new WPDBConnector();
    }

    protected function restRoutes($app): void
    {
        $router = $app->router;
        require_once $this->basePath . 'routes/api.php';
    }

    protected function ajaxRoutes($app): void
    {
        $ajax = $app->ajax;
        require_once $this->basePath . 'routes/ajax.php';
    }
}
