<?php

namespace AutoCraftPlayer\Framework\Database\Schema;

use AutoCraftPlayer\Framework\Support\Facades\DB;

class Builder
{
    protected $app;
    protected $connection;

    public function __construct($app)
    {
        $this->app        = $app;
        $this->connection = DB::connection(); // Assuming DB facade is configured properly
    }

    private function isDev()
    {
        return $this->app->config->get('app.env') === 'dev';
    }

    public function migrate()
    {
        $this->runMigrationsForSite();
    }

    private function runMigrationsForSite(): void
    {
        $migrationsPath = $this->getMigrationsPath();
        if (!$this->ensureMigrationsDirectory($migrationsPath)) {
            return;
        }

        $migrationFiles = $this->getMigrationFiles($migrationsPath);
        if (empty($migrationFiles)) {
            return;
        }

        foreach ($migrationFiles as $file) {
            $this->processMigrationFile($file);
        }
    }

    private function getMigrationsPath(): string
    {
        return $this->app['path.database'] . 'migrations/';
    }

    private function ensureMigrationsDirectory(string $path): bool
    {
        if (!is_dir($path)) {
            if ($this->isDev()) {
                error_log('Migrations directory not found: ' . $path);
            }

            return false;
        }

        return true;
    }

    private function getMigrationFiles(string $path): array
    {
        $files = glob($path . '*.sql');
        if (!$files) {
            if ($this->isDev()) {
                error_log('No migration files found in: ' . $path);
            }
        }

        return $files;
    }

    private function processMigrationFile(string $file): void
    {
        $prefix = $this->connection->getTablePrefix();

        $filePath     = realpath($file);
        $expectedPath = realpath($this->getMigrationsPath());
        if ($filePath === false || strpos($filePath, $expectedPath) !== 0) {
            if ($this->isDev()) {
                error_log('Invalid migration file path: ' . $file);
            }

            return;
        }

        $tableName = $prefix . basename($file, '.sql');

        $columns = file_get_contents($file);
        if ($columns === false) {
            if ($this->isDev()) {
                error_log('Failed to read migration file: ' . $file);
            }

            return;
        }

        if ($this->hasTable(basename($file, '.sql'))) { // Checking without prefix
            $this->updateTable($tableName, $columns);
        } else {
            global $wpdb;
            $collate = $wpdb->get_charset_collate();

            $createTableQuery = "CREATE TABLE $tableName ($columns) $collate";
            $this->executeSqlQuery($createTableQuery);
        }
    }

    private function updateTable(string $tableName, string $columns): void
    {
        $currentColumns = $this->getCurrentColumns($tableName);
        $newColumns     = $this->parseColumns($columns);

        $alterQueries = [];
        $dropColumns  = [];

        foreach ($newColumns as $columnName => $columnDefinition) {
            if (!array_key_exists($columnName, $currentColumns)) {
                $alterQueries[] = "ADD COLUMN $columnDefinition";
            } elseif ($this->shouldModifyColumn($currentColumns[$columnName], $columnDefinition)) {
                $alterQueries[] = "MODIFY COLUMN $columnDefinition";
            }
        }

        // Identify columns to drop
        foreach ($currentColumns as $currentColumnName => $currentDefinition) {
            if (!array_key_exists($currentColumnName, $newColumns)) {
                $dropColumns[] = "DROP COLUMN $currentColumnName";
            }
        }

        if (!empty($alterQueries)) {
            $alterTableQuery = "ALTER TABLE $tableName " . implode(', ', $alterQueries);
            $this->executeSqlQuery($alterTableQuery);
        }

        if (!empty($dropColumns)) {
            $dropTableQuery = "ALTER TABLE $tableName " . implode(', ', $dropColumns);
            $this->executeSqlQuery($dropTableQuery);
        }
    }

    private function shouldModifyColumn(string $currentDefinition, string $newDefinition): bool
    {
        // Ignore modifications that would result in multiple primary keys or unique constraints
        if (strpos($newDefinition, 'PRIMARY KEY') !== false || strpos($newDefinition, 'UNIQUE') !== false) {
            return false;
        }

        return $currentDefinition !== $newDefinition;
    }

    private function getCurrentColumns(string $tableName): array
    {
        $columns = [];
        $results = $this->connection->select("SHOW COLUMNS FROM $tableName");
        foreach ($results as $column) {
            $columns[$column->Field] = $column->Field . ' ' . $column->Type .
                                       ($column->Null === 'NO' ? ' NOT NULL' : '') .
                                       ($column->Default !== null ? ' DEFAULT \'' . $column->Default . '\'' : '') .
                                       ($column->Extra ? ' ' . $column->Extra : '');
        }

        return $columns;
    }

    private function parseColumns(string $columns): array
    {
        $columnDefinitions = explode(',', $columns);
        $parsedColumns     = [];
        foreach ($columnDefinitions as $definition) {
            $definition                 = trim($definition);
            $parts                      = preg_split('/\s+/', $definition, 2);
            $columnName                 = $parts[0];
            $parsedColumns[$columnName] = $definition;
        }

        return $parsedColumns;
    }

    private function executeSqlQuery(string $query): void
    {
        try {
            $this->connection->statement($query);
        } catch (\Exception $e) {
            if ($this->isDev()) {
                error_log('SQL query failed: ' . $query . ' - Error: ' . $e->getMessage());
            }
        }
    }

    /**
     * Determine if the given table exists.
     *
     * @param string $table
     *
     * @return bool
     */
    public function hasTable($table)
    {
        $prefix = $this->connection->getTablePrefix();
        $query  = "SHOW TABLES LIKE '$prefix$table'";
        $result = $this->connection->select($query);

        return !empty($result);
    }

    /**
     * Determine if the given table has a given column.
     *
     * @param string $table
     * @param string $column
     *
     * @return bool
     */
    public function hasColumn($table, $column)
    {
        return in_array(
            strtolower($column),
            array_map('strtolower', $this->getColumnListing($table))
        );
    }

    /**
     * Determine if the given table has given columns.
     *
     * @param string $table
     * @param array $columns
     *
     * @return bool
     */
    public function hasColumns($table, array $columns)
    {
        $tableColumns = array_map('strtolower', $this->getColumnListing($table));

        foreach ($columns as $column) {
            if (!in_array(strtolower($column), $tableColumns)) {
                return false;
            }
        }

        return true;
    }

    /**
     * Get the column listing for a given table.
     *
     * @param string $table
     *
     * @return array
     */
    public function getColumnListing($table)
    {
        $table = $this->connection->getTablePrefix() . $table;

        $columns = $this->connection->select("SHOW COLUMNS FROM $table");

        return array_map(function ($column) {
            return $column->Field;
        }, $columns);
    }

    /**
     * Get the names of the tables that belong to the database.
     *
     * @return array
     */
    public function getTableListing()
    {
        return array_column($this->getTables(), 'name');
    }

    /**
     * Get the tables that belong to the database.
     *
     * @return array
     */
    public function getTables($full = true)
    {
        $tables = $this->connection->select('SHOW TABLES');
        $result = [];
        foreach ($tables as $table) {
            $tableName = array_values((array)$table)[0]; // Get the first value in the stdClass
            $result[]  = $full ? ['name' => $tableName] : $tableName;
        }

        return $result;
    }

    public function dropTable($tableName)
    {
        global $wpdb;
        $prefix = $wpdb->prefix;

        if (!str_starts_with($tableName, $prefix)) {
            $tableName = $prefix . $tableName;
        }

        $this->connection->statement("DROP TABLE IF EXISTS `$tableName`");
    }

    public function dropTables(array $tables)
    {
        foreach ($tables as $table) {
            $this->dropTable($table);
        }
    }
}
