<?php

namespace AutoCraftPlayer\Framework\Database\Query\Processors;

use AutoCraftPlayer\Framework\Database\Query\Builder;
use AutoCraftPlayer\Framework\Support\GlobalClass;

class PostgresProcessor extends Processor
{
    /**
     * Process an "insert get ID" query.
     *
     * @param  \AutoCraftPlayer\Framework\Database\Query\Builder  $query
     * @param  string  $sql
     * @param  array  $values
     * @param  string|null  $sequence
     * @return int
     */
    public function processInsertGetId(Builder $query, $sql, $values, $sequence = null)
    {
        $connection = $query->getConnection();

        $connection->recordsHaveBeenModified();

        $result = $connection->selectFromWriteConnection($sql, $values)[0];

        $sequence = $sequence ?: 'id';

        $id = is_object($result) ? $result->{$sequence} : $result[$sequence];

        return is_numeric($id) ? (int) $id : $id;
    }

    /**
     * Process the results of a types query.
     *
     * @param  array  $results
     * @return array
     */
    public function processTypes($results)
    {
        return array_map(function ($result) {
            $result = (object) $result;

            $type = strtolower($result->type);
            switch ($type) {
                case 'b':
                    $type = 'base';
                    break;
                case 'c':
                    $type = 'composite';
                    break;
                case 'd':
                    $type = 'domain';
                    break;
                case 'e':
                    $type = 'enum';
                    break;
                case 'p':
                    $type = 'pseudo';
                    break;
                case 'r':
                    $type = 'range';
                    break;
                case 'm':
                    $type = 'multirange';
                    break;
                default:
                    $type = null;
            }

            $category = strtolower($result->category);
            switch ($category) {
                case 'a':
                    $category = 'array';
                    break;
                case 'b':
                    $category = 'boolean';
                    break;
                case 'c':
                    $category = 'composite';
                    break;
                case 'd':
                    $category = 'date_time';
                    break;
                case 'e':
                    $category = 'enum';
                    break;
                case 'g':
                    $category = 'geometric';
                    break;
                case 'i':
                    $category = 'network_address';
                    break;
                case 'n':
                    $category = 'numeric';
                    break;
                case 'p':
                    $category = 'pseudo';
                    break;
                case 'r':
                    $category = 'range';
                    break;
                case 's':
                    $category = 'string';
                    break;
                case 't':
                    $category = 'timespan';
                    break;
                case 'u':
                    $category = 'user_defined';
                    break;
                case 'v':
                    $category = 'bit_string';
                    break;
                case 'x':
                    $category = 'unknown';
                    break;
                case 'z':
                    $category = 'internal_use';
                    break;
                default:
                    $category = null;
            }

            return [
                'name' => $result->name,
                'schema' => $result->schema,
                'implicit' => (bool) $result->implicit,
                'type' => $type,
                'category' => $category,
            ];
        }, $results);
    }

    /**
     * Process the results of a columns query.
     *
     * @param  array  $results
     * @return array
     */
    public function processColumns($results)
    {
        return array_map(function ($result) {
            $result = (object) $result;

            $autoincrement = $result->default !== null && str_starts_with($result->default, 'nextval(');

            $generationType = null;
            if ($result->generated) {
                switch ($result->generated) {
                    case 's':
                        $generationType = 'stored';
                        break;
                    default:
                        $generationType = null;
                }
            }

            return [
                'name' => $result->name,
                'type_name' => $result->type_name,
                'type' => $result->type,
                'collation' => $result->collation,
                'nullable' => (bool) $result->nullable,
                'default' => $result->generated ? null : $result->default,
                'auto_increment' => $autoincrement,
                'comment' => $result->comment,
                'generation' => $result->generated ? [
                    'type' => $generationType,
                    'expression' => $result->default,
                ] : null,
            ];
        }, $results);
    }

    /**
     * Process the results of an indexes query.
     *
     * @param  array  $results
     * @return array
     */
    public function processIndexes($results)
    {
        return array_map(function ($result) {
            $result = (object) $result;

            return [
                'name' => strtolower($result->name),
                'columns' => explode(',', $result->columns),
                'type' => strtolower($result->type),
                'unique' => (bool) $result->unique,
                'primary' => (bool) $result->primary,
            ];
        }, $results);
    }

    /**
     * Process the results of a foreign keys query.
     *
     * @param  array  $results
     * @return array
     */
    public function processForeignKeys($results)
    {
        return array_map(function ($result) {
            $result = (object) $result;

            $onUpdate = strtolower($result->on_update);
            switch ($onUpdate) {
                case 'a':
                    $onUpdate = 'no action';
                    break;
                case 'r':
                    $onUpdate = 'restrict';
                    break;
                case 'c':
                    $onUpdate = 'cascade';
                    break;
                case 'n':
                    $onUpdate = 'set null';
                    break;
                case 'd':
                    $onUpdate = 'set default';
                    break;
                default:
                    $onUpdate = null;
            }

            $onDelete = strtolower($result->on_delete);
            switch ($onDelete) {
                case 'a':
                    $onDelete = 'no action';
                    break;
                case 'r':
                    $onDelete = 'restrict';
                    break;
                case 'c':
                    $onDelete = 'cascade';
                    break;
                case 'n':
                    $onDelete = 'set null';
                    break;
                case 'd':
                    $onDelete = 'set default';
                    break;
                default:
                    $onDelete = null;
            }

            return [
                'name' => $result->name,
                'columns' => explode(',', $result->columns),
                'foreign_schema' => $result->foreign_schema,
                'foreign_table' => $result->foreign_table,
                'foreign_columns' => explode(',', $result->foreign_columns),
                'on_update' => $onUpdate,
                'on_delete' => $onDelete,
            ];
        }, $results);
    }
}
