<?php

namespace AutoCraftPlayer\Framework\Database;

use Exception;
use AutoCraftPlayer\Framework\Database\Query\Grammars\PostgresGrammar as QueryGrammar;
use AutoCraftPlayer\Framework\Database\Query\Processors\PostgresProcessor;
use AutoCraftPlayer\Framework\Database\Schema\Grammars\PostgresGrammar as SchemaGrammar;
use AutoCraftPlayer\Framework\Database\Schema\PostgresBuilder;
use AutoCraftPlayer\Framework\Database\Schema\PostgresSchemaState;
use AutoCraftPlayer\Framework\Filesystem\Filesystem;

class PostgresConnection extends Connection
{
    /**
     * Escape a binary value for safe SQL embedding.
     *
     * @param  string  $value
     * @return string
     */
    protected function escapeBinary($value)
    {
        $hex = bin2hex($value);

        return "'\x{$hex}'::bytea";
    }

    /**
     * Escape a bool value for safe SQL embedding.
     *
     * @param  bool  $value
     * @return string
     */
    protected function escapeBool($value)
    {
        return $value ? 'true' : 'false';
    }

    /**
     * Determine if the given database exception was caused by a unique constraint violation.
     *
     * @param  \Exception  $exception
     * @return bool
     */
    protected function isUniqueConstraintError(Exception $exception)
    {
        return '23505' === $exception->getCode();
    }

    /**
     * Get the default query grammar instance.
     *
     * @return \AutoCraftPlayer\Framework\Database\Query\Grammars\PostgresGrammar
     */
    protected function getDefaultQueryGrammar()
    {
        ($grammar = new QueryGrammar)->setConnection($this);

        return $this->withTablePrefix($grammar);
    }

    /**
     * Get a schema builder instance for the connection.
     *
     * @return \AutoCraftPlayer\Framework\Database\Schema\PostgresBuilder
     */
    public function getSchemaBuilder()
    {
        if (is_null($this->schemaGrammar)) {
            $this->useDefaultSchemaGrammar();
        }

        return new PostgresBuilder($this);
    }

    /**
     * Get the default schema grammar instance.
     *
     * @return \AutoCraftPlayer\Framework\Database\Schema\Grammars\PostgresGrammar
     */
    protected function getDefaultSchemaGrammar()
    {
        ($grammar = new SchemaGrammar)->setConnection($this);

        return $this->withTablePrefix($grammar);
    }

    /**
     * Get the schema state for the connection.
     *
     * @param  \AutoCraftPlayer\Framework\Filesystem\Filesystem|null  $files
     * @param  callable|null  $processFactory
     * @return \AutoCraftPlayer\Framework\Database\Schema\PostgresSchemaState
     */
    public function getSchemaState(?Filesystem $files = null, ?callable $processFactory = null)
    {
        return new PostgresSchemaState($this, $files, $processFactory);
    }

    /**
     * Get the default post processor instance.
     *
     * @return \AutoCraftPlayer\Framework\Database\Query\Processors\PostgresProcessor
     */
    protected function getDefaultPostProcessor()
    {
        return new PostgresProcessor;
    }
}
