<?php

namespace AutoCraftPlayer\Framework\Database;

use AutoCraftPlayer\Framework\Database\Query\Grammars\MariaDbGrammar as QueryGrammar;
use AutoCraftPlayer\Framework\Database\Query\Processors\MariaDbProcessor;
use AutoCraftPlayer\Framework\Database\Schema\Grammars\MariaDbGrammar as SchemaGrammar;
use AutoCraftPlayer\Framework\Database\Schema\MariaDbBuilder;
use AutoCraftPlayer\Framework\Database\Schema\MariaDbSchemaState;
use AutoCraftPlayer\Framework\Filesystem\Filesystem;
use AutoCraftPlayer\Framework\Support\Str;

class MariaDbConnection extends MySqlConnection
{
    /**
     * Determine if the connected database is a MariaDB database.
     *
     * @return bool
     */
    public function isMaria()
    {
        return true;
    }

    /**
     * Get the server version for the connection.
     *
     * @return string
     */
    public function getServerVersion(): string
    {
        return Str::between(parent::getServerVersion(), '5.5.5-', '-MariaDB');
    }

    /**
     * Get the default query grammar instance.
     *
     * @return \AutoCraftPlayer\Framework\Database\Query\Grammars\MariaDbGrammar
     */
    protected function getDefaultQueryGrammar()
    {
        ($grammar = new QueryGrammar)->setConnection($this);

        return $this->withTablePrefix($grammar);
    }

    /**
     * Get a schema builder instance for the connection.
     *
     * @return \AutoCraftPlayer\Framework\Database\Schema\MariaDbBuilder
     */
    public function getSchemaBuilder()
    {
        if (is_null($this->schemaGrammar)) {
            $this->useDefaultSchemaGrammar();
        }

        return new MariaDbBuilder($this);
    }

    /**
     * Get the default schema grammar instance.
     *
     * @return \AutoCraftPlayer\Framework\Database\Schema\Grammars\MariaDbGrammar
     */
    protected function getDefaultSchemaGrammar()
    {
        ($grammar = new SchemaGrammar)->setConnection($this);

        return $this->withTablePrefix($grammar);
    }

    /**
     * Get the schema state for the connection.
     *
     * @param  \AutoCraftPlayer\Framework\Filesystem\Filesystem|null  $files
     * @param  callable|null  $processFactory
     * @return \AutoCraftPlayer\Framework\Database\Schema\MariaDbSchemaState
     */
    public function getSchemaState(?Filesystem $files = null, ?callable $processFactory = null)
    {
        return new MariaDbSchemaState($this, $files, $processFactory);
    }

    /**
     * Get the default post processor instance.
     *
     * @return \AutoCraftPlayer\Framework\Database\Query\Processors\MariaDbProcessor
     */
    protected function getDefaultPostProcessor()
    {
        return new MariaDbProcessor;
    }
}
