(function(root, factory) {
    if (typeof exports === "object" && typeof module !== "undefined") {
        var videojs = require("video.js");
        module.exports = factory(videojs.default || videojs);
    } else if (typeof define === "function" && define.amd) {
        define(["videojs"], function(videojs) {
            return root.Vimeo = factory(videojs);
        });
    } else {
        root.Vimeo = factory(root.videojs);
    }
})(this, function(videojs) {
    "use strict";

    var _isOnMobile = videojs.browser.IS_IOS || videojs.browser.IS_NATIVE_ANDROID;
    var Tech = videojs.getTech("Tech");

    // Fix for createTimeRange vs createTimeRanges
    var createTimeRange = function(start, end) {
        if (videojs.time && videojs.time.createTimeRanges) {
            return videojs.time.createTimeRanges(start, end);
        }
        return videojs.createTimeRange(start, end);
    };

    class Vimeo extends Tech {
        constructor(options, ready) {
            super(options, ready);

            // Initialize state variables
            this.paused_ = true;
            this.ended_ = false;
            this.currentTime_ = 0;
            this.duration_ = 1; // Start with non-zero value to fix remaining time
            this.buffered_ = 0;
            this.volume_ = 1;
            this.muted_ = false;
            this.playbackRate_ = 1;

            this.setPoster(options.poster);
            this.setSrc(this.options_.source, true);
            this.setTimeout(function() {
                if (this.el_) {
                    this.el_.parentNode.className += " vjs-vimeo";
                    if (_isOnMobile) {
                        this.el_.parentNode.className += " vjs-vimeo-mobile";
                    }
                    if (Vimeo.isApiReady) {
                        this.initVimeoPlayer();
                    } else {
                        Vimeo.apiReadyQueue.push(this);
                    }
                }
            }.bind(this));
        }

        dispose() {
            if (this.vimeoPlayer) {
                if (this.vimeoPlayer.pause) {
                    this.vimeoPlayer.pause();
                }
                if (this.vimeoPlayer.destroy) {
                    this.vimeoPlayer.destroy();
                }
            } else {
                var index = Vimeo.apiReadyQueue.indexOf(this);
                if (index !== -1) {
                    Vimeo.apiReadyQueue.splice(index, 1);
                }
            }
            this.vimeoPlayer = null;
            this.el_.parentNode.className = this.el_.parentNode.className
                .replace(" vjs-vimeo", "")
                .replace(" vjs-vimeo-mobile", "");
            this.el_.parentNode.removeChild(this.el_);
            Tech.prototype.dispose.call(this);
        }

        createEl() {
            var div = document.createElement("div");
            div.setAttribute("id", this.options_.techId);
            div.setAttribute("style", "width:100%;height:100%;top:0;left:0;position:absolute");
            div.setAttribute("class", "vjs-tech vjs-tech-vimeo");

            var divWrapper = document.createElement("div");
            divWrapper.appendChild(div);

            // Always create a blocker to prevent clicks on Vimeo interface
            var divBlocker = document.createElement("div");
            divBlocker.setAttribute("class", "vjs-iframe-blocker");
            divBlocker.setAttribute("style", "position:absolute;top:0;left:0;width:100%;height:100%;z-index:1;");
            divBlocker.onclick = function(event) {
                // Prevent event from reaching the iframe
                event.preventDefault();
                event.stopPropagation();

                if (this.paused()) {
                    this.play();
                } else {
                    this.pause();
                }
            }.bind(this);

            divWrapper.appendChild(divBlocker);

            return divWrapper;
        }

        initVimeoPlayer() {
            var self = this;

            // Load the Vimeo API if it's not loaded already
            if (!window.Vimeo || !window.Vimeo.Player) {
                loadScript("https://player.vimeo.com/api/player.js", function() {
                    self.initVimeoPlayer();
                });
                return;
            }

            this.techInfo_ = 'Vimeo';

            // Get the source
            var source = this.options_.source;
            var videoId = this.parseUrl(source.src).videoId;

            // Create the player options
            var playerOptions = {
                id: videoId,
                controls: false,
                autopause: false,
                playsinline: true,
                autoplay: this.options_.autoplay || false,
                muted: this.options_.muted || false,
                loop: this.options_.loop || false,
                title: false,
                byline: false,
                portrait: false,
                background: false,
                transparent: true,
                responsive: true
            };

            // Initialize the player
            var playerElement = document.getElementById(this.options_.techId);
            this.vimeoPlayer = new Vimeo.Player(playerElement, playerOptions);

            // Register event listeners
            this.vimeoPlayer.on('loaded', function() {
                self.trigger('loadedmetadata');
                self.trigger('durationchange');
                self.trigger('canplay');

                // After loading, set the pointer-events to none to prevent
                // Vimeo controls from capturing clicks
                var iframe = playerElement.querySelector('iframe');
                if (iframe) {
                    iframe.style.pointerEvents = 'none';
                }

                // If we should autoplay, do so now
                if (self.options_.autoplay && !_isOnMobile) {
                    self.play();
                }

                // Get video info from Vimeo if needed
                self.updateVideoInfo();
            });

            this.vimeoPlayer.on('play', function() {
                self.paused_ = false;
                self.ended_ = false; // Reset ended flag when playing
                self.trigger('play');
                self.trigger('playing');
            });

            this.vimeoPlayer.on('pause', function() {
                self.paused_ = true;
                self.trigger('pause');
            });

            this.vimeoPlayer.on('ended', function() {
                self.ended_ = true;
                self.paused_ = true; // Make sure paused is also set
                self.trigger('ended');
            });

            this.vimeoPlayer.on('volumechange', function(data) {
                if (data && typeof data.volume !== 'undefined') {
                    self.volume_ = data.volume;
                }
                self.trigger('volumechange');
            });

            this.vimeoPlayer.on('error', function(data) {
                self.errorObject_ = {
                    code: 3, // MEDIA_ERR_DECODE
                    message: data.message || 'Vimeo player error'
                };
                self.trigger('error', self.errorObject_);
            });

            this.vimeoPlayer.on('timeupdate', function(data) {
                self.currentTime_ = data.seconds;
                self.trigger('timeupdate');
            });

            this.vimeoPlayer.on('progress', function(data) {
                self.buffered_ = data.percent;
                self.trigger('progress');
            });

            this.vimeoPlayer.on('seeking', function() {
                self.trigger('seeking');
            });

            this.vimeoPlayer.on('seeked', function() {
                self.trigger('seeked');
            });

            this.vimeoPlayer.on('bufferstart', function() {
                self.trigger('waiting');
            });

            this.vimeoPlayer.on('bufferend', function() {
                self.trigger('canplay');
            });

            this.vimeoPlayer.on('playbackratechange', function(data) {
                if (data && typeof data.playbackRate !== 'undefined') {
                    self.playbackRate_ = data.playbackRate;
                }
                self.trigger('ratechange');
            });

            // Mark player as ready
            this.triggerReady();
        }

        updateVideoInfo() {
            var self = this;

            // Get duration
            this.vimeoPlayer.getDuration().then(function(duration) {
                self.duration_ = duration || 1; // Use 1 as fallback to avoid division by zero
                self.trigger('durationchange');
            });

            // Get poster if not set
            if (!this.poster_) {
                this.updatePoster();
            }
        }

        updatePoster() {
            var self = this;

            // Get the video ID
            var videoId = this.parseUrl(this.currentSrc()).videoId;
            if (!videoId) return;

            // Use the Vimeo oEmbed API to get the thumbnail URL
            var oEmbedUrl = 'https://vimeo.com/api/oembed.json?url=https://vimeo.com/' + videoId;

            // Create a JSONP request
            var script = document.createElement('script');
            window['setPoster' + videoId] = function(data) {
                if (data && data.thumbnail_url) {
                    self.setPoster(data.thumbnail_url);
                    self.trigger('posterchange');
                }
                // Cleanup
                delete window['setPoster' + videoId];
            };

            script.src = oEmbedUrl + '&callback=setPoster' + videoId;
            document.body.appendChild(script);
        }

        parseUrl(url) {
            var result = { videoId: null };
            var regex = /(?:vimeo\.com\/|player\.vimeo\.com\/video\/)([0-9]+)/;
            var match = url.match(regex);

            if (match && match[1]) {
                result.videoId = match[1];
            }

            return result;
        }

        // Tech Interface Methods

        poster() {
            return this.poster_;
        }

        setPoster(val) {
            this.poster_ = val;
        }

        src(src) {
            if (src) {
                this.setSrc({ src: src });
            }
            return this.source ? this.source.src : '';
        }

        currentSrc() {
            return this.source ? this.source.src : '';
        }

        setSrc(source) {
            if (!source || !source.src) {
                return;
            }

            this.source = source;
            var videoId = this.parseUrl(source.src).videoId;

            if (!videoId) {
                console.error('Invalid Vimeo URL');
                return;
            }

            // If we already have a player, load the new video
            if (this.vimeoPlayer) {
                this.vimeoPlayer.loadVideo(videoId).then(() => {
                    // Update poster
                    this.updatePoster();
                }).catch((error) => {
                    console.error('Error loading Vimeo video:', error);
                });
            }

            // If we don't have a poster and have a videoId, try to get one
            if (!this.poster_ && videoId) {
                this.updatePoster();
            }
        }

        play() {
            if (this.vimeoPlayer) {
                this.vimeoPlayer.play().catch((error) => {
                    console.error('Error playing Vimeo video:', error);
                });
            }
        }

        pause() {
            if (this.vimeoPlayer) {
                this.vimeoPlayer.pause();
            }
        }

        // FIX: Return the stored value directly instead of a promise
        paused() {
            return this.paused_;
        }

        currentTime() {
            if (!this.vimeoPlayer) {
                return 0;
            }

            return this.currentTime_ || 0;
        }

        setCurrentTime(seconds) {
            if (this.vimeoPlayer) {
                this.vimeoPlayer.setCurrentTime(seconds).then(() => {
                    this.currentTime_ = seconds;
                }).catch((error) => {
                    console.error('Error setting current time:', error);
                });
            }
        }

        duration() {
            if (!this.vimeoPlayer) {
                return 0;
            }

            // FIX: Return at least 1 to avoid division by zero in remaining time calculation
            return this.duration_ || 1;
        }

        // FIX: Use compatible timeRange creation
        buffered() {
            if (!this.vimeoPlayer) {
                return createTimeRange(0, 0);
            }

            var duration = this.duration();
            var buffered = this.buffered_ || 0;

            return createTimeRange(0, duration * buffered);
        }

        volume() {
            if (!this.vimeoPlayer) {
                return 1;
            }

            return this.volume_ !== undefined ? this.volume_ : 1;
        }

        setVolume(percentAsDecimal) {
            if (!this.vimeoPlayer) {
                return;
            }

            this.volume_ = percentAsDecimal;

            this.vimeoPlayer.setVolume(percentAsDecimal).then(() => {
                this.trigger('volumechange');
            });
        }

        muted() {
            if (!this.vimeoPlayer) {
                return false;
            }

            return this.volume_ === 0;
        }

        setMuted(muted) {
            if (!this.vimeoPlayer) {
                return;
            }

            this.muted_ = muted;

            if (muted) {
                this.volumeBefore_ = this.volume_;
                this.setVolume(0);
            } else if (this.volumeBefore_) {
                this.setVolume(this.volumeBefore_);
            }
        }

        // FIX: Added this method which is required for Video.js
        seekable() {
            return createTimeRange(0, this.duration() || 0);
        }

        // FIX: Added this method which is required for Video.js
        ended() {
            return this.ended_;
        }

        supportsFullScreen() {
            return true;
        }

        enterFullScreen() {
            if (this.vimeoPlayer) {
                this.vimeoPlayer.requestFullscreen();
            }
        }

        playbackRate() {
            if (!this.vimeoPlayer) {
                return 1;
            }

            return this.playbackRate_ || 1;
        }

        setPlaybackRate(rate) {
            if (!this.vimeoPlayer) {
                return;
            }

            this.playbackRate_ = rate;

            this.vimeoPlayer.setPlaybackRate(rate).then(() => {
                this.trigger('ratechange');
            });
        }

        // Stubbed methods
        preload() { return 'auto'; }
        load() {}
        reset() {}

        // FIX: Added error method
        error() {
            return this.errorObject_ || null;
        }
    }

    Vimeo.isSupported = function() {
        return true;
    };

    Vimeo.canPlaySource = function(srcObj) {
        return srcObj.type === 'video/vimeo';
    };

    Vimeo.canPlayType = function(type) {
        return type === 'video/vimeo' ? 'maybe' : '';
    };

    // Utility methods
    function loadScript(src, callback) {
        var script = document.createElement('script');
        script.src = src;
        script.onload = callback;
        script.onerror = function() {
            console.error('Failed to load script:', src);
        };
        document.head.appendChild(script);
    }

    function injectCss() {
        var css =
            // Basic styling
            ".vjs-vimeo .vjs-iframe-blocker { display: none; }" +
            ".vjs-vimeo.vjs-user-inactive .vjs-iframe-blocker { display: block; }" +
            ".vjs-vimeo .vjs-poster { background-size: cover; }" +
            ".vjs-vimeo-mobile .vjs-big-play-button { display: none; }" +
            // Hide Vimeo UI
            ".vjs-vimeo iframe { pointer-events: none !important; }" +
            ".vjs-vimeo .vjs-tech-vimeo { pointer-events: none; }" +
            // Ensure videojs controls show up
            ".vjs-vimeo .vjs-control-bar { display: flex !important; }" +
            // Make iframe transparent to capture clicks
            ".vjs-tech-vimeo iframe { opacity: 1 !important; }" +
            // Hide download button and other UI elements
            ".vjs-vimeo .vp-sidedock, " +
            ".vjs-vimeo [data-clip-download], " +
            ".vjs-vimeo [class*='download'], " +
            ".vjs-vimeo .vp-menu, " +
            ".vjs-vimeo .vp-title { display: none !important; }";

        // Add CSS
        var style = document.createElement('style');
        style.type = 'text/css';
        if (style.styleSheet) {
            style.styleSheet.cssText = css;
        } else {
            style.appendChild(document.createTextNode(css));
        }
        document.head.appendChild(style);
    }

    // Set up the Vimeo player class
    Vimeo.Player = window.Vimeo ? window.Vimeo.Player : null;
    Vimeo.apiReadyQueue = [];
    Vimeo.isApiReady = false;

    // Register CSS
    if (typeof document !== 'undefined') {
        injectCss();
    }

    // Load Vimeo API
    if (typeof document !== 'undefined' && !window.Vimeo) {
        loadScript("https://player.vimeo.com/api/player.js", function() {
            Vimeo.isApiReady = true;

            // Process any queued players
            for (var i = 0; i < Vimeo.apiReadyQueue.length; i++) {
                Vimeo.apiReadyQueue[i].initVimeoPlayer();
            }

            Vimeo.apiReadyQueue = [];
        });
    } else if (window.Vimeo && window.Vimeo.Player) {
        Vimeo.isApiReady = true;
    }

    // Register the tech
    if (typeof videojs.registerTech !== 'undefined') {
        videojs.registerTech('Vimeo', Vimeo);
    } else {
        videojs.registerComponent('Vimeo', Vimeo);
    }

    return Vimeo;
});
