<?php

namespace AutoCraftPlayer\App\Services;

use Exception;
use DOMDocument;

/**
 * SVG Sanitizer Helper Class
 *
 * This class provides methods to sanitize SVG content before storing it in a database
 * to prevent XSS attacks and other security vulnerabilities.
 */
class SvgSanitizer
{
    /**
     * List of allowed SVG elements
     * @var array
     */
    private $allowedElements = [
        'svg',
        'a',
        'altglyph',
        'altglyphdef',
        'altglyphitem',
        'animatecolor',
        'animatemotion',
        'animatetransform',
        'circle',
        'clippath',
        'defs',
        'desc',
        'ellipse',
        'filter',
        'font',
        'g',
        'glyph',
        'glyphref',
        'hkern',
        'image',
        'line',
        'lineargradient',
        'marker',
        'mask',
        'metadata',
        'mpath',
        'path',
        'pattern',
        'polygon',
        'polyline',
        'radialgradient',
        'rect',
        'stop',
        'style',
        'switch',
        'symbol',
        'text',
        'textpath',
        'title',
        'tref',
        'tspan',
        'use',
        'view',
        'vkern'
    ];

    /**
     * List of allowed SVG attributes
     * @var array
     */
    private $allowedAttributes = [
        'accent-height',
        'accumulate',
        'additive',
        'alignment-baseline',
        'ascent',
        'attributename',
        'attributetype',
        'azimuth',
        'basefrequency',
        'baseline-shift',
        'begin',
        'bias',
        'by',
        'class',
        'clip',
        'clip-path',
        'clip-rule',
        'color',
        'color-interpolation',
        'color-interpolation-filters',
        'color-profile',
        'color-rendering',
        'cx',
        'cy',
        'd',
        'dx',
        'dy',
        'diffuseconstant',
        'direction',
        'display',
        'divisor',
        'dur',
        'edgemode',
        'elevation',
        'end',
        'fill',
        'fill-opacity',
        'fill-rule',
        'filter',
        'flood-color',
        'flood-opacity',
        'font-family',
        'font-size',
        'font-size-adjust',
        'font-stretch',
        'font-style',
        'font-variant',
        'font-weight',
        'fx',
        'fy',
        'g1',
        'g2',
        'glyph-name',
        'glyphref',
        'gradientunits',
        'gradienttransform',
        'height',
        'id',
        'image-rendering',
        'in',
        'in2',
        'k',
        'k1',
        'k2',
        'k3',
        'k4',
        'kerning',
        'keypoints',
        'keysplines',
        'keytimes',
        'lang',
        'lengthadjust',
        'letter-spacing',
        'kernelmatrix',
        'kernelunitlength',
        'lighting-color',
        'local',
        'marker-end',
        'marker-mid',
        'marker-start',
        'markerheight',
        'markerunits',
        'markerwidth',
        'maskcontentunits',
        'maskunits',
        'max',
        'mask',
        'media',
        'method',
        'mode',
        'min',
        'name',
        'numoctaves',
        'offset',
        'operator',
        'opacity',
        'order',
        'orient',
        'orientation',
        'origin',
        'overflow',
        'paint-order',
        'path',
        'pathlength',
        'patterncontentunits',
        'patterntransform',
        'patternunits',
        'points',
        'preservealpha',
        'preserveaspectratio',
        'r',
        'rx',
        'ry',
        'radius',
        'refx',
        'refy',
        'repeatcount',
        'repeatdur',
        'restart',
        'result',
        'rotate',
        'scale',
        'seed',
        'shape-rendering',
        'specularconstant',
        'specularexponent',
        'spreadmethod',
        'stddeviation',
        'stitchtiles',
        'stop-color',
        'stop-opacity',
        'stroke-dasharray',
        'stroke-dashoffset',
        'stroke-linecap',
        'stroke-linejoin',
        'stroke-miterlimit',
        'stroke-opacity',
        'stroke',
        'stroke-width',
        'style',
        'surfacescale',
        'tabindex',
        'targetx',
        'targety',
        'transform',
        'transform-origin',
        'text-anchor',
        'text-decoration',
        'text-rendering',
        'textlength',
        'type',
        'u1',
        'u2',
        'unicode',
        'values',
        'viewbox',
        'visibility',
        'version',
        'vert-adv-y',
        'vert-origin-x',
        'vert-origin-y',
        'width',
        'word-spacing',
        'wrap',
        'writing-mode',
        'xchannelselector',
        'ychannelselector',
        'x',
        'x1',
        'x2',
        'xmlns',
        'y',
        'y1',
        'y2',
        'z',
        'zoomandpan'
    ];

    /**
     * Disallowed attributes that could contain JavaScript
     * @var array
     */
    private $disallowedAttributes = [
        'onload',
        'onclick',
        'onmouseover',
        'onmouseout',
        'onmousemove',
        'onmousedown',
        'onmouseup',
        'onfocusin',
        'onfocusout',
        'onactivate',
        'onbegin',
        'onend',
        'onerror',
        'onrepeat',
        'animate',
        'animatetransform',
        'set',
        'onabort',
        'onblur',
        'oncanplay',
        'oncanplaythrough',
        'onchange',
        'oncontextmenu',
        'ondblclick',
        'ondrag',
        'ondragend',
        'ondragenter',
        'ondragleave',
        'ondragover',
        'ondragstart',
        'ondrop',
        'ondurationchange',
        'onemptied',
        'onended',
        'oninput',
        'oninvalid',
        'onkeydown',
        'onkeypress',
        'onkeyup',
        'onloadeddata',
        'onloadedmetadata',
        'onloadstart',
        'onpause',
        'onplay',
        'onplaying',
        'onprogress',
        'onratechange',
        'onreset',
        'onresize',
        'onscroll',
        'onseeked',
        'onseeking',
        'onselect',
        'onshow',
        'onstalled',
        'onsubmit',
        'onsuspend',
        'ontimeupdate',
        'onvolumechange',
        'onwaiting',
        'formaction',
        'xlink:href'
    ];

    /**
     * Sanitize SVG content
     *
     * @param string $svgContent The raw SVG content to sanitize
     *
     * @return string The sanitized SVG content
     * @throws Exception If the content is not valid SVG
     */
    public function sanitize($svgContent)
    {
        if (empty($svgContent) || !is_string($svgContent)) {
            throw new Exception('SVG content must be a non-empty string');
        }

        // Load the SVG content into a DOMDocument
        $dom                     = new DOMDocument();
        $dom->preserveWhiteSpace = false;
        $dom->formatOutput       = true;

        // Suppress XML parsing errors and warnings
        libxml_use_internal_errors(true);

        // Load SVG as XML
        $success = $dom->loadXML($svgContent);
        $errors  = libxml_get_errors();
        libxml_clear_errors();

        if (!$success || !empty($errors)) {
            throw new Exception('Invalid SVG content');
        }

        // Check if the root element is an SVG
        $rootElement = $dom->documentElement;
        if ($rootElement->nodeName !== 'svg') {
            throw new Exception('Root element must be <svg>');
        }

        // Clean the SVG recursively
        $this->cleanElement($rootElement);

        // Convert back to string
        $sanitizedSvg = $dom->saveXML($dom->documentElement);

        // Additional string-based filtering for malicious content
        $sanitizedSvg = $this->performFinalSanitization($sanitizedSvg);

        return $sanitizedSvg;
    }

    /**
     * Validate if a string is a valid SVG
     *
     * @param string $svgContent The SVG content to validate
     *
     * @return bool Whether the content is a valid SVG
     */
    public function isValidSvg($svgContent)
    {
        if (empty($svgContent) || !is_string($svgContent)) {
            return false;
        }

        // Check for basic SVG structure
        if (strpos($svgContent, '<svg') === false || strpos($svgContent, '</svg>') === false) {
            return false;
        }

        // Try parsing it
        $dom = new DOMDocument();
        libxml_use_internal_errors(true);
        $result = $dom->loadXML($svgContent);
        $errors = libxml_get_errors();
        libxml_clear_errors();

        if (!$result || !empty($errors)) {
            return false;
        }

        // Check if the root element is an SVG
        return $dom->documentElement->nodeName === 'svg';
    }

    /**
     * Clean an element and its children recursively
     *
     * @param DOMElement $element The element to clean
     */
    private function cleanElement($element)
    {
        $nodeName = strtolower($element->nodeName);

        // If not in our allowlist, remove the element
        if (!in_array($nodeName, $this->allowedElements)) {
            $element->parentNode->removeChild($element);

            return;
        }

        // Clean attributes
        $this->cleanAttributes($element);

        // Clean all child nodes
        $childNodes = [];
        foreach ($element->childNodes as $child) {
            $childNodes[] = $child;
        }

        foreach ($childNodes as $child) {
            if ($child->nodeType === XML_ELEMENT_NODE) {
                $this->cleanElement($child);
            }
        }
    }

    /**
     * Clean attributes of an element
     *
     * @param DOMElement $element The element whose attributes to clean
     */
    private function cleanAttributes($element)
    {
        $attributes = [];
        foreach ($element->attributes as $attr) {
            $attributes[] = $attr;
        }

        foreach ($attributes as $attr) {
            $attrName = strtolower($attr->name);

            // Remove disallowed attributes
            if (in_array($attrName, $this->disallowedAttributes)) {
                $element->removeAttribute($attr->name);
                continue;
            }

            // Remove attributes not in the allowlist
            if (!in_array($attrName, $this->allowedAttributes)) {
                $element->removeAttribute($attr->name);
                continue;
            }

            // Special handling for href attributes
            if ($attrName === 'href' || $attrName === 'xlink:href') {
                $value = strtolower($attr->value);
                if (preg_match('/^\s*javascript:/i', $value)) {
                    $element->removeAttribute($attr->name);
                }
            }

            // Special handling for style attributes
            if ($attrName === 'style') {
                $value = strtolower($attr->value);
                if (strpos($value, 'javascript:') !== false ||
                    strpos($value, 'expression(') !== false ||
                    strpos($value, 'url(') !== false ||
                    preg_match('/behaviour\s*:\s*url/i', $value)) {
                    $element->removeAttribute($attr->name);
                }
            }
        }
    }

    /**
     * Perform final string-based sanitization
     *
     * @param string $svgContent The partially sanitized SVG content
     *
     * @return string The fully sanitized SVG content
     */
    private function performFinalSanitization($svgContent)
    {
        // Remove any script tags
        $svgContent = preg_replace('/<script\b[^<]*(?:(?!<\/script>)<[^<]*)*<\/script>/i', '', $svgContent);

        // Remove javascript: protocol
        $svgContent = preg_replace('/javascript:/i', 'removed:', $svgContent);

        // Remove data: protocol (could be used for base64 encoded scripts)
        $svgContent = preg_replace('/data:/i', 'removed:', $svgContent);

        // Remove event handlers
        $svgContent = preg_replace('/on\w+=/i', 'removed=', $svgContent);

        return $svgContent;
    }

    /**
     * Convert a sanitized SVG to a data URL for use in img tags
     *
     * @param string $svgContent The sanitized SVG content
     *
     * @return string A data URL containing the SVG
     */
    public function toDataUrl($svgContent)
    {
        $sanitized = $this->sanitize($svgContent);

        return 'data:image/svg+xml;base64,' . base64_encode($sanitized);
    }
}
