<?php

namespace AutoCraftPlayer\App\Services;

use AutoCraftPlayer\App\App;
use AutoCraftPlayer\Framework\Support\Arr;
use AutoCraftPlayer\Framework\Support\Str;

class Helper
{
    public static function uid($length = 16)
    {
        return Str::random($length) . time();
    }

    public static function getCurrentUserId()
    {
        return get_current_user_id();
    }

    public static function mimeTypes($type = '')
    {
        $imageMimeTypes = ['image/jpeg', 'image/png', 'image/gif'];
        $audioMimeTypes = ['audio/mpeg', 'audio/mp3', 'audio/ogg', 'audio/wav'];
        $videoMimeTypes = ['video/mp4', 'video/quicktime', 'video/webm', 'video/ogg'];

        switch ($type) {
            case 'image':
                return $imageMimeTypes;
            case 'audio':
                return $audioMimeTypes;
            case 'video':
                return $videoMimeTypes;
            default:
                return array_merge($audioMimeTypes, $imageMimeTypes, $videoMimeTypes);
        }
    }

    public static function getUploadDirectory($path = ''): string
    {
        $slug        = App::getInstance('config')->get('app.slug');
        $wpUploadDir = wp_upload_dir();

        if (!empty($path)) {
            $path = '/' . $path . '/';
        } else {
            $path = '/';
        }

        $uploadDir = Arr::get($wpUploadDir, 'basedir') . '/' . $slug . $path;
        $uploadDir = preg_replace('#/+#', '/', $uploadDir);
        $uploadDir = sanitize_text_field($uploadDir);

        FileSystem::ensureDirectory($uploadDir);
        FileSystem::createIndexPhp($uploadDir);

        return $uploadDir;
    }

    public static function getUploadDirectorys($path = ''): string
    {
        global $wp_filesystem;

        if (empty($wp_filesystem)) {
            require_once(ABSPATH . 'wp-admin/includes/file.php');
            WP_Filesystem();
        }

        $slug        = App::getInstance('config')->get('app.slug');
        $wpUploadDir = wp_upload_dir();

        if (!empty($path)) {
            $path = '/' . $path . '/';
        } else {
            $path = '/';
        }

        $uploadDir = Arr::get($wpUploadDir, 'basedir') . '/' . $slug . $path;
        $uploadDir = preg_replace('#/+#', '/', $uploadDir);
        $uploadDir = sanitize_text_field($uploadDir);

        if (!$wp_filesystem->is_dir($uploadDir)) {
            $wp_filesystem->mkdir($uploadDir, FS_CHMOD_DIR);
        }

        $indexFilePath = $uploadDir . 'index.php';
        if (!$wp_filesystem->exists($indexFilePath)) {
            $indexContent = "<?php\n// Silence is golden.";
            $wp_filesystem->put_contents($indexFilePath, $indexContent, FS_CHMOD_FILE);
        }

        return $uploadDir;
    }

    public static function getUploadedUrl($uploadedDir = ''): string
    {
        $wpUploadDir = wp_upload_dir();
        $slug        = App::getInstance('config')->get('app.slug');

        if (!empty($uploadedDir)) {
            $uploadedDir = '/' . ltrim($uploadedDir, '/') . '/';
        } else {
            $uploadedDir = '/';
        }

        $uploadUrl = Arr::get($wpUploadDir, 'baseurl') . '/' . $slug . $uploadedDir;

        return esc_url($uploadUrl);
    }

    public static function getMaxUploadSize()
    {
        $maxUploadSize = wp_max_upload_size();

        return apply_filters('upload_size_limit', $maxUploadSize);
    }

    public static function formatBytes($bytes, $precision = 2)
    {
        if ($bytes >= 1073741824) { // 1 GB
            $bytes = number_format($bytes / 1073741824, $precision);
            $unit  = 'GB';
        } elseif ($bytes >= 1048576) { // 1 MB
            $bytes = number_format($bytes / 1048576, $precision);
            $unit  = 'MB';
        } elseif ($bytes >= 1024) { // 1 KB
            $bytes = number_format($bytes / 1024, $precision);
            $unit  = 'KB';
        } else {
            $unit = 'bytes';
        }

        return $bytes . ' ' . $unit;
    }

    public static function getMimeTypeByUrl($url): string
    {
        return self::getMimeTypeByProvider(self::getMediaProviderByUrl($url), $url);
    }

    public static function getMediaProviderByUrl($url)
    {
        try {
            $parsedUrl = wp_parse_url($url);
            if (!isset($parsedUrl['host'])) {
                throw new \Exception('Invalid URL: Missing host');
            }

            $hostname = strtolower(sanitize_text_field($parsedUrl['host']));

            $currentHost = strtolower(sanitize_text_field(Arr::get($_SERVER, 'HTTP_HOST')));
            if (!filter_var($currentHost, FILTER_VALIDATE_DOMAIN, FILTER_FLAG_HOSTNAME)) {
                throw new \Exception('Invalid HTTP_HOST');
            }

            if ($hostname === $currentHost) {
                return 'local';
            }

            if (strpos($hostname, 'youtube.com') !== false || strpos($hostname, 'youtu.be') !== false) {
                return 'youtube';
            } elseif (strpos($hostname, 'vimeo.com') !== false) {
                return 'vimeo';
            } else {
                return 'remote';
            }
        } catch (\Exception $e) {
            return 'custom';
        }
    }

    public static function getMimeTypeByProvider($provider, $url)
    {
        switch ($provider) {
            case 'youtube':
                return 'video/youtube';
            case 'vimeo':
                return 'video/vimeo';
            case 'custom_url':
                return self::getMimeTypeFromExtension($url) ?: 'video/mp4';
            default:
                return '';
        }
    }

    public static function getMimeTypeFromExtension($url): string
    {
        $path      = strtok($url, '?');
        $extension = strtolower(pathinfo($path, PATHINFO_EXTENSION));

        $extensionToMimeType = [
            // Video formats
            'mp4'  => 'video/mp4',
            'webm' => 'video/webm',
            'ogg'  => 'video/ogg',
            'ogv'  => 'video/ogg',
            'mov'  => 'video/quicktime',
            'flv'  => 'video/x-flv',
            'm3u8' => 'application/vnd.apple.mpegurl',
            'ts'   => 'video/mp2t',
            '3gp'  => 'video/3gpp',
            'avi'  => 'video/x-msvideo',
            'mkv'  => 'video/x-matroska',
            // Audio formats
            'mp3'  => 'audio/mpeg',
            'wav'  => 'audio/wav',
            'oga'  => 'audio/ogg',
            'flac' => 'audio/flac',
            'aac'  => 'audio/aac',
            'm4a'  => 'audio/mp4',
            'wma'  => 'audio/x-ms-wma',
            // Image formats
            'jpg'  => 'image/jpeg',
            'jpeg' => 'image/jpeg',
            'png'  => 'image/png',
            'gif'  => 'image/gif',
            'bmp'  => 'image/bmp',
            'webp' => 'image/webp',
            'svg'  => 'image/svg+xml',
            'tiff' => 'image/tiff',
            'ico'  => 'image/vnd.microsoft.icon',
            // Add more mappings as needed
        ];

        return $extensionToMimeType[$extension] ?? '';
    }

    public static function arrayMergeRecursive($array1, $array2)
    {
        $merged = $array1;

        foreach ($array2 as $key => $value) {
            if (is_array($value) && isset($merged[$key]) && is_array($merged[$key])) {
                $merged[$key] = self::arrayMergeRecursive($merged[$key], $value);
            } else {
                $merged[$key] = $value;
            }
        }

        return $merged;
    }

    public static function sanitizeSVG($svgContent)
    {
        $sanitizer = new SvgSanitizer();

        try {
            if (!$sanitizer->isValidSvg($svgContent)) {
                return false;
            }

            return $sanitizer->sanitize($svgContent);
        } catch (\Exception $e) {
            return false;
        }
    }

    public static function svgToDataUrl($svgContent)
    {
        return (new SvgSanitizer())->toDataUrl($svgContent);
    }

    public static function sanitizeJS($code)
    {
        // Normalize line endings
        $code = str_replace(["\r\n", "\r"], "\n", $code);

        // Remove potentially unsafe script tags
        $code = preg_replace('/<\s*script[^>]*>|<\/\s*script\s*>/i', '', $code);

        // Remove HTML comments
        $code = preg_replace('/<!--(.*?)-->/s', '', $code);

        // Convert HTML entities but preserve formatting
        return htmlspecialchars_decode($code);
    }

    public static function isCustomJsField($value)
    {
        // First, check if this is in a known JS field
        if (is_string($value) && strpos($value, 'customJS') !== false) {
            return true;
        }

        // If it's not a string, it's definitely not JS code
        if (!is_string($value)) {
            return false;
        }

        // Common JS patterns that indicate code rather than simple text
        $jsPatterns = [
            // Function definitions and arrow functions
            '/function\s*\(/',
            '/=>\s*{/',
            '/\)\s*{/',
            // Common JS methods and statements
            '/\.(on|addEventListener|querySelector|getElementById)/',
            '/document\./',
            '/window\./',
            // JS keywords in context
            '/const\s+[a-zA-Z_$]/',
            '/let\s+[a-zA-Z_$]/',
            '/var\s+[a-zA-Z_$]/',
            '/if\s*\(/',
            '/else\s*{/',
            '/for\s*\(/',
            '/while\s*\(/',
            // Method chains and property access
            '/\)\s*\./',
            '/}\s*\./',
            // Contains both opening and closing braces
            '/{\s*.*\s*}/s'
        ];

        // Test all patterns
        foreach ($jsPatterns as $pattern) {
            if (preg_match($pattern, $value)) {
                return true;
            }
        }

        return false;
    }
}
