<?php

namespace AutoCraftPlayer\App\Services;

class FileSystem
{
    /**
     * @var \WP_Filesystem_Base
     */
    private static $instance = null;

    /**
     * Initialize WordPress Filesystem
     *
     * @return \WP_Filesystem_Base
     */
    public static function init()
    {
        if (self::$instance === null) {
            require_once(ABSPATH . 'wp-admin/includes/file.php');
            WP_Filesystem();
            global $wp_filesystem;
            self::$instance = $wp_filesystem;
        }
        return self::$instance;
    }

    /**
     * Ensure a directory exists and is writable
     *
     * @param string $path
     * @param int $permissions
     * @return bool
     */
    public static function ensureDirectory($path, $permissions = 0755)
    {
        $fs = self::init();

        if (!$fs->exists($path)) {
            return $fs->mkdir($path, $permissions);
        }

        return $fs->is_writable($path);
    }

    /**
     * Copy a file
     *
     * @param string $source
     * @param string $destination
     * @param bool $overwrite
     * @return bool
     */
    public static function copy($source, $destination, $overwrite = true)
    {
        $fs = self::init();
        return $fs->copy($source, $destination, $overwrite);
    }

    /**
     * Delete a file
     *
     * @param string $path
     * @return bool
     */
    public static function delete($path)
    {
        $fs = self::init();
        return $fs->delete($path);
    }

    /**
     * Check if a file exists
     *
     * @param string $path
     * @return bool
     */
    public static function exists($path)
    {
        $fs = self::init();
        return $fs->exists($path);
    }

    /**
     * Put contents into a file
     *
     * @param string $path
     * @param string $contents
     * @param int $permissions
     * @return bool
     */
    public static function put($path, $contents, $permissions = 0644)
    {
        $fs = self::init();
        return $fs->put_contents($path, $contents, $permissions);
    }

    /**
     * Get contents of a file
     *
     * @param string $path
     * @return string|false
     */
    public static function get($path)
    {
        $fs = self::init();
        return $fs->get_contents($path);
    }

    /**
     * Move/rename a file
     *
     * @param string $source
     * @param string $destination
     * @return bool
     */
    public static function move($source, $destination)
    {
        $fs = self::init();
        return $fs->move($source, $destination);
    }

    /**
     * Create an index.php file in a directory for security
     *
     * @param string $directory
     * @return bool
     */
    public static function createIndexPhp($directory)
    {
        $indexPath = rtrim($directory, '/') . '/index.php';
        if (!self::exists($indexPath)) {
            return self::put($indexPath, "<?php\n// Silence is golden.", FS_CHMOD_FILE);
        }
        return true;
    }

    /**
     * Check if path is writable
     *
     * @param string $path
     * @return bool
     */
    public static function isWritable($path)
    {
        $fs = self::init();
        return $fs->is_writable($path);
    }

    /**
     * Check if path is readable
     *
     * @param string $path
     * @return bool
     */
    public static function isReadable($path)
    {
        $fs = self::init();
        return $fs->is_readable($path);
    }
}
