<?php

namespace AutoCraftPlayer\App\Http\Controllers;

use AutoCraftPlayer\App\Models\Upload;
use AutoCraftPlayer\App\Services\FileSystem;
use AutoCraftPlayer\App\Services\Helper;
use AutoCraftPlayer\Framework\Http\Request;

use AutoCraftPlayer\Framework\Support\Arr;

use AutoCraftPlayer\Framework\Support\Str;

use function AutoCraftPlayer\sanitizeArray;

class UploadController extends Controller
{
    public function index()
    {
    }

    public function create()
    {
    }

    public function store(Request $request)
    {
        if (!isset($_FILES['file'])) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
            wp_send_json([
                'message' => __('File not found', 'autocraft-player')
            ], 400);
        }

        $this->validateServerConfig($request->all());

        $oldMediaUid = sanitize_text_field(Arr::get($request->all(), 'old_media_uid'));

        if ($oldMediaUid && $oldMediaUid !== 'null') {
            $this->destroy($oldMediaUid);
        }

        $file = $request->get('file', []);
        $file = sanitizeArray($file);

        $fileName   = Arr::get($file, 'name');
        $mimeType   = Arr::get($file, 'type');
        $tempName   = Arr::get($file, 'tmp_name');
        $uid        = Helper::uid();
        $objectType = sanitize_text_field(Arr::get($request->all(), 'object_type'));
        $objectId   = intval(Arr::get($request->all(), 'object_id'));
        $meta       = sanitizeArray(Arr::get($request->all(), 'meta'));
        $isActive   = intval(Arr::get($request->all(), 'is_active'));

        if (!in_array($mimeType, Helper::mimeTypes($objectType))) {
            wp_send_json([
                'message' => __('Invalid file type', 'autocraft-player')
            ], 400);
        }

        if (!$objectType) {
            $objectType = Str::before($mimeType, '/');
        }

        $uploadDir   = Helper::getUploadDirectory();
        $newFileName = $uid . '-' . $fileName;

        $upload = Upload::create([
            'uid'         => $uid,
            'user_id'     => Helper::getCurrentUserId(),
            'file_path'   => $uploadDir . $newFileName,
            'url'         => Helper::getUploadedUrl() . $newFileName,
            'mime_type'   => $mimeType,
            'object_type' => $objectType,
            'object_id'   => $objectId,
            'meta'        => $meta,
            'is_active'   => $isActive
        ]);

        $copied = FileSystem::copy(
            $tempName,
            $uploadDir . $newFileName
        );

        if (!$copied) {
            $upload->delete();
            wp_send_json([
                'message' => __('Failed to upload file', 'autocraft-player')
            ], 500);
        }

        return [
            'data' => $upload
        ];
    }

    public function show(int $id)
    {
    }

    public function edit(int $id)
    {
    }

    public function update(Request $request, int $id)
    {
    }

    public function destroy($uid)
    {
        $uid       = sanitize_text_field($uid);
        $thumbnail = Upload::where('uid', $uid)->first();
        if ($thumbnail) {
            $thumbnail->delete();

            return [
                'message' => __('File deleted successfully', 'autocraft-player'),
                'data'    => null
            ];
        } else {
            return false;
        }
    }

    private function validateServerConfig($requestData)
    {
        $maxFileSize = Helper::getMaxUploadSize(); // Max file size in bytes
        $fileSize    = intval(Arr::get($requestData, 'file_size'));

        if ($fileSize > $maxFileSize) {
            $formattedFileSize = Helper::formatBytes($fileSize);
            $formattedMaxSize  = Helper::formatBytes($maxFileSize);

            wp_send_json([
                'message' => sprintf(
                /* translators: 1: formatted file size (e.g. "2 MB"), 2: formatted maximum upload size (e.g. "8 MB") */
                    __('File size (%1$s) exceeds the maximum upload size (%2$s).', 'autocraft-player'),
                    $formattedFileSize,
                    $formattedMaxSize
                )
            ], 400);
        }
    }
}
