<?php

namespace AutoCraftPlayer\App\Http\Controllers;

use AutoCraftPlayer\App\Helpers\MediaHelper;
use AutoCraftPlayer\App\Helpers\ValueFormatter;
use AutoCraftPlayer\App\Http\Requests\PlaylistRequest;
use AutoCraftPlayer\App\Models\Playlist;
use AutoCraftPlayer\App\Services\Helper;
use AutoCraftPlayer\Framework\Http\Request;
use AutoCraftPlayer\Framework\Support\Arr;
use AutoCraftPlayer\Framework\Support\Str;

use function AutoCraftPlayer\sanitizeArray;

class PlaylistController extends Controller
{
    public function index(Request $request)
    {
        $page    = intval($request->get('page', 1));
        $perPage = intval($request->get('per_page', 20));

        $query = Playlist::query()->orderBy('id', 'desc');

        if ($filters = $request->get('filters')) {
            $query = apply_filters('autocraft_player/multiple_filters_query', $query, $filters);
        }

        $playlists = $query->paginate($perPage, '*', 'page', $page);

        return [
            'data' => $playlists
        ];
    }

    public function create()
    {
    }

    public function store(PlaylistRequest $request)
    {
        $defaultConfig = MediaHelper::getConfigByType($request->get('type') . '_playlist');

        $mediaIds = array_map('intval', $request->get('media_ids', []));

        $playlist = Playlist::create([
            'uid'                 => Helper::uid(),
            'user_id'             => get_current_user_id(),
            'title'               => $request->get('title'),
            'slug'                => Str::slug($request->get('title')),
            'description'         => $request->get('description'),
            'url'                 => $request->get('url'),
            'privacy'             => $request->get('privacy', 'public'),
            'status'              => $request->get('status', 'published'),
            'default_media_order' => $request->get('default_media_order'),
            'type'                => $request->get('type'),
            'parent_id'           => $request->get('parent_id'),
            'meta'                => [
                'config' => $defaultConfig,
            ]
        ]);

        if (!empty($mediaIds)) {
            $playlist->media()->sync($mediaIds);
        }

        return [
            'message' => __('Playlist created successfully', 'autocraft-player'),
            'data'    => $playlist
        ];
    }

    public function show(int $id)
    {
        return [
            'data' => Playlist::findOrFail($id)
        ];
    }

    public function edit(int $id)
    {
        $playlist       = Playlist::with('user')->findOrFail($id);
        $config         = MediaHelper::getConfigByType($playlist->type . '_playlist');
        $meta           = $playlist->meta;
        $updatedConfig  = Helper::arrayMergeRecursive($config, Arr::get($meta, 'config', []));
        $meta['config'] = $updatedConfig;
        $playlist->meta = $meta;

        $updatedPlaylist = apply_filters('autocraft_player/playlist_edit', $playlist);

        return [
            'data' => $updatedPlaylist
        ];
    }

    public function update(PlaylistRequest $request, int $id)
    {
        $requestData              = $request->all();
        $parentId                 = intval($request->get('parent_id', 0));
        $requestData              = sanitizeArray($requestData);
        $requestData['parent_id'] = $parentId;
        $requestData['meta']      = ValueFormatter::formatRequestData(Arr::get($requestData, 'meta', []));

        $playlist = Playlist::findOrFail($id);
        $playlist->update($requestData);

        return [
            'message' => __('Playlist updated successfully', 'autocraft-player')
        ];
    }

    public function destroy(int $id)
    {
        $playlist = Playlist::findOrFail($id);
        $playlist->playlistItems()->delete();
        $playlist->delete();

        return [
            'message' => __('Playlist deleted Successfully', 'autocraft-player')
        ];
    }

    public function bulkActions(Request $request)
    {
        $ids    = sanitizeArray($request->get('ids', []));
        $action = sanitize_text_field($request->get('action', ''));

        if (empty($ids) || empty($action)) {
            return [
                'message' => __('Invalid request', 'autocraft-player')
            ];
        }

        if ($action === 'delete_selected') {
            Playlist::whereIn('id', $ids)->delete();
        }

        return [
            'message' => __('Bulk action completed successfully', 'autocraft-player')
        ];
    }

    public function getMediaByPlaylist(Request $request, int $id)
    {
        $page    = intval($request->get('page', 1));
        $perPage = intval($request->get('per_page', 20));
        $query   = Playlist::findOrFail($id)
                           ->media();

        if ($filters = $request->get('filters')) {
            $filters = sanitizeArray($filters);
            $query   = $this->filters($query, $filters);
        }

        $data = $query->orderBy('position', 'asc')
                      ->paginate($perPage);

        return [
            'data' => $data
        ];
    }

    public static function slugify($text, string $divider = '-')
    {
        // replace non letter or digits by divider
        $text = preg_replace('~[^\pL\d]+~u', $divider, $text);

        // transliterate
        $text = iconv('utf-8', 'us-ascii//TRANSLIT', $text);

        // remove unwanted characters
        $text = preg_replace('~[^-\w]+~', '', $text);

        // trim
        $text = trim($text, $divider);

        // remove duplicate divider
        $text = preg_replace('~-+~', $divider, $text);

        // lowercase
        $text = strtolower($text);

        if (empty($text)) {
            return 'n-a';
        }

        return $text;
    }
}
