<?php

namespace AutoCraftPlayer\App\Http\Controllers;

use AutoCraftPlayer\App\Models\WpModels\WpPost;
use AutoCraftPlayer\Framework\Support\Facades\Config;
use AutoCraftPlayer\Framework\Http\Request;
use AutoCraftPlayer\Framework\Support\Arr;

use function AutoCraftPlayer\arrayMergeRecursive;
use function AutoCraftPlayer\getOption;
use function AutoCraftPlayer\sanitizeArray;
use function AutoCraftPlayer\updateOption;

class GlobalSettingController extends Controller
{
    private $validTypes = ['video', 'audio', 'audio_playlist', 'video_playlist', 'general'];

    public function index()
    {
        return $this->getAllSettings();
    }

    public function store(Request $request)
    {
        return $this->saveAllSettings($request->all());
    }

    private function getAllSettings()
    {
        $globalsSettings = getOption('_global_settings', []);
        $result          = [];

        foreach ($this->validTypes as $type) {
            $configByType  = Config::get($type, []);
            $data          = Arr::get($globalsSettings, $type, []);
            $merged        = arrayMergeRecursive($configByType, $data);
            $result[$type] = $merged;
        }

        return apply_filters("autocraft_player/global_settings/before_get", $result);
    }

    private function saveAllSettings($requestData)
    {
        $settings      = getOption('_global_settings', []);
        $sanitizedData = sanitizeArray($requestData);

        foreach ($this->validTypes as $type) {
            if (isset($sanitizedData[$type])) {
                $settings[$type] = $sanitizedData[$type];
            }
        }

        $settings = apply_filters("autocraft_player/global_settings/before_set", $settings);
        updateOption('_global_settings', $settings);

        return [
            'message' => __('All settings updated successfully', 'autocraft-player'),
        ];
    }

    public function getBytype($type)
    {
        if (!in_array($type, $this->validTypes)) {
            wp_send_json([
                'message' => __('Invalid type', 'autocraft-player')
            ], 400);
        }

        $globalsSettings = getOption('_global_settings', []);
        $configByType    = Config::get($type, []);
        $data            = Arr::get($globalsSettings, $type, []);

        $settings = arrayMergeRecursive($configByType, $data);

        return [
            $type => apply_filters("autocraft_player/global_settings/before_get", $settings, $type),
        ];
    }

    public function setBytype($type, $requestData)
    {
        if (!in_array($type, $this->validTypes)) {
            wp_send_json([
                'message' => __('Invalid type', 'autocraft-player')
            ], 400);
        }

        $settings      = getOption('_global_settings', []);
        $sanitizedData = sanitizeArray($requestData);

        if (isset($sanitizedData[$type])) {
            $settings[$type] = apply_filters(
                "autocraft_player/global_settings/before_save",
                $sanitizedData[$type],
                $type
            );
        }

        updateOption('_global_settings', $settings);

        return [
            'message' => __('Settings updated successfully', 'autocraft-player'),
            'type'    => $type
        ];
    }

    public function getWpPages(Request $request)
    {
        $page    = (int)$request->get('page', 1);
        $perPage = (int)$request->get('per_page', 20);
        $search  = sanitize_text_field($request->get('search', ''));

        $publicPostTypes = get_post_types(['public' => true], 'names');

        $query = WpPost::query()
                       ->whereIn('post_type', $publicPostTypes)
                       ->where('post_status', 'publish');


        if (!empty($search)) {
            $query->where('post_title', 'LIKE', '%' . $search . '%');
        }

        $total = $query->count();

        $results = $query->select('ID as id', 'post_title as title')
                         ->orderBy('post_title')
                         ->skip(($page - 1) * $perPage)
                         ->take($perPage)
                         ->get();

        return [
            'data'     => $results,
            'total'    => $total,
            'page'     => $page,
            'per_page' => $perPage
        ];
    }
}
