<?php

namespace AutoCraftPlayer\App\Http\Controllers;

use AutoCraftPlayer\App\Helpers\MediaHelper;
use AutoCraftPlayer\App\Helpers\ValueFormatter;
use AutoCraftPlayer\App\Http\Requests\MediumRequest;
use AutoCraftPlayer\App\Models\Upload;
use AutoCraftPlayer\App\Services\Helper;

use AutoCraftPlayer\Framework\Http\Request;
use AutoCraftPlayer\Framework\Support\Arr;

use AutoCraftPlayer\Framework\Support\Str;

use function AutoCraftPlayer\sanitizeArray;

abstract class BaseMediaController extends Controller
{
    protected $mediaType;

    public function index(Request $request)
    {
        $page    = intval($request->get('page', 1));
        $perPage = intval($request->get('per_page', 20));

        $query = $this->mediaType::query()->orderBy('id', 'desc');

        if ($filters = $request->get('filters')) {
            $query = apply_filters('autocraft_player/multiple_filters_query', $query, $filters);
        }

        $media = $query->paginate($perPage, ['*'], 'page', $page);

        return [
            'data' => $media
        ];
    }

    public function store(MediumRequest $request)
    {
        $medium = $this->mediaType::create([
            'uid'         => Helper::uid(32),
            'user_id'     => Helper::getCurrentUserId(),
            'title'       => $request->get('title'),
            'slug'        => Str::slug($request->get('title')),
            'description' => $request->get('description'),
            'type'        => $this->mediaType::MEDIA_TYPE,
            'url'         => $request->get('url'),
            'privacy'     => $request->get('privacy', 'public'),
            'status'      => $request->get('status', 'publish'),
            'provider'      => $request->get('provider', 'local'), // Todo:: need to add remote for third party media
            'meta'        => ValueFormatter::formatRequestData($request->get('meta', [])),
        ]);

        $this->handleUploads($request->all(), $medium);

        return [
            'data'    => $medium,
            'message' => __('Media created successfully', 'autocraft-player')
        ];
    }

    public function edit(int $id)
    {
        $medium         = $this->mediaType::with('playlists')->findOrFail($id);
        $config         = MediaHelper::getConfigByType($this->mediaType::MEDIA_TYPE);
        $meta           = $medium->meta;
        $updatedConfig  = Helper::arrayMergeRecursive($config, Arr::get($meta, 'config', []));
        $meta['config'] = $updatedConfig;
        $medium->meta   = $meta;

        return [
            'data' => $medium
        ];
    }

    public function update(MediumRequest $request, int $id)
    {
        $medium = $this->mediaType::findOrFail($id);
        $this->handleMediaUpdate($request->all(), $medium);

        return [
            'data'    => $medium,
            'message' => __('Media updated successfully', 'autocraft-player')
        ];
    }

    public function destroy(int $id)
    {
        $media = $this->mediaType::findOrFail($id);
        $media->delete();

        return [
            'message' => __('Media deleted Successfully', 'autocraft-player')
        ];
    }

    public function bulkActions(Request $request)
    {
        $ids    = sanitizeArray($request->get('ids', []));
        $action = sanitize_text_field($request->get('action', ''));

        if (empty($ids) || empty($action)) {
            return [
                'message' => __('Invalid request', 'autocraft-player')
            ];
        }

        if ($action === 'delete_selected') {
            $this->mediaType::whereIn('id', $ids)->delete();
        }

        return [
            'message' => __('Bulk action completed successfully', 'autocraft-player')
        ];
    }

    public function updatePlaylistsForMedia($id, Request $request)
    {
        $request->validate([
            'playlist_ids' => 'required|exists:autocraft_player_playlists,id|array'
        ]);

        $playlistIds = array_map('intval', $request->get('playlist_ids', []));
        $media       = $this->mediaType::findOrFail($id);
        $media->playlists()->sync($playlistIds);

        return [
            'message' => __('Playlists updated successfully', 'autocraft-player')
        ];
    }

    protected function handleUploads(array $data, $medium)
    {
        if ($mediaData = Arr::get($data, $this->mediaType::MEDIA_TYPE)) {
            Upload::updateObjectIdById(Arr::get($mediaData, 'id'), $medium->id);
            $medium->load($this->mediaType::MEDIA_TYPE);
        }

        if (Arr::get($data, 'image')) {
            Upload::updateObjectIdById(Arr::get($data, 'image.id'), $medium->id);
            $medium->load('image');
        }

        if (Arr::get($data, 'playlist_ids')) {
            $playlistIds = array_map('intval', Arr::get($data, 'playlist_ids', []));
            $medium->playlists()->sync($playlistIds);
            $medium->load('playlists');
        }
    }

    protected function handleMediaUpdate(array $data, $medium)
    {
        $url    = Arr::get($data, 'url');
        $provider = Helper::getMediaProviderByUrl($url);

        if (($mediaData = Arr::get($data, $this->mediaType::MEDIA_TYPE)) && $provider === 'local') {
            $url = sanitize_url(Arr::get($mediaData, 'url'));
            if ($mediaData['is_active'] == 0) {
                if ($medium->{$this->mediaType::MEDIA_TYPE}) {
                    $medium->{$this->mediaType::MEDIA_TYPE}->delete();
                }
                Upload::updateObjectIdById($mediaData['id'], $medium->id);
                $medium->load($this->mediaType::MEDIA_TYPE);
            }
        } elseif ($medium->{$this->mediaType::MEDIA_TYPE}) {
            $medium->{$this->mediaType::MEDIA_TYPE}->delete();
        }

        $medium->update([
            'user_id'     => Helper::getCurrentUserId(),
            'title'       => Arr::get($data, 'title'),
            'slug'        => Arr::get($data, 'slug'),
            'description' => Arr::get($data, 'description'),
            'url'         => $url,
            'privacy'     => Arr::get($data, 'privacy', 'public'),
            'status'      => Arr::get($data, 'status', 'publish'),
            'provider'      => $provider,
            'meta'        => ValueFormatter::formatRequestData(Arr::get($data, 'meta', [])),
        ]);

        if ($image = Arr::get($data, 'image')) {
            if ($image['is_active'] == 0) {
                if ($medium->image) {
                    $medium->image->delete();
                }
                Upload::updateObjectIdById($image['id'], $medium->id);
                $medium->load('image');
            }
        } elseif ($medium->image) {
            $medium->image->delete();
        }

        if (Arr::get($data, 'playlist_ids')) {
            $playlistIds = array_map('intval', Arr::get($data, 'playlist_ids', []));
            $medium->playlists()->sync($playlistIds);
            $medium->load('playlists');
        } else {
            $medium->playlists()->detach();
        }
    }
}
